!
!   const
!     Definitions of constant numbers, mathematical constants/parameters,
!     limits of error functions and error type usage
!   Remarks:
!     "(kind = ik)" and "(kind = rk)" are definied in "kinds.f95"
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    module const
!
    use kinds, only : ik, rk
!
    implicit none
!
!-----------------------------------------------------------------------------
!
!   constant numbers
!
    real   ( kind = rk ), parameter    :: zero          =  0.00E+00_rk
    real   ( kind = rk ), parameter    :: half          =  0.50E+00_rk
    real   ( kind = rk ), parameter    :: one           =  1.00E+00_rk
    real   ( kind = rk ), parameter    :: two           =  2.00E+00_rk
    real   ( kind = rk ), parameter    :: three         =  3.00E+00_rk
    real   ( kind = rk ), parameter    :: four          =  4.00E+00_rk
    real   ( kind = rk ), parameter    :: five          =  5.00E+00_rk
    real   ( kind = rk ), parameter    :: six           =  6.00E+00_rk
    real   ( kind = rk ), parameter    :: seven         =  7.00E+00_rk
    real   ( kind = rk ), parameter    :: eight         =  8.00E+00_rk
    real   ( kind = rk ), parameter    :: nine          =  9.00E+00_rk
    real   ( kind = rk ), parameter    :: ten           = 10.00E+00_rk
    real   ( kind = rk ), parameter    :: twelve        = 12.00E+00_rk
    real   ( kind = rk ), parameter    :: sixteen       = 16.00E+00_rk
    real   ( kind = rk ), parameter    :: fortyeight    = 48.00E+00_rk
!
!-----------------------------------------------------------------------------
!
!   mathematical constants and parameters (up to 40 digits precision)
!
    real   ( kind = rk ), parameter    :: pi            = &
    3.1415926535897932384626433832795028841972E+00_rk   !         pi
!
    real   ( kind = rk ), parameter    :: twodivpi      = &
    0.6366197723675813430755350534900574481378E+00_rk   !       2/pi
!
    real   ( kind = rk ), parameter    :: sqrtpi        = &
    1.7724538509055160272981674833411451827975E+00_rk   !    SQRT(pi)
!
    real   ( kind = rk ), parameter    :: sqrtpidiv2    = &
    0.8862269254527580136490837416705725913988E+00_rk   !    SQRT(pi)/2
!
    real   ( kind = rk ), parameter    :: onedivsqrtpi  = &
    0.5641895835477562869480794515607725858441E+00_rk   !  1/SQRT(pi)
!
    real   ( kind = rk ), parameter    :: twodivsqrtpi  = &
    1.1283791670955125738961589031215451716881E+00_rk   !  2/SQRT(pi)
!
    real   ( kind = rk ), parameter    :: fourdivsqrtpi = &
    2.2567583341910251477923178062430903433762E+00_rk   !  4/SQRT(pi)
!
    real   ( kind = rk ), parameter    :: sqrt2         = &
    1.4142135623730950488016887242096980785697E+00_rk   !    SQRT(2)
!
    real   ( kind = rk ), parameter    :: onedivsqrt2   = &
    0.7071067811865475244008443621048490392848E+00_rk   !  1/SQRT(2)
!
    real   ( kind = rk ), parameter    :: tendivsqrt2   = &
    7.0710678118654752440084436210484903928484E+00_rk   ! 10/SQRT(2)
!
    real   ( kind = rk ), parameter    :: sqrt2pi       = &
    2.5066282746310005024157652848110452530070E+00_rk   !    SQRT(2*pi)
!
!-----------------------------------------------------------------------------
!
!   Precision and/or Termination Accuracy
!
!   Termination accuracy for iterative algorithms and/or formulas
!   with     kind = 1 (sp) : 0.5960464566356904460...E-0008
!            kind = 2 (dp) : 0.1110223024625156602...E-0016
!            kind = 3 (ep) : 0.5421010862427522160...E-0020
!                     (qp) : 9.6296497219361792652...E-0036
    real   ( kind = rk ), parameter            :: eps           = &
                                   0.10E+00_rk * EPSILON(one) / two
!
!-----------------------------------------------------------------------------
!
!   Limit of the intrinsic function EXP
!
!   Maximum value, where "EXP( eMax*eMax )" does not generate an error
!   and/or an overflow
!   with     kind = 1 (sp) :   9.19586658477783203...E+0000
!            kind = 2 (dp) :  26.54289391760732997...E+0000
!            kind = 3 (ep) : 106.53716393181042460...E+0000
!                     (qp) : 106.53716393181042460...E+0000
    real   ( kind = rk ), parameter            :: eMax          = &
                          SQRT( LOG(one / TINY(one) / fortyeight) )
!
!-----------------------------------------------------------------------------
!
!   Limit for very small x-values
!
!   Limit where y = erf  ( |x| ≤ sqrteps ) :=        2.00/SQRT(pi)*x
!   Limit where y = erfc ( |x| ≤ sqrteps ) := 1.00 - 2.00/SQRT(pi)*x
!   Limit where y = erfcx( |x| ≤ sqrteps ) := 1.00 - 2.00/SQRT(pi)*x
!   with     kind = 1 (sp) : +3.452669770922511816...E-0004
!            kind = 2 (dp) : +1.490116119384765625...E-0008
!            kind = 3 (ep) : +3.292722539913596233...E-0010
!                     (qp) : +1.387778780781445675...E-0017
    real   ( kind = rk ), parameter            :: sqrteps       = &
                                               SQRT( EPSILON(one) )
!
!-----------------------------------------------------------------------------
!
!   Limits for y = erf(x)
!
!   Limit where y = erf( |x| ≥ limiterf ) := +1.00
!   with     kind = 1 (sp) : +4.007888317108154297...E+0000
!            kind = 2 (dp) : +6.013687356917750471...E+0000
!            kind = 3 (ep) : +6.617329870341344830...E+0000
!                     (qp) : +8.817781266302148406...E+0000
    real   ( kind = rk ), parameter            :: limiterf      = &
                          SQRT( -LOG( sqrtpi * EPSILON(one)/two ) )
!
!-----------------------------------------------------------------------------
!
!   Limits for y = erfc(x)
!
!   Limit where y = erfc( x ≤ limiterfc ) := +2.00
!   with     kind = 1 (sp) : -4.007888317108154297...E+0000
!            kind = 2 (dp) : -6.013687356917750471...E+0000
!            kind = 3 (ep) : -6.617329870341344830...E+0000
!                     (qp) : -8.817781266302148406...E+0000
    real   ( kind = rk ), parameter            :: limiterfc     = &
                          -SQRT( -LOG( sqrtpi * EPSILON(one)/two) )
!
!   Limit where y = erfc( z=|x| > limiterfcH ) := +0.00
!   to avoid an underflow, x should be less than limiterfcH
!   currently not used, compare with eMax, if EXP is used.
!   with     kind = 1 (sp) :   +9.3147287368774414...E+0000
!            kind = 2 (dp) :  +26.6049629503470548...E+0000
!            kind = 3 (ep) : +106.5578000288580439...E+0000
!                     (qp) : +106.5578000288580439...E+0000
    real   ( kind = rk ), parameter            :: limiterfcH    = &
                                 SQRT( -LOG( sqrtpi * TINY(one) ) )
!
!-----------------------------------------------------------------------------
!
!   Limits for y = erfcx(x)
!
!   Very Lower Limit where y = erfcx( x ≤ limiterfcxL ) := +∞
!   to avoid an underflow, x must be greater than limiterfcxL
!   with     kind = 1 (sp) :   -9.3824138641357421...E+0000
!            kind = 2 (dp) :  -26.6287357137514888...E+0000
!            kind = 3 (ep) : -106.5637380121098417...E+0000
!                     (qp) : -106.5637380121098417...E+0000
    real   ( kind = rk ), parameter            :: limiterfcxL   = &
                                      -SQRT( LOG( HUGE(one)/two ) )
!
!   Lower limit where y = erfcx( x ≤ limiterfcxL1 ) := 2*exp(x²)
!   with     kind = 1 (sp) : -4.007888317108154297...E+0000
!            kind = 2 (dp) : -6.013687356917750471...E+0000
!            kind = 3 (ep) : -6.617329870341344830...E+0000
!                     (qp) : -8.817781266302148406...E+0000
    real   ( kind = rk ), parameter            :: limiterfcxL1  = &
                          -SQRT( -LOG( sqrtpi * EPSILON(one)/two) )
!
!   Very Big Limit where y = erfcx( z < limiterfcxBig ) := 1/(√pi*(x + 0.5/x))
!   with     kind = 1 (sp) : 2.8963093261718750000...E+0003
!            kind = 2 (dp) : 6.7108864000000000000...E+0007
!            kind = 3 (ep) : 3.0370004999760496925...E+0009
!                     (qp) : 7.2057594037927936000...E+0016
    real   ( kind = rk ), parameter            :: limiterfcxBig = &
                                           SQRT( one/EPSILON(one) )
!
!   Upper limit where y = erfcx( z ≤ limiterfcxH ) := (1/√pi)/x = 1/(√pi*x)
!               where y = erfcx( z > limiterfcxH ) := 0.00
!   with     kind = 1 (sp) : 4.7995939143913248987...E+0037
!            kind = 2 (dp) : 2.5355993527615761936...E+0307
!            kind = 3 (ep) : 1.6780852922481148345...E+4931
!                     (qp) : 1.6780852922481148245...E+4931
    real   ( kind = rk ), parameter            :: limiterfcxH   = &
                                       one / ( sqrtpi * TINY(one) )
!
!-----------------------------------------------------------------------------
!
!   Definition of deviations (error)
!
!   typerr = 1: deviation      := true value - measured value
!   typerr = 2: absolute error := ABS( true value - measured value )
!   typerr = 3: relative error := absolute error / true value
!
    integer( kind = ik ), parameter            :: typerr        = 3_ik
!
!-----------------------------------------------------------------------------
!
    end module const
!
!-----------------------------------------------------------------------------
!