!
!   DatError
!     Deliver for x (taberf,taberfc,taberfcx) resp. y (taberfinv,
!     taberfcinv,taberfcxinv) the corresponding tabulated function
!     value.
!   Reference:
!     Milton Abramowitz, Irene A. Stegun, "Handbook of Mathematical
!       Functions with Formulas, Graphs, and Mathematical Tables",
!       Ninth Dover printing, 1972, pages 310-318, tables 7.1 - 7.4,
!     Georgi Z. Georgiev, "Error Function Calculator", [online] 
!       Available at: https://www.gigacalculator.com/calculators/
!       error-function-calculator.php URL [Accessed Date: 30 Dec,
!       2023]
!     Keisan Online Calculator, provided by Casio Computer Co., Ltd.
!       ("Casio"), https://keisan.casio.com/menu/system/000000000760
!       Now (30 Dec 2023) only available in Japanese language.
!     Wolfram Alpha LLC. 2009. Wolfram|Alpha.
!       http://www.wolframalpha.com (access August 2024), terms of
!       use: https://www.wolframalpha.com/termsofuse
!   Remarks:
!     ./.
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    subroutine DatError( func, k, xy, taberf )
!
    use kinds, only : ik, rk, rki, rkx
!
    implicit none
!
!   interface
     integer(kind=ik), intent(in )   :: func     ! input: function number
!                                                  1 = erf,     2 = erfc
!                                                  3 = erfcx,   4 = erfinv
!                                                  5 = erfcinv, 6 = erfcxinv
     integer(kind=ik), intent(in )   :: k        ! k.-row of table or i=i+1
!                                                  if k>0 := direct selection
     real   (kind=rk), intent(out)   :: xy       ! func=1,2,3: x
!                                                  func=4,5,6: y
     real   (kind=rk), intent(out)   :: taberf   ! func=1,2,3: y =erf(x)...
!                                                  func=4,5,6: x =erfinv(y)...
!   end interface
!
!   local real "kind" (r4 = real*4, r8 = real*8, r16 = real*10/real*16)
    integer(kind=ik), parameter      :: r4       = rkx( 1_ik )
    integer(kind=ik), parameter      :: r8       = rkx( 2_ik )
    integer(kind=ik), parameter      :: r16      = rkx( 3_ik )
    integer(kind=ik)                 :: h        ! help var.
!
!   dimension of the tables
    integer(kind=ik), parameter      :: derf        = 47_ik
    integer(kind=ik), parameter      :: derfc       = 65_ik
    integer(kind=ik), parameter      :: derfcx      = 78_ik
    integer(kind=ik), parameter      :: derfinv     = 31_ik
    integer(kind=ik), parameter      :: derfcinv    = 59_ik
    integer(kind=ik), parameter      :: derfcxinv   = 74_ik
    integer(kind=ik), save           :: i           =  0_ik
!
!------------------------------------ ERF --------------------------------------------------------------------------------------
!
!   Error Function: y = erf(x)
!
!   erf: REAL*4
!
!   random variable x for y = erf(x), kind=1 = real*4 = REAL32
    real   (kind=r4 ), parameter, dimension(derf   )     :: dxerfr4  =           (/ & ! 00
        -HUGE(1.000E+0000_r4 ),        -40.000E+0000_r4,         -28.000E+0000_r4,  & ! 03
            -26.555E+0000_r4,          -26.000E+0000_r4,         -11.000E+0000_r4,  & ! 06
            -10.000E+0000_r4,           -9.000E+0000_r4,          -8.000E+0000_r4,  & ! 09
             -7.000E+0000_r4,           -6.000E+0000_r4,          -5.000E+0000_r4,  & ! 12
             -4.000E+0000_r4,           -3.000E+0000_r4,          -2.000E+0000_r4,  & ! 15
             -1.500E+0000_r4,           -1.000E+0000_r4,          -0.500E+0000_r4,  & ! 18
             -0.250E+0000_r4,           -0.030E+0000_r4,          -1.150E-0016_r4,  & ! 21
             -2.220E-0020_r4,            0.000E+0000_r4,           0.000E+0000_r4,  & ! 24
              0.000E+0000_r4,            2.220E-0020_r4,           1.150E-0016_r4,  & ! 27
              0.030E+0000_r4,            0.250E+0000_r4,           0.500E+0000_r4,  & ! 30
              1.000E+0000_r4,            1.500E+0000_r4,           2.000E+0000_r4,  & ! 33
              3.000E+0000_r4,            4.000E+0000_r4,           5.000E+0000_r4,  & ! 36
              6.000E+0000_r4,            7.000E+0000_r4,           8.000E+0000_r4,  & ! 39
              9.000E+0000_r4,           10.000E+0000_r4,          11.000E+0000_r4,  & ! 42
             26.000E+0000_r4,           26.555E+0000_r4,          28.000E+0000_r4,  & ! 45
             40.000E+0000_r4,       HUGE(1.000E+0000_r4 )                          /) ! 47
!
!   error function, y = erf(x), kind=1 = real*4 = REAL32
    real   (kind=r4 ), parameter, dimension(derf     )   :: dwerfr4  =           (/ & ! 00
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 01
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 02
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 03
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 04
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 05
      -0.999999999999999999999999999999999999999999999999999998559133862E+0000_r4,  & ! 06
      -0.999999999999999999999999999999999999999999997911512416237455242E+0000_r4,  & ! 07
      -0.999999999999999999999999999999999999586296825348618976194609653E+0000_r4,  & ! 08
      -0.999999999999999999999999999988775702827017072920032111556829720E+0000_r4,  & ! 09
      -0.999999999999999999999958161743922205856013859897761000677499704E+0000_r4,  & ! 10
      -0.999999999999999978480263287501086883406649600812615369522485938E+0000_r4,  & ! 11
      -0.999999999998462540205571965149811656514616621109881949685276620E+0000_r4,  & ! 12
      -0.999999984582742099719981147840326513115951427854746410808831657E+0000_r4,  & ! 13
      -0.999977909503001414558627223870417679620152292912600750342761045E+0000_r4,  & ! 14
      -0.995322265018952734162069256367252928610891797040060076738352326E+0000_r4,  & ! 15
      -0.966105146475310727066976261645947858681410479257636780449967846E+0000_r4,  & ! 16
      -0.842700792949714869341220635082609259296066997966302908459937897E+0000_r4,  & ! 17
      -0.520499877813046537682746653891964528736451575757963700058805725E+0000_r4,  & ! 18
      -0.276326390168236932985068267764815712065353977892311254082471931E+0000_r4,  & ! 19
      -0.033841222341735433302216654256875465225733912120593527612714135E+0000_r4,  & ! 20
      -1.297636042159839459980582738589771227029097259497744623005824543E-0016_r4,  & ! 21
      -2.505001750952037914049472764929830281147173272533108523599220557E-0020_r4,  & ! 22
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 23
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 24
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 25
       2.505001750952037914049472764929830281147173272533108523599220557E-0020_r4,  & ! 26
       1.297636042159839459980582738589771227029097259497744623005824543E-0016_r4,  & ! 27
       0.033841222341735433302216654256875465225733912120593527612714135E+0000_r4,  & ! 28
       0.276326390168236932985068267764815712065353977892311254082471931E+0000_r4,  & ! 29
       0.520499877813046537682746653891964528736451575757963700058805725E+0000_r4,  & ! 30
       0.842700792949714869341220635082609259296066997966302908459937897E+0000_r4,  & ! 31
       0.966105146475310727066976261645947858681410479257636780449967846E+0000_r4,  & ! 32
       0.995322265018952734162069256367252928610891797040060076738352326E+0000_r4,  & ! 33
       0.999977909503001414558627223870417679620152292912600750342761045E+0000_r4,  & ! 34
       0.999999984582742099719981147840326513115951427854746410808831657E+0000_r4,  & ! 35
       0.999999999998462540205571965149811656514616621109881949685276620E+0000_r4,  & ! 36
       0.999999999999999978480263287501086883406649600812615369522485938E+0000_r4,  & ! 37
       0.999999999999999999999958161743922205856013859897761000677499704E+0000_r4,  & ! 38
       0.999999999999999999999999999988775702827017072920032111556829720E+0000_r4,  & ! 39
       0.999999999999999999999999999999999999586296825348618976194609653E+0000_r4,  & ! 40
       0.999999999999999999999999999999999999999999997911512416237455242E+0000_r4,  & ! 41
       0.999999999999999999999999999999999999999999999999999998559133862E+0000_r4,  & ! 42
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 43
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 44
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 45
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 46
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4  /) ! 47
!
!   erf: REAL*8
!
!   random variable x for y = erf(x), kind=2 = real*8 = REAL64
    real   (kind=r8 ), parameter, dimension(derf   )     :: dxerfr8  =           (/ & ! 00
        -HUGE(1.000E+0000_r8 ),        -40.000E+0000_r8,         -28.000E+0000_r8,  & ! 03
            -26.555E+0000_r8,          -26.000E+0000_r8,         -11.000E+0000_r8,  & ! 06
            -10.000E+0000_r8,           -9.000E+0000_r8,          -8.000E+0000_r8,  & ! 09
             -7.000E+0000_r8,           -6.000E+0000_r8,          -5.000E+0000_r8,  & ! 12
             -4.000E+0000_r8,           -3.000E+0000_r8,          -2.000E+0000_r8,  & ! 15
             -1.500E+0000_r8,           -1.000E+0000_r8,          -0.500E+0000_r8,  & ! 18
             -0.250E+0000_r8,           -0.030E+0000_r8,          -1.150E-0016_r8,  & ! 21
             -2.220E-0020_r8,           -1.500E-0301_r8,           0.000E+0000_r8,  & ! 24
              1.500E-0301_r8,            2.220E-0020_r8,           1.150E-0016_r8,  & ! 27
              0.030E+0000_r8,            0.250E+0000_r8,           0.500E+0000_r8,  & ! 30
              1.000E+0000_r8,            1.500E+0000_r8,           2.000E+0000_r8,  & ! 33
              3.000E+0000_r8,            4.000E+0000_r8,           5.000E+0000_r8,  & ! 36
              6.000E+0000_r8,            7.000E+0000_r8,           8.000E+0000_r8,  & ! 39
              9.000E+0000_r8,           10.000E+0000_r8,          11.000E+0000_r8,  & ! 42
             26.000E+0000_r8,           26.555E+0000_r8,          28.000E+0000_r8,  & ! 45
             40.000E+0000_r8,       HUGE(1.000E+0000_r8 )                          /) ! 47
!
!   error function, y = erf(x), kind=2 = real*8 = REAL64
    real   (kind=r8 ), parameter, dimension(derf     )   :: dwerfr8  =           (/ & ! 00
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 01
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 02
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 03
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 04
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 05
      -0.999999999999999999999999999999999999999999999999999998559133862E+0000_r8,  & ! 06
      -0.999999999999999999999999999999999999999999997911512416237455242E+0000_r8,  & ! 07
      -0.999999999999999999999999999999999999586296825348618976194609653E+0000_r8,  & ! 08
      -0.999999999999999999999999999988775702827017072920032111556829720E+0000_r8,  & ! 09
      -0.999999999999999999999958161743922205856013859897761000677499704E+0000_r8,  & ! 10
      -0.999999999999999978480263287501086883406649600812615369522485938E+0000_r8,  & ! 11
      -0.999999999998462540205571965149811656514616621109881949685276620E+0000_r8,  & ! 12
      -0.999999984582742099719981147840326513115951427854746410808831657E+0000_r8,  & ! 13
      -0.999977909503001414558627223870417679620152292912600750342761045E+0000_r8,  & ! 14
      -0.995322265018952734162069256367252928610891797040060076738352326E+0000_r8,  & ! 15
      -0.966105146475310727066976261645947858681410479257636780449967846E+0000_r8,  & ! 16
      -0.842700792949714869341220635082609259296066997966302908459937897E+0000_r8,  & ! 17
      -0.520499877813046537682746653891964528736451575757963700058805725E+0000_r8,  & ! 18
      -0.276326390168236932985068267764815712065353977892311254082471931E+0000_r8,  & ! 19
      -0.033841222341735433302216654256875465225733912120593527612714135E+0000_r8,  & ! 20
      -1.297636042159839459980582738589771227029097259497744623005824543E-0016_r8,  & ! 21
      -2.505001750952037914049472764929830281147173272533108523599220557E-0020_r8,  & ! 22
      -1.692568750643268860844238354682317757532151887986996570532257165E-0301_r8,  & ! 23
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 24
       1.692568750643268860844238354682317757532151887986996570532257165E-0301_r8,  & ! 25
       2.505001750952037914049472764929830281147173272533108523599220557E-0020_r8,  & ! 26
       1.297636042159839459980582738589771227029097259497744623005824543E-0016_r8,  & ! 27
       0.033841222341735433302216654256875465225733912120593527612714135E+0000_r8,  & ! 28
       0.276326390168236932985068267764815712065353977892311254082471931E+0000_r8,  & ! 29
       0.520499877813046537682746653891964528736451575757963700058805725E+0000_r8,  & ! 30
       0.842700792949714869341220635082609259296066997966302908459937897E+0000_r8,  & ! 31
       0.966105146475310727066976261645947858681410479257636780449967846E+0000_r8,  & ! 32
       0.995322265018952734162069256367252928610891797040060076738352326E+0000_r8,  & ! 33
       0.999977909503001414558627223870417679620152292912600750342761045E+0000_r8,  & ! 34
       0.999999984582742099719981147840326513115951427854746410808831657E+0000_r8,  & ! 35
       0.999999999998462540205571965149811656514616621109881949685276620E+0000_r8,  & ! 36
       0.999999999999999978480263287501086883406649600812615369522485938E+0000_r8,  & ! 37
       0.999999999999999999999958161743922205856013859897761000677499704E+0000_r8,  & ! 38
       0.999999999999999999999999999988775702827017072920032111556829720E+0000_r8,  & ! 39
       0.999999999999999999999999999999999999586296825348618976194609653E+0000_r8,  & ! 40
       0.999999999999999999999999999999999999999999997911512416237455242E+0000_r8,  & ! 41
       0.999999999999999999999999999999999999999999999999999998559133862E+0000_r8,  & ! 42
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 43
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 44
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 45
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 46
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8  /) ! 47
!
!   erf: REAL*10/16
!
!   random variable x for y = erf(x), kind=3 = real*16 = REAL128 (Silverfrost real*10)
    real   (kind=r16), parameter, dimension(derf   )     :: dxerfr16 =           (/ & ! 00
        -HUGE(1.000E+0000_r16),        -40.000E+0000_r16,        -28.000E+0000_r16, & ! 03
            -26.555E+0000_r16,         -26.000E+0000_r16,        -11.000E+0000_r16, & ! 06
            -10.000E+0000_r16,          -9.000E+0000_r16,         -8.000E+0000_r16, & ! 09
             -7.000E+0000_r16,          -6.000E+0000_r16,         -5.000E+0000_r16, & ! 12
             -4.000E+0000_r16,          -3.000E+0000_r16,         -2.000E+0000_r16, & ! 15
             -1.500E+0000_r16,          -1.000E+0000_r16,         -0.500E+0000_r16, & ! 18
             -0.250E+0000_r16,          -0.030E+0000_r16,         -1.150E-0016_r16, & ! 21
             -2.220E-0020_r16,          -1.500E-0301_r16,          0.000E+0000_r16, & ! 24
              1.500E-0301_r16,           2.220E-0020_r16,          1.150E-0016_r16, & ! 27
              0.030E+0000_r16,           0.250E+0000_r16,          0.500E+0000_r16, & ! 30
              1.000E+0000_r16,           1.500E+0000_r16,          2.000E+0000_r16, & ! 33
              3.000E+0000_r16,           4.000E+0000_r16,          5.000E+0000_r16, & ! 36
              6.000E+0000_r16,           7.000E+0000_r16,          8.000E+0000_r16, & ! 39
              9.000E+0000_r16,          10.000E+0000_r16,         11.000E+0000_r16, & ! 42
             26.000E+0000_r16,          26.555E+0000_r16,         28.000E+0000_r16, & ! 45
             40.000E+0000_r16,      HUGE(1.000E+0000_r16)                          /) ! 47
!
!   error function, y = erf(x), kind=3 = real*16 = REAL128 (Silverfrost real*10)
    real   (kind=r16), parameter, dimension(derf     )   :: dwerfr16 =           (/ & ! 00
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 01
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 02
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 03
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 04
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 05
      -0.999999999999999999999999999999999999999999999999999998559133862E+0000_r16, & ! 06
      -0.999999999999999999999999999999999999999999997911512416237455242E+0000_r16, & ! 07
      -0.999999999999999999999999999999999999586296825348618976194609653E+0000_r16, & ! 08
      -0.999999999999999999999999999988775702827017072920032111556829720E+0000_r16, & ! 09
      -0.999999999999999999999958161743922205856013859897761000677499704E+0000_r16, & ! 10
      -0.999999999999999978480263287501086883406649600812615369522485938E+0000_r16, & ! 11
      -0.999999999998462540205571965149811656514616621109881949685276620E+0000_r16, & ! 12
      -0.999999984582742099719981147840326513115951427854746410808831657E+0000_r16, & ! 13
      -0.999977909503001414558627223870417679620152292912600750342761045E+0000_r16, & ! 14
      -0.995322265018952734162069256367252928610891797040060076738352326E+0000_r16, & ! 15
      -0.966105146475310727066976261645947858681410479257636780449967846E+0000_r16, & ! 16
      -0.842700792949714869341220635082609259296066997966302908459937897E+0000_r16, & ! 17
      -0.520499877813046537682746653891964528736451575757963700058805725E+0000_r16, & ! 18
      -0.276326390168236932985068267764815712065353977892311254082471931E+0000_r16, & ! 19
      -0.033841222341735433302216654256875465225733912120593527612714135E+0000_r16, & ! 20
      -1.297636042159839459980582738589771227029097259497744623005824543E-0016_r16, & ! 21
      -2.505001750952037914049472764929830281147173272533108523599220557E-0020_r16, & ! 22
      -1.692568750643268860844238354682317757532151887986996570532257165E-0301_r16, & ! 23
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 24
       1.692568750643268860844238354682317757532151887986996570532257165E-0301_r16, & ! 25
       2.505001750952037914049472764929830281147173272533108523599220557E-0020_r16, & ! 26
       1.297636042159839459980582738589771227029097259497744623005824543E-0016_r16, & ! 27
       0.033841222341735433302216654256875465225733912120593527612714135E+0000_r16, & ! 28
       0.276326390168236932985068267764815712065353977892311254082471931E+0000_r16, & ! 29
       0.520499877813046537682746653891964528736451575757963700058805725E+0000_r16, & ! 30
       0.842700792949714869341220635082609259296066997966302908459937897E+0000_r16, & ! 31
       0.966105146475310727066976261645947858681410479257636780449967846E+0000_r16, & ! 32
       0.995322265018952734162069256367252928610891797040060076738352326E+0000_r16, & ! 33
       0.999977909503001414558627223870417679620152292912600750342761045E+0000_r16, & ! 34
       0.999999984582742099719981147840326513115951427854746410808831657E+0000_r16, & ! 35
       0.999999999998462540205571965149811656514616621109881949685276620E+0000_r16, & ! 36
       0.999999999999999978480263287501086883406649600812615369522485938E+0000_r16, & ! 37
       0.999999999999999999999958161743922205856013859897761000677499704E+0000_r16, & ! 38
       0.999999999999999999999999999988775702827017072920032111556829720E+0000_r16, & ! 39
       0.999999999999999999999999999999999999586296825348618976194609653E+0000_r16, & ! 40
       0.999999999999999999999999999999999999999999997911512416237455242E+0000_r16, & ! 41
       0.999999999999999999999999999999999999999999999999999998559133862E+0000_r16, & ! 42
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 43
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 44
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 45
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 46
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16 /) ! 47
!
!   erf: table
!
!      k   x                 y = erf(x), 64 digits exact
!          123.456E+0000     1.234567890123456789012345678901234567890123456789012345678901234E+0000
!
!      1      -HUGE( x )    -1.000000000000000000000000000000000000000000000000000000000000000E+0000  -∞ => -1.00
!      2   -40.000E+0000    -1.000000000000000000000000000000000000000000000000000000000000000E+0000
!      3   -28.000E+0000    -1.000000000000000000000000000000000000000000000000000000000000000E+0000
!      4   -26.555E+0000    -1.000000000000000000000000000000000000000000000000000000000000000E+0000
!      5   -26.000E+0000    -1.000000000000000000000000000000000000000000000000000000000000000E+0000
!      6   -11.000E+0000    -0.999999999999999999999999999999999999999999999999999998559133862E+0000
!      7   -10.000E+0000    -0.999999999999999999999999999999999999999999997911512416237455242E+0000
!      8    -9.000E+0000    -0.999999999999999999999999999999999999586296825348618976194609653E+0000
!      9    -8.000E+0000    -0.999999999999999999999999999988775702827017072920032111556829720E+0000
!     10    -7.000E+0000    -0.999999999999999999999958161743922205856013859897761000677499704E+0000
!     11    -6.000E+0000    -0.999999999999999978480263287501086883406649600812615369522485938E+0000
!     12    -5.000E+0000    -0.999999999998462540205571965149811656514616621109881949685276620E+0000
!     13    -4.000E+0000    -0.999999984582742099719981147840326513115951427854746410808831657E+0000
!     14    -3.000E+0000    -0.999977909503001414558627223870417679620152292912600750342761045E+0000
!     15    -2.000E+0000    -0.995322265018952734162069256367252928610891797040060076738352326E+0000
!     16    -1.500E+0000    -0.966105146475310727066976261645947858681410479257636780449967846E+0000
!     17    -1.000E+0000    -0.842700792949714869341220635082609259296066997966302908459937897E+0000
!     18    -0.500E+0000    -0.520499877813046537682746653891964528736451575757963700058805725E+0000
!     19    -0.250E+0000    -0.276326390168236932985068267764815712065353977892311254082471931E+0000
!     20    -0.030E+0000    -0.033841222341735433302216654256875465225733912120593527612714135E+0000
!     21    -1.150E-0016    -1.297636042159839459980582738589771227029097259497744623005824543E-0016
!     22    -2.220E-0020    -2.505001750952037914049472764929830281147173272533108523599220557E-0020
!     23    -1.500E-0301    -1.692568750643268860844238354682317757532151887986996570532257165E-0301  r10/16, r8, r4=0
!     24     0.000E+0000     0.000000000000000000000000000000000000000000000000000000000000000E+0000
!     25     1.500E-0301     1.692568750643268860844238354682317757532151887986996570532257165E-0301  r10/16, r8, r4=0
!     26     2.220E-0020     2.505001750952037914049472764929830281147173272533108523599220557E-0020
!     27     1.150E-0016     1.297636042159839459980582738589771227029097259497744623005824543E-0016
!     28     0.030E+0000     0.033841222341735433302216654256875465225733912120593527612714135E+0000
!     29     0.250E+0000     0.276326390168236932985068267764815712065353977892311254082471931E+0000
!     30     0.500E+0000     0.520499877813046537682746653891964528736451575757963700058805725E+0000
!     31     1.000E+0000     0.842700792949714869341220635082609259296066997966302908459937897E+0000
!     32     1.500E+0000     0.966105146475310727066976261645947858681410479257636780449967846E+0000
!     33     2.000E+0000     0.995322265018952734162069256367252928610891797040060076738352326E+0000
!     34     3.000E+0000     0.999977909503001414558627223870417679620152292912600750342761045E+0000
!     35     4.000E+0000     0.999999984582742099719981147840326513115951427854746410808831657E+0000
!     36     5.000E+0000     0.999999999998462540205571965149811656514616621109881949685276620E+0000
!     37     6.000E+0000     0.999999999999999978480263287501086883406649600812615369522485938E+0000
!     38     7.000E+0000     0.999999999999999999999958161743922205856013859897761000677499704E+0000
!     39     8.000E+0000     0.999999999999999999999999999988775702827017072920032111556829720E+0000
!     40     9.000E+0000     0.999999999999999999999999999999999999586296825348618976194609653E+0000
!     41    10.000E+0000     0.999999999999999999999999999999999999999999997911512416237455242E+0000
!     42    11.000E+0000     0.999999999999999999999999999999999999999999999999999998559133862E+0000
!     43    26.000E+0000     1.000000000000000000000000000000000000000000000000000000000000000E+0000
!     44    26.555E+0000     1.000000000000000000000000000000000000000000000000000000000000000E+0000
!     45    28.000E+0000     1.000000000000000000000000000000000000000000000000000000000000000E+0000
!     46    40.000E+0000     1.000000000000000000000000000000000000000000000000000000000000000E+0000
!     47      +HUGE( x )     1.000000000000000000000000000000000000000000000000000000000000000E+0000  +∞ => +1.00
!
!------------------------------------ ERFC -------------------------------------------------------------------------------------
!
!   Complementary Error Function: y = erfc(x)
!
!   erfc: REAL*4
!
!   random variable x for y = erfc(x), kind=1 = real*4 = REAL32
    real   (kind=r4 ), parameter, dimension(derfc    )   :: dxerfcr4  =          (/ & ! 00
        -HUGE(1.000E+0000_r4 ),        -27.000E+0000_r4,         -26.555E+0000_r4,  & ! 03
            -26.000E+0000_r4,          -13.000E+0000_r4,         -12.000E+0000_r4,  & ! 06
            -10.000E+0000_r4,           -9.000E+0000_r4,          -8.000E+0000_r4,  & ! 09
             -7.000E+0000_r4,           -6.000E+0000_r4,          -5.000E+0000_r4,  & ! 12
             -4.000E+0000_r4,           -3.000E+0000_r4,          -2.000E+0000_r4,  & ! 15
             -1.500E+0000_r4,           -1.000E+0000_r4,          -0.750E+0000_r4,  & ! 18
             -0.500E+0000_r4,           -0.250E+0000_r4,          -0.030E+0000_r4,  & ! 21
             -1.150E-0016_r4,           -2.220E-0020_r4,           0.000E+0000_r4,  & ! 24
              2.220E-0020_r4,            1.150E-0016_r4,           0.030E+0000_r4,  & ! 27
              0.250E+0000_r4,            0.500E+0000_r4,           0.750E+0000_r4,  & ! 30
              1.000E+0000_r4,            1.500E+0000_r4,           2.000E+0000_r4,  & ! 33
              2.500E+0000_r4,            3.000E+0000_r4,           3.500E+0000_r4,  & ! 36
              4.000E+0000_r4,            4.500E+0000_r4,           5.000E+0000_r4,  & ! 39
              6.000E+0000_r4,            7.000E+0000_r4,           8.000E+0000_r4,  & ! 42
              9.000E+0000_r4,           10.000E+0000_r4,          11.000E+0000_r4,  & ! 45
             12.000E+0000_r4,           13.000E+0000_r4,          14.000E+0000_r4,  & ! 48
             15.000E+0000_r4,           16.000E+0000_r4,          17.000E+0000_r4,  & ! 51
             18.000E+0000_r4,           19.000E+0000_r4,          20.000E+0000_r4,  & ! 54
             21.000E+0000_r4,           22.000E+0000_r4,          23.000E+0000_r4,  & ! 57
             24.000E+0000_r4,           25.000E+0000_r4,          26.000E+0000_r4,  & ! 60
             26.555E+0000_r4,           27.000E+0000_r4,          28.000E+0000_r4,  & ! 63
             49.000E+0000_r4,      +HUGE(1.000E+0000_r4)                           /) ! 66
!
!   complementary error Function, y = erfc(x), kind=1 = real*4 = REAL32
    real   (kind=r4 ), parameter, dimension(derfc    )   :: dwerfcr4  =         (/  & ! 00
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 01  -∞ => +2.00
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 02
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 03
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 04
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 05
       1.999999999999999999999999999999999999999999999999999999999999999E+0000_r4,  & ! 06
       1.999999999999999999999999999999999999999999997911512416237455242E+0000_r4,  & ! 07
       1.999999999999999999999999999999999999586296825348618976194609653E+0000_r4,  & ! 08
       1.999999999999999999999999999988775702827017072920032111556829720E+0000_r4,  & ! 09
       1.999999999999999999999958161743922205856013859897761000677499703E+0000_r4,  & ! 10
       1.999999999999999978480263287501086883406649600812615369522485938E+0000_r4,  & ! 11
       1.999999999998462540205571965149811656514616621109881949685276620E+0000_r4,  & ! 12
       1.999999984582742099719981147840326513115951427854746410808831657E+0000_r4,  & ! 13
       1.999977909503001414558627223870417679620152292912600750342761045E+0000_r4,  & ! 14
       1.995322265018952734162069256367252928610891797040060076738352326E+0000_r4,  & ! 15
       1.966105146475310727066976261645947858681410479257636780449967846E+0000_r4,  & ! 16
       1.842700792949714869341220635082609259296066997966302908459937897E+0000_r4,  & ! 17
       1.711155633653515131598937834591410777374205954096537232278133397E+0000_r4,  & ! 18
       1.520499877813046537682746653891964528736451575757963700058805725E+0000_r4,  & ! 19
       1.276326390168236932985068267764815712065353977892311254082471931E+0000_r4,  & ! 20
       1.033841222341735433302216654256875465225733912120593527612714135E+0000_r4,  & ! 21
       1.000000000000000129763604215983945998058273858977122702909725949E+0000_r4,  & ! 22
       1.000000000000000000025050017509520379140494727649298302811471732E+0000_r4,  & ! 23
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 24
       9.999999999999999999749499824904796208595052723507016971885282672E-0001_r4,  & ! 25
       9.999999999999998702363957840160540019417261410228772970902740502E-0001_r4,  & ! 26
       9.661587776582645666977833457431245347742660878794064723872858648E-0001_r4,  & ! 27
       7.236736098317630670149317322351842879346460221076887459175280687E-0001_r4,  & ! 28
       4.795001221869534623172533461080354712635484242420362999411942743E-0001_r4,  & ! 29
       2.888443663464848684010621654085892226257940459034627677218666028E-0001_r4,  & ! 30
       1.572992070502851306587793649173907407039330020336970915400621021E-0001_r4,  & ! 31
       3.389485352468927293302373835405214131858952074236321955003215355E-0002_r4,  & ! 32
       4.677734981047265837930743632747071389108202959939923261647673799E-0003_r4,  & ! 33
       4.069520174449589395642157399749127203486774037134201391207787690E-0004_r4,  & ! 34
       2.209049699858544137277612958232037984770708739924965723895484294E-0005_r4,  & ! 35
       7.430983723414127455236837560956357206600921727974625911095699435E-0007_r4,  & ! 36
       1.541725790028001885215967348688404857214525358919116834290499421E-0008_r4,  & ! 37
       1.966160441542887476279160367664332660577670599611646975681185960E-0010_r4,  & ! 38
       1.537459794428034850188343485383378890118050314723379930687914055E-0012_r4,  & ! 39
       2.151973671249891311659335039918738463047751406168854210052789205E-0017_r4,  & ! 40
       4.183825607779414398614010223899932250029617413812460152041634353E-0023_r4,  & ! 41
       1.122429717298292707996788844317027909343192916447896338591267524E-0029_r4,  & ! 42
       4.137031746513810238053903467362524595710191985947962459805909901E-0037_r4,  & ! 43
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 44  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 45  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 46  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 47  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 48  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 49  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 50  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 51  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 52  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 53  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 54  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 55  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 56  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 57  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 58  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 59  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 60  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 61  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 62  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 63  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 64  r4 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4  /) ! 65  +∞ => +0.00
!
!   erfc: REAL*8
!
!   random variable x for y = erfc(x), kind=2 = real*8 = REAL64
    real   (kind=r8 ), parameter, dimension(derfc    )   :: dxerfcr8  =          (/ & ! 00
        -HUGE(1.000E+0000_r8 ),        -27.000E+0000_r8,         -26.555E+0000_r8,  & ! 03
            -26.000E+0000_r8,          -13.000E+0000_r8,         -12.000E+0000_r8,  & ! 06
            -10.000E+0000_r8,           -9.000E+0000_r8,          -8.000E+0000_r8,  & ! 09
             -7.000E+0000_r8,           -6.000E+0000_r8,          -5.000E+0000_r8,  & ! 12
             -4.000E+0000_r8,           -3.000E+0000_r8,          -2.000E+0000_r8,  & ! 15
             -1.500E+0000_r8,           -1.000E+0000_r8,          -0.750E+0000_r8,  & ! 18
             -0.500E+0000_r8,           -0.250E+0000_r8,          -0.030E+0000_r8,  & ! 21
             -1.150E-0016_r8,           -2.220E-0020_r8,           0.000E+0000_r8,  & ! 24
              2.220E-0020_r8,            1.150E-0016_r8,           0.030E+0000_r8,  & ! 27
              0.250E+0000_r8,            0.500E+0000_r8,           0.750E+0000_r8,  & ! 30
              1.000E+0000_r8,            1.500E+0000_r8,           2.000E+0000_r8,  & ! 33
              2.500E+0000_r8,            3.000E+0000_r8,           3.500E+0000_r8,  & ! 36
              4.000E+0000_r8,            4.500E+0000_r8,           5.000E+0000_r8,  & ! 39
              6.000E+0000_r8,            7.000E+0000_r8,           8.000E+0000_r8,  & ! 42
              9.000E+0000_r8,           10.000E+0000_r8,          11.000E+0000_r8,  & ! 45
             12.000E+0000_r8,           13.000E+0000_r8,          14.000E+0000_r8,  & ! 48
             15.000E+0000_r8,           16.000E+0000_r8,          17.000E+0000_r8,  & ! 51
             18.000E+0000_r8,           19.000E+0000_r8,          20.000E+0000_r8,  & ! 54
             21.000E+0000_r8,           22.000E+0000_r8,          23.000E+0000_r8,  & ! 57
             24.000E+0000_r8,           25.000E+0000_r8,          26.000E+0000_r8,  & ! 60
             26.555E+0000_r8,           27.000E+0000_r8,          28.000E+0000_r8,  & ! 63
             49.000E+0000_r8,      +HUGE(1.000E+0000_r8)                           /) ! 65
!
!   complementary error Function, y = erfc(x), kind=2 = real*8 = REAL64
    real   (kind=r8 ), parameter, dimension(derfc    )   :: dwerfcr8  =         (/  & ! 00
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 01  -∞ => +2.00
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 02
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 03
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 04
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 05
       1.999999999999999999999999999999999999999999999999999999999999999E+0000_r8,  & ! 06
       1.999999999999999999999999999999999999999999997911512416237455242E+0000_r8,  & ! 07
       1.999999999999999999999999999999999999586296825348618976194609653E+0000_r8,  & ! 08
       1.999999999999999999999999999988775702827017072920032111556829720E+0000_r8,  & ! 09
       1.999999999999999999999958161743922205856013859897761000677499703E+0000_r8,  & ! 10
       1.999999999999999978480263287501086883406649600812615369522485938E+0000_r8,  & ! 11
       1.999999999998462540205571965149811656514616621109881949685276620E+0000_r8,  & ! 12
       1.999999984582742099719981147840326513115951427854746410808831657E+0000_r8,  & ! 13
       1.999977909503001414558627223870417679620152292912600750342761045E+0000_r8,  & ! 14
       1.995322265018952734162069256367252928610891797040060076738352326E+0000_r8,  & ! 15
       1.966105146475310727066976261645947858681410479257636780449967846E+0000_r8,  & ! 16
       1.842700792949714869341220635082609259296066997966302908459937897E+0000_r8,  & ! 17
       1.711155633653515131598937834591410777374205954096537232278133397E+0000_r8,  & ! 18
       1.520499877813046537682746653891964528736451575757963700058805725E+0000_r8,  & ! 19
       1.276326390168236932985068267764815712065353977892311254082471931E+0000_r8,  & ! 20
       1.033841222341735433302216654256875465225733912120593527612714135E+0000_r8,  & ! 21
       1.000000000000000129763604215983945998058273858977122702909725949E+0000_r8,  & ! 22
       1.000000000000000000025050017509520379140494727649298302811471732E+0000_r8,  & ! 23
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 24
       9.999999999999999999749499824904796208595052723507016971885282672E-0001_r8,  & ! 25
       9.999999999999998702363957840160540019417261410228772970902740502E-0001_r8,  & ! 26
       9.661587776582645666977833457431245347742660878794064723872858648E-0001_r8,  & ! 27
       7.236736098317630670149317322351842879346460221076887459175280687E-0001_r8,  & ! 28
       4.795001221869534623172533461080354712635484242420362999411942743E-0001_r8,  & ! 29
       2.888443663464848684010621654085892226257940459034627677218666028E-0001_r8,  & ! 30
       1.572992070502851306587793649173907407039330020336970915400621021E-0001_r8,  & ! 31
       3.389485352468927293302373835405214131858952074236321955003215355E-0002_r8,  & ! 32
       4.677734981047265837930743632747071389108202959939923261647673799E-0003_r8,  & ! 33
       4.069520174449589395642157399749127203486774037134201391207787690E-0004_r8,  & ! 34
       2.209049699858544137277612958232037984770708739924965723895484294E-0005_r8,  & ! 35
       7.430983723414127455236837560956357206600921727974625911095699435E-0007_r8,  & ! 36
       1.541725790028001885215967348688404857214525358919116834290499421E-0008_r8,  & ! 37
       1.966160441542887476279160367664332660577670599611646975681185960E-0010_r8,  & ! 38
       1.537459794428034850188343485383378890118050314723379930687914055E-0012_r8,  & ! 39
       2.151973671249891311659335039918738463047751406168854210052789205E-0017_r8,  & ! 40
       4.183825607779414398614010223899932250029617413812460152041634353E-0023_r8,  & ! 41
       1.122429717298292707996788844317027909343192916447896338591267524E-0029_r8,  & ! 42
       4.137031746513810238053903467362524595710191985947962459805909901E-0037_r8,  & ! 43
       2.088487583762544757000786294957788611560818119321163727012213713E-0045_r8,  & ! 44
       1.440866137943694680339809702856082753964395230973580405075797246E-0054_r8,  & ! 45
       1.356261169205904212780306156590417572666782233288104929912379537E-0064_r8,  & ! 46
       1.739557315466724521804198548243026985088574583449008314936058740E-0075_r8,  & ! 47
       3.037229847750311665115172806783328447911896669343422595150070601E-0087_r8,  & ! 48
       7.212994172451206666565066558692927109934090929825383240467341996E-0100_r8,  & ! 49
       2.328485751571530693364872854573442597534396948094948021516495095E-0113_r8,  & ! 50
       1.021228015094260881145599235077652994401543730587571172553506555E-0127_r8,  & ! 51
       6.082369231816399307668466715702274949587610076867576851592785891E-0143_r8,  & ! 52
       4.917722839256475446413297625239608170930672249989259036315216983E-0159_r8,  & ! 53
       5.395865611607900928934999167905345604088272670923605283470103784E-0176_r8,  & ! 54
       8.032453871022455669021356947138268888967875692513378852786023784E-0194_r8,  & ! 55
       1.621905860933472513052034647026123265177675596121851615118740214E-0212_r8,  & ! 56
       4.441265948088057244074884428946738565068812405566605852511837779E-0232_r8,  & ! 57
       1.648982583151933514218512437543746903943061790303826174801039526E-0252_r8,  & ! 58
       8.300172571196522752044012769513722768714223191886699596054448582E-0274_r8,  & ! 59
       5.663192408856142846475727896926092580328777560925807368646253847E-0296_r8,  & ! 60
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 61  r8 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 62  r8 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 63  r8 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 64  r8 -> +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8  /) ! 65  +∞ => +0.00
!
!   erfc: REAL*10/16
!
!   random variable x for y = erfc(x), kind=3 = real*16 = REAL128 (Silverfrost real*10)
    real   (kind=r16), parameter, dimension(derfc    )   :: dxerfcr16 =          (/ & ! 00
        -HUGE(1.000E+0000_r16),        -27.000E+0000_r16,        -26.555E+0000_r16, & ! 03
            -26.000E+0000_r16,         -13.000E+0000_r16,        -12.000E+0000_r16, & ! 06
            -10.000E+0000_r16,          -9.000E+0000_r16,         -8.000E+0000_r16, & ! 09
             -7.000E+0000_r16,          -6.000E+0000_r16,         -5.000E+0000_r16, & ! 12
             -4.000E+0000_r16,          -3.000E+0000_r16,         -2.000E+0000_r16, & ! 15
             -1.500E+0000_r16,          -1.000E+0000_r16,         -0.750E+0000_r16, & ! 18
             -0.500E+0000_r16,          -0.250E+0000_r16,         -0.030E+0000_r16, & ! 21
             -1.150E-0016_r16,          -2.220E-0020_r16,          0.000E+0000_r16, & ! 24
              2.220E-0020_r16,           1.150E-0016_r16,          0.030E+0000_r16, & ! 27
              0.250E+0000_r16,           0.500E+0000_r16,          0.750E+0000_r16, & ! 30
              1.000E+0000_r16,           1.500E+0000_r16,          2.000E+0000_r16, & ! 33
              2.500E+0000_r16,           3.000E+0000_r16,          3.500E+0000_r16, & ! 36
              4.000E+0000_r16,           4.500E+0000_r16,          5.000E+0000_r16, & ! 39
              6.000E+0000_r16,           7.000E+0000_r16,          8.000E+0000_r16, & ! 42
              9.000E+0000_r16,          10.000E+0000_r16,         11.000E+0000_r16, & ! 45
             12.000E+0000_r16,          13.000E+0000_r16,         14.000E+0000_r16, & ! 48
             15.000E+0000_r16,          16.000E+0000_r16,         17.000E+0000_r16, & ! 51
             18.000E+0000_r16,          19.000E+0000_r16,         20.000E+0000_r16, & ! 54
             21.000E+0000_r16,          22.000E+0000_r16,         23.000E+0000_r16, & ! 57
             24.000E+0000_r16,          25.000E+0000_r16,         26.000E+0000_r16, & ! 60
             26.555E+0000_r16,          27.000E+0000_r16,         28.000E+0000_r16, & ! 63
             49.000E+0000_r16,     +HUGE(1.000E+0000_r16)                          /) ! 65
!
!   complementary error Function, y = erfc(x), kind=3 = real*16 = REAL128 (S.-frost r*10)
    real   (kind=r16), parameter, dimension(derfc    )   :: dwerfcr16 =         (/  & ! 00
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 01  -∞ => +2.00
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 02
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 03
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 04
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 05
       1.999999999999999999999999999999999999999999999999999999999999999E+0000_r16, & ! 06
       1.999999999999999999999999999999999999999999997911512416237455242E+0000_r16, & ! 07
       1.999999999999999999999999999999999999586296825348618976194609653E+0000_r16, & ! 08
       1.999999999999999999999999999988775702827017072920032111556829720E+0000_r16, & ! 09
       1.999999999999999999999958161743922205856013859897761000677499703E+0000_r16, & ! 10
       1.999999999999999978480263287501086883406649600812615369522485938E+0000_r16, & ! 11
       1.999999999998462540205571965149811656514616621109881949685276620E+0000_r16, & ! 12
       1.999999984582742099719981147840326513115951427854746410808831657E+0000_r16, & ! 13
       1.999977909503001414558627223870417679620152292912600750342761045E+0000_r16, & ! 14
       1.995322265018952734162069256367252928610891797040060076738352326E+0000_r16, & ! 15
       1.966105146475310727066976261645947858681410479257636780449967846E+0000_r16, & ! 16
       1.842700792949714869341220635082609259296066997966302908459937897E+0000_r16, & ! 17
       1.711155633653515131598937834591410777374205954096537232278133397E+0000_r16, & ! 18
       1.520499877813046537682746653891964528736451575757963700058805725E+0000_r16, & ! 19
       1.276326390168236932985068267764815712065353977892311254082471931E+0000_r16, & ! 20
       1.033841222341735433302216654256875465225733912120593527612714135E+0000_r16, & ! 21
       1.000000000000000129763604215983945998058273858977122702909725949E+0000_r16, & ! 22
       1.000000000000000000025050017509520379140494727649298302811471732E+0000_r16, & ! 23
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 24
       9.999999999999999999749499824904796208595052723507016971885282672E-0001_r16, & ! 25
       9.999999999999998702363957840160540019417261410228772970902740502E-0001_r16, & ! 26
       9.661587776582645666977833457431245347742660878794064723872858648E-0001_r16, & ! 27
       7.236736098317630670149317322351842879346460221076887459175280687E-0001_r16, & ! 28
       4.795001221869534623172533461080354712635484242420362999411942743E-0001_r16, & ! 29
       2.888443663464848684010621654085892226257940459034627677218666028E-0001_r16, & ! 30
       1.572992070502851306587793649173907407039330020336970915400621021E-0001_r16, & ! 31
       3.389485352468927293302373835405214131858952074236321955003215355E-0002_r16, & ! 32
       4.677734981047265837930743632747071389108202959939923261647673799E-0003_r16, & ! 33
       4.069520174449589395642157399749127203486774037134201391207787690E-0004_r16, & ! 34
       2.209049699858544137277612958232037984770708739924965723895484294E-0005_r16, & ! 35
       7.430983723414127455236837560956357206600921727974625911095699435E-0007_r16, & ! 36
       1.541725790028001885215967348688404857214525358919116834290499421E-0008_r16, & ! 37
       1.966160441542887476279160367664332660577670599611646975681185960E-0010_r16, & ! 38
       1.537459794428034850188343485383378890118050314723379930687914055E-0012_r16, & ! 39
       2.151973671249891311659335039918738463047751406168854210052789205E-0017_r16, & ! 40
       4.183825607779414398614010223899932250029617413812460152041634353E-0023_r16, & ! 41
       1.122429717298292707996788844317027909343192916447896338591267524E-0029_r16, & ! 42
       4.137031746513810238053903467362524595710191985947962459805909901E-0037_r16, & ! 43
       2.088487583762544757000786294957788611560818119321163727012213713E-0045_r16, & ! 44
       1.440866137943694680339809702856082753964395230973580405075797246E-0054_r16, & ! 45
       1.356261169205904212780306156590417572666782233288104929912379537E-0064_r16, & ! 46
       1.739557315466724521804198548243026985088574583449008314936058740E-0075_r16, & ! 47
       3.037229847750311665115172806783328447911896669343422595150070601E-0087_r16, & ! 48
       7.212994172451206666565066558692927109934090929825383240467341996E-0100_r16, & ! 49
       2.328485751571530693364872854573442597534396948094948021516495095E-0113_r16, & ! 50
       1.021228015094260881145599235077652994401543730587571172553506555E-0127_r16, & ! 51
       6.082369231816399307668466715702274949587610076867576851592785891E-0143_r16, & ! 52
       4.917722839256475446413297625239608170930672249989259036315216983E-0159_r16, & ! 53
       5.395865611607900928934999167905345604088272670923605283470103784E-0176_r16, & ! 54
       8.032453871022455669021356947138268888967875692513378852786023784E-0194_r16, & ! 55
       1.621905860933472513052034647026123265177675596121851615118740214E-0212_r16, & ! 56
       4.441265948088057244074884428946738565068812405566605852511837779E-0232_r16, & ! 57
       1.648982583151933514218512437543746903943061790303826174801039526E-0252_r16, & ! 58
       8.300172571196522752044012769513722768714223191886699596054448582E-0274_r16, & ! 59
       5.663192408856142846475727896926092580328777560925807368646253847E-0296_r16, & ! 60
       1.192310292190008994893575123620341321165641961929944346694965152E-0308_r16, & ! 61  r*10/16
       5.237048923789255685016067682849547090933912547968670799599215945E-0319_r16, & ! 62  r*10/16
       6.563215840328784152380910481606274774134162689948282618021777806E-0343_r16, & ! 63  r*10/16
       2.089717312149417570964225632118560593923169250964555005070936907E-1045_r16, & ! 64  r*10/16
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r16 /) ! 65  +∞ => +0.00
!
!   erfc: table
!
!      k   x                 y = erfc(x), 64 digits exact
!          123.456E+0000     1.234567890123456789012345678901234567890123456789012345678901234E+0000
!
!      1      -HUGE( x )     2.000000000000000000000000000000000000000000000000000000000000000E+0000  -∞ => +2.00
!      2   -27.000E+0000     2.000000000000000000000000000000000000000000000000000000000000000E+0000  r4
!      3   -26.555E+0000     2.000000000000000000000000000000000000000000000000000000000000000E+0000
!      4   -26.000E+0000     2.000000000000000000000000000000000000000000000000000000000000000E+0000
!      5   -13.000E+0000     2.000000000000000000000000000000000000000000000000000000000000000E+0000
!      6   -12.000E+0000     1.999999999999999999999999999999999999999999999999999999999999999E+0000
!      7   -10.000E+0000     1.999999999999999999999999999999999999999999997911512416237455242E+0000
!      8    -9.000E+0000     1.999999999999999999999999999999999999586296825348618976194609653E+0000
!      9    -8.000E+0000     1.999999999999999999999999999988775702827017072920032111556829720E+0000
!     10    -7.000E+0000     1.999999999999999999999958161743922205856013859897761000677499703E+0000
!     11    -6.000E+0000     1.999999999999999978480263287501086883406649600812615369522485938E+0000
!     12    -5.000E+0000     1.999999999998462540205571965149811656514616621109881949685276620E+0000
!     13    -4.000E+0000     1.999999984582742099719981147840326513115951427854746410808831657E+0000
!     14    -3.000E+0000     1.999977909503001414558627223870417679620152292912600750342761045E+0000
!     15    -2.000E+0000     1.995322265018952734162069256367252928610891797040060076738352326E+0000
!     16    -1.500E+0000     1.966105146475310727066976261645947858681410479257636780449967846E+0000
!     17    -1.000E+0000     1.842700792949714869341220635082609259296066997966302908459937897E+0000
!     18    -0.750E+0000     1.711155633653515131598937834591410777374205954096537232278133397E+0000
!     19    -0.500E+0000     1.520499877813046537682746653891964528736451575757963700058805725E+0000
!     20    -0.250E+0000     1.276326390168236932985068267764815712065353977892311254082471931E+0000
!     21    -0.030E+0000     1.033841222341735433302216654256875465225733912120593527612714135E+0000
!     22    -1.150E-0016     1.000000000000000129763604215983945998058273858977122702909725949E+0000
!     23    -2.220E-0020     1.000000000000000000025050017509520379140494727649298302811471732E+0000
!     24     0.000E+0000     1.000000000000000000000000000000000000000000000000000000000000000E+0000
!     25     2.220E-0020     9.999999999999999999749499824904796208595052723507016971885282672E-0001
!     26     1.150E-0016     9.999999999999998702363957840160540019417261410228772970902740502E-0001
!     27     0.030E+0000     9.661587776582645666977833457431245347742660878794064723872858648E-0001
!     28     0.250E+0000     7.236736098317630670149317322351842879346460221076887459175280687E-0001
!     29     0.500E+0000     4.795001221869534623172533461080354712635484242420362999411942743E-0001
!     30     0.750E+0000     2.888443663464848684010621654085892226257940459034627677218666028E-0001
!     31     1.000E+0000     1.572992070502851306587793649173907407039330020336970915400621021E-0001
!     32     1.500E+0000     3.389485352468927293302373835405214131858952074236321955003215355E-0002
!     33     2.000E+0000     4.677734981047265837930743632747071389108202959939923261647673799E-0003
!     34     2.500E+0000     4.069520174449589395642157399749127203486774037134201391207787690E-0004
!     35     3.000E+0000     2.209049699858544137277612958232037984770708739924965723895484294E-0005
!     36     3.500E+0000     7.430983723414127455236837560956357206600921727974625911095699435E-0007
!     37     4.000E+0000     1.541725790028001885215967348688404857214525358919116834290499421E-0008
!     38     4.500E+0000     1.966160441542887476279160367664332660577670599611646975681185960E-0010
!     39     5.000E+0000     1.537459794428034850188343485383378890118050314723379930687914055E-0012
!     40     6.000E+0000     2.151973671249891311659335039918738463047751406168854210052789205E-0017
!     41     7.000E+0000     4.183825607779414398614010223899932250029617413812460152041634353E-0023
!     42     8.000E+0000     1.122429717298292707996788844317027909343192916447896338591267524E-0029
!     43     9.000E+0000     4.137031746513810238053903467362524595710191985947962459805909901E-0037
!     44    10.000E+0000     2.088487583762544757000786294957788611560818119321163727012213713E-0045  r8
!     45    11.000E+0000     1.440866137943694680339809702856082753964395230973580405075797246E-0054
!     46    12.000E+0000     1.356261169205904212780306156590417572666782233288104929912379537E-0064
!     47    13.000E+0000     1.739557315466724521804198548243026985088574583449008314936058740E-0075
!     48    14.000E+0000     3.037229847750311665115172806783328447911896669343422595150070601E-0087
!     49    15.000E+0000     7.212994172451206666565066558692927109934090929825383240467341996E-0100
!     50    16.000E+0000     2.328485751571530693364872854573442597534396948094948021516495095E-0113
!     51    17.000E+0000     1.021228015094260881145599235077652994401543730587571172553506555E-0127
!     52    18.000E+0000     6.082369231816399307668466715702274949587610076867576851592785891E-0143
!     53    19.000E+0000     4.917722839256475446413297625239608170930672249989259036315216983E-0159
!     54    20.000E+0000     5.395865611607900928934999167905345604088272670923605283470103784E-0176
!     55    21.000E+0000     8.032453871022455669021356947138268888967875692513378852786023784E-0194
!     56    22.000E+0000     1.621905860933472513052034647026123265177675596121851615118740214E-0212
!     57    23.000E+0000     4.441265948088057244074884428946738565068812405566605852511837779E-0232
!     58    24.000E+0000     1.648982583151933514218512437543746903943061790303826174801039526E-0252
!     59    25.000E+0000     8.300172571196522752044012769513722768714223191886699596054448582E-0274
!     60    26.000E+0000     5.663192408856142846475727896926092580328777560925807368646253847E-0296
!     61    26.555E+0000     1.192310292190008994893575123620341321165641961929944346694965152E-0308  r*10/16
!     62    27.000E+0000     5.237048923789255685016067682849547090933912547968670799599215945E-0319
!     63    28.000E+0000     6.563215840328784152380910481606274774134162689948282618021777806E-0343
!     64    49.000E+0000     2.089717312149417570964225632118560593923169250964555005070936907E-1045
!     65      +HUGE( x )     0.000000000000000000000000000000000000000000000000000000000000000E+0000  +∞ => +0.00
!
!------------------------------------ ERFCX ------------------------------------------------------------------------------------
!
!   Scaled Complementary Error Function: y = erfcx(x)
!
!   erfcx: REAL*4
!
!   random variable x for y = erfcx(x), kind=1 = real*4 = REAL32
    real   (kind=r4 ), parameter, dimension(derfcx   )   :: dxerfcxr4 =          (/ & ! 00
        -HUGE(1.000E+0000_r4),     -HUGE(1.000E+0000_r4),    -HUGE(1.000E+0000_r4), & ! 03
        -HUGE(1.000E+0000_r4),     -HUGE(1.000E+0000_r4),    -HUGE(1.000E+0000_r4), & ! 06
             -9.000E+0000_r4,           -8.000E+0000_r4,          -7.000E+0000_r4,  & ! 09
             -6.000E+0000_r4,           -5.000E+0000_r4,          -4.000E+0000_r4,  & ! 12
             -3.000E+0000_r4,           -2.000E+0000_r4,          -1.500E+0000_r4,  & ! 15
             -1.000E+0000_r4,           -0.750E+0000_r4,          -0.500E+0000_r4,  & ! 18
             -0.250E+0000_r4,           -0.030E+0000_r4,          -1.150E-0016_r4,  & ! 21
             -2.220E-0020_r4,            0.000E+0000_r4,           2.220E-0020_r4,  & ! 24
              1.150E-0016_r4,            0.030E+0000_r4,           0.250E+0000_r4,  & ! 27
              0.500E+0000_r4,            0.750E+0000_r4,           1.000E+0000_r4,  & ! 30
              1.500E+0000_r4,            2.000E+0000_r4,           2.500E+0000_r4,  & ! 33
              3.000E+0000_r4,            3.500E+0000_r4,           4.000E+0000_r4,  & ! 36
              4.500E+0000_r4,            5.000E+0000_r4,           6.000E+0000_r4,  & ! 39
              7.000E+0000_r4,            8.000E+0000_r4,           9.000E+0000_r4,  & ! 42
             10.000E+0000_r4,           11.000E+0000_r4,          12.000E+0000_r4,  & ! 45
             13.000E+0000_r4,           14.000E+0000_r4,          15.000E+0000_r4,  & ! 48
             16.000E+0000_r4,           17.000E+0000_r4,          18.000E+0000_r4,  & ! 51
             19.000E+0000_r4,           20.000E+0000_r4,          21.000E+0000_r4,  & ! 54
             22.000E+0000_r4,           23.000E+0000_r4,          24.000E+0000_r4,  & ! 57
             25.000E+0000_r4,           26.000E+0000_r4,          26.555E+0000_r4,  & ! 60
             27.000E+0000_r4,           48.000E+0000_r4,           1.000E+0002_r4,  & ! 63
              1.500E+0002_r4,            1.000E+0003_r4,           1.000E+0004_r4,  & ! 66
              1.500E+0004_r4,            4.800E+0004_r4,           6.000E+0004_r4,  & ! 69
              7.000E+0007_r4,      +HUGE(1.000E+0000_r4),    +HUGE(1.000E+0000_r4), & ! 72
        +HUGE(1.000E+0000_r4),     +HUGE(1.000E+0000_r4),    +HUGE(1.000E+0000_r4), & ! 75
        +HUGE(1.000E+0000_r4),    +HUGE(1.000E+0000_r4),     +HUGE(1.000E+0000_r4) /) ! 78
!
!   scaled complementary error function, y = erfcx(x), kind=1 = real*4 = REAL32
    real   (kind=r4 ), parameter, dimension(derfcx   )   :: dwerfcxr4 =          (/ & ! 00
                                                               +HUGE(1.0E+0000_r4), & ! 01  -∞ => +∞
                                                               +HUGE(1.0E+0000_r4), & ! 02  -∞ => +∞
                                                               +HUGE(1.0E+0000_r4), & ! 03  -∞ => +∞
                                                               +HUGE(1.0E+0000_r4), & ! 04  -∞ => +∞
                                                               +HUGE(1.0E+0000_r4), & ! 05  -∞ => +∞
                                                               +HUGE(1.0E+0000_r4), & ! 06  -∞ => +∞
       3.012194629170061096705188260335349963175715743878784001794360069E+0035_r4,  & ! 07
       1.247029816162323376581847741778695432346190276474884757933513822E+0028_r4,  & ! 08 
       3.814693144990199381050119881853367741842889596589865930650911766E+0021_r4,  & ! 09
       8.622463094230390361450276785175496392091021908830770647053340636E+0015_r4,  & ! 10
       1.440097986746610404105896343058821037439553145762016106106219209E+0011_r4,  & ! 11
       1.777222090401628764846465759211772930179995731346404213150261392E+0007_r4,  & ! 12
       1.620598885399958662546957408405020630903572419029912007078465534E+0004_r4,  & ! 13
       1.089409043899779724123554338248132140422788747719728953862052389E+0002_r4,  & ! 14
       1.865388625626273393874641550130021127851120319405663701880037332E+0001_r4,  & ! 15
       5.008980080762283466309824598214809814694334684235666486188395484E+0000_r4,  & ! 16
       3.003171663627452308696662251659107601538747047728341647273491144E+0000_r4,  & ! 17
       1.952360489182557093276047713441130979890255339297294572744188004E+0000_r4,  & ! 18
       1.358642370104722115210042016948988220013808502272091357355815027E+0000_r4,  & ! 19
       1.034772098273178019832013143072154484345890514100185925367614013E+0000_r4,  & ! 20
       1.000000000000000129763604215983959223058273858978838826575482337E+0000_r4,  & ! 21
       1.000000000000000000025050017509520379140987567649298302811484078E+0000_r4,  & ! 22
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 23
       9.999999999999999999749499824904796208599981123507016971885159216E-0001_r4,  & ! 24
       9.999999999999998702363957840160672269417261410211611734245176626E-0001_r4,  & ! 25
       9.670287119698766650109632717495573341360302147775554874047279156E-0001_r4,  & ! 26
       7.703465477309967439167391723367911261876423850266152456827999940E-0001_r4,  & ! 27
       6.156903441929258748707934226837419367823063912656316056908265890E-0001_r4,  & ! 28
       5.069376502931448057914318215388277759377280127631554128297343480E-0001_r4,  & ! 29
       4.275835761558070044107503444905151808201595031642526637455397707E-0001_r4,  & ! 30
       3.215854164543175023543225877232655690292467412681370014065243116E-0001_r4,  & ! 31
       2.553956763105057438650885809085427633025993052552420654479489936E-0001_r4,  & ! 32
       2.108063640611435806471120484067023193383079571604570144749339935E-0001_r4,  & ! 33
       1.790011511813899504192948153136209872279853641068542156627588395E-0001_r4,  & ! 34
       1.552936556088942974027264975818789317720293465561794469954531758E-0001_r4,  & ! 35
       1.369994576250613898894451713998054823302492355480653627115098493E-0001_r4,  & ! 36
       1.224848042738414175492254544846529190937124368407817995771874189E-0001_r4,  & ! 37
       1.107046377330686263702120864917530505889468618208519881185565845E-0001_r4,  & ! 38
       9.277656780053835438948670532472992481448699505136245821506569383E-0002_r4,  & ! 39
       7.980005432915293348986449770087429955765181389059070298608217360E-0002_r4,  & ! 40
       6.998516620088092772275224944204516970748441002661746054466808797E-0002_r4,  & ! 41
       6.230772403777468414653749202684764733993894726072122845426847610E-0002_r4,  & ! 42
       5.614099274382258585751738722046831156515725665507548351903492024E-0002_r4,  & ! 43
       5.108059475808844370998558301402977200031441462396825283402806716E-0002_r4,  & ! 44
       4.685422101489376261958841339939665904323931915718803315921704839E-0002_r4,  & ! 45
       4.327192186460969266318759898303380115440527301308371350230131664E-0002_r4,  & ! 46
       4.019722865021845930606200422921333952729661347532428877805797534E-0002_r4,  & ! 47
       3.752960638850576574606117818254821602507161134468271480347284945E-0002_r4,  & ! 48
       3.519337782493083756637297489953351970142777219206202563404219860E-0002_r4,  & ! 49
       3.313049999972553669977277294801872035208122227669309138887114781E-0002_r4,  & ! 50
       3.129571781590520988566477909161383437317081613057417169179835202E-0002_r4,  & ! 51
       2.965323064126216352519623531435461785871688420417814149329632135E-0002_r4,  & ! 52
       2.817434874105131931864915453447075835251573225296480740389798765E-0002_r4,  & ! 53
       2.683581315864795664219687801521825439579034377858379753676716914E-0002_r4,  & ! 54
       2.561857000587945266761757947455846538023360563274733504207114582E-0002_r4,  & ! 55
       2.450686208928260590649277351694335889447765285330176577636292659E-0002_r4,  & ! 56
       2.348754606368264051860302591666840906927531002911224978936485192E-0002_r4,  & ! 57
       2.254957243264135894360458395939187726006906701640961131274410338E-0002_r4,  & ! 58
       2.168358485056290661617299168729320269579531998199312854310461437E-0002_r4,  & ! 59
       2.123104471385057441351983960547894758287566770903743534120367476E-0002_r4,  & ! 60
       2.088160799042094067409449019293500894020380993699978983587139087E-0002_r4,  & ! 61
       1.175140054577856629795290707654389501031917423007629905837252269E-0002_r4,  & ! 62
       5.641613782989432903556457006951550718706021244402940434087137078E-0003_r4,  & ! 63
       3.761180312247991929988893646909876170332109011097388096342584477E-0003_r4,  & ! 64
       5.641893014533876541997450280616957271664021150069653916736389707E-0004_r4,  & ! 65
       5.641895807268084115235157250466647220428627387517711960423140320E-0005_r4,  & ! 66
       3.761263881960011101335456515271453406623074429520676259864805054E-0005_r4,  & ! 67
       1.175394965469415232148936972038803256109546890520404832346625762E-0005_r4,  & ! 68
       9.403159724489943709984941567757250737905257587667931986806605262E-0006_r4,  & ! 69
       8.059851193539374705395911191748479743861879040456803369569588902E-0009_r4,  & ! 70
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 71  +∞ => +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 72  +∞ => +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 73  +∞ => +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 74  +∞ => +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 75  +∞ => +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 76  +∞ => +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 77  +∞ => +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4  /) ! 78  +∞ => +0.00
!
!   erfcx: REAL*8
!
!   random variable x for y = erfcx(x), kind=2 = real*8 = REAL64
    real   (kind=r8 ), parameter, dimension(derfcx   )   :: dxerfcxr8 =          (/ & ! 00
        -HUGE(1.000E+0000_r8),     -HUGE(1.000E+0000_r8),    -HUGE(1.000E+0000_r8), & ! 03
        -HUGE(1.000E+0000_r8),         -26.000E+0000_r8,         -10.000E+0000_r8,  & ! 06
             -9.000E+0000_r8,           -8.000E+0000_r8,          -7.000E+0000_r8,  & ! 09
             -6.000E+0000_r8,           -5.000E+0000_r8,          -4.000E+0000_r8,  & ! 12
             -3.000E+0000_r8,           -2.000E+0000_r8,          -1.500E+0000_r8,  & ! 15
             -1.000E+0000_r8,           -0.750E+0000_r8,          -0.500E+0000_r8,  & ! 18
             -0.250E+0000_r8,           -0.030E+0000_r8,          -1.150E-0016_r8,  & ! 21
             -2.220E-0020_r8,            0.000E+0000_r8,           2.220E-0020_r8,  & ! 24
              1.150E-0016_r8,            0.030E+0000_r8,           0.250E+0000_r8,  & ! 27
              0.500E+0000_r8,            0.750E+0000_r8,           1.000E+0000_r8,  & ! 30
              1.500E+0000_r8,            2.000E+0000_r8,           2.500E+0000_r8,  & ! 33
              3.000E+0000_r8,            3.500E+0000_r8,           4.000E+0000_r8,  & ! 36
              4.500E+0000_r8,            5.000E+0000_r8,           6.000E+0000_r8,  & ! 39
              7.000E+0000_r8,            8.000E+0000_r8,           9.000E+0000_r8,  & ! 42
             10.000E+0000_r8,           11.000E+0000_r8,          12.000E+0000_r8,  & ! 45
             13.000E+0000_r8,           14.000E+0000_r8,          15.000E+0000_r8,  & ! 48
             16.000E+0000_r8,           17.000E+0000_r8,          18.000E+0000_r8,  & ! 51
             19.000E+0000_r8,           20.000E+0000_r8,          21.000E+0000_r8,  & ! 54
             22.000E+0000_r8,           23.000E+0000_r8,          24.000E+0000_r8,  & ! 57
             25.000E+0000_r8,           26.000E+0000_r8,          26.555E+0000_r8,  & ! 60
             27.000E+0000_r8,           48.000E+0000_r8,           1.000E+0002_r8,  & ! 63
              1.500E+0002_r8,            1.000E+0003_r8,           1.000E+0004_r8,  & ! 66
              1.500E+0004_r8,            4.800E+0004_r8,           6.000E+0004_r8,  & ! 69
              7.000E+0007_r8,            7.000E+0070_r8,           4.500E+0150_r8,  & ! 72
              2.400E+0300_r8,            2.400E+0306_r8,     +HUGE(1.000E+0000_r8), & ! 75
        +HUGE(1.000E+0000_r8),     +HUGE(1.000E+0000_r8),    +HUGE(1.000E+0000_r8) /) ! 78

!
!   scaled complementary error function, y = erfcx(x), kind=2 = real*8 = REAL64
    real   (kind=r8 ), parameter, dimension(derfcx   )   :: dwerfcxr8 =          (/ & ! 00
                                                               +HUGE(1.0E+0000_r8), & ! 01  -∞ => +∞
                                                               +HUGE(1.0E+0000_r8), & ! 02  -∞ => +∞
                                                               +HUGE(1.0E+0000_r8), & ! 03  -∞ => +∞
                                                               +HUGE(1.0E+0000_r8), & ! 04  -∞ => +∞
       7.657724931490568351526773657706215529330480507562346940485035732E+0293_r8,  & ! 05
       5.376234283632270896825251103160027174722223749134285208656063137E+0043_r8,  & ! 06
       3.012194629170061096705188260335349963175715743878784001794360069E+0035_r8,  & ! 07
       1.247029816162323376581847741778695432346190276474884757933513822E+0028_r8,  & ! 08
       3.814693144990199381050119881853367741842889596589865930650911766E+0021_r8,  & ! 09
       8.622463094230390361450276785175496392091021908830770647053340636E+0015_r8,  & ! 10
       1.440097986746610404105896343058821037439553145762016106106219209E+0011_r8,  & ! 11
       1.777222090401628764846465759211772930179995731346404213150261392E+0007_r8,  & ! 12
       1.620598885399958662546957408405020630903572419029912007078465534E+0004_r8,  & ! 13
       1.089409043899779724123554338248132140422788747719728953862052389E+0002_r8,  & ! 14
       1.865388625626273393874641550130021127851120319405663701880037332E+0001_r8,  & ! 15
       5.008980080762283466309824598214809814694334684235666486188395484E+0000_r8,  & ! 16
       3.003171663627452308696662251659107601538747047728341647273491144E+0000_r8,  & ! 17
       1.952360489182557093276047713441130979890255339297294572744188004E+0000_r8,  & ! 18
       1.358642370104722115210042016948988220013808502272091357355815027E+0000_r8,  & ! 19
       1.034772098273178019832013143072154484345890514100185925367614013E+0000_r8,  & ! 20
       1.000000000000000129763604215983959223058273858978838826575482337E+0000_r8,  & ! 21
       1.000000000000000000025050017509520379140987567649298302811484078E+0000_r8,  & ! 22
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 23
       9.999999999999999999749499824904796208599981123507016971885159216E-0001_r8,  & ! 24
       9.999999999999998702363957840160672269417261410211611734245176626E-0001_r8,  & ! 25
       9.670287119698766650109632717495573341360302147775554874047279156E-0001_r8,  & ! 26
       7.703465477309967439167391723367911261876423850266152456827999940E-0001_r8,  & ! 27
       6.156903441929258748707934226837419367823063912656316056908265890E-0001_r8,  & ! 28
       5.069376502931448057914318215388277759377280127631554128297343480E-0001_r8,  & ! 29
       4.275835761558070044107503444905151808201595031642526637455397707E-0001_r8,  & ! 30
       3.215854164543175023543225877232655690292467412681370014065243116E-0001_r8,  & ! 31
       2.553956763105057438650885809085427633025993052552420654479489936E-0001_r8,  & ! 32
       2.108063640611435806471120484067023193383079571604570144749339935E-0001_r8,  & ! 33
       1.790011511813899504192948153136209872279853641068542156627588395E-0001_r8,  & ! 34
       1.552936556088942974027264975818789317720293465561794469954531758E-0001_r8,  & ! 35
       1.369994576250613898894451713998054823302492355480653627115098493E-0001_r8,  & ! 36
       1.224848042738414175492254544846529190937124368407817995771874189E-0001_r8,  & ! 37
       1.107046377330686263702120864917530505889468618208519881185565845E-0001_r8,  & ! 38
       9.277656780053835438948670532472992481448699505136245821506569383E-0002_r8,  & ! 39
       7.980005432915293348986449770087429955765181389059070298608217360E-0002_r8,  & ! 40
       6.998516620088092772275224944204516970748441002661746054466808797E-0002_r8,  & ! 41
       6.230772403777468414653749202684764733993894726072122845426847610E-0002_r8,  & ! 42
       5.614099274382258585751738722046831156515725665507548351903492024E-0002_r8,  & ! 43
       5.108059475808844370998558301402977200031441462396825283402806716E-0002_r8,  & ! 44
       4.685422101489376261958841339939665904323931915718803315921704839E-0002_r8,  & ! 45
       4.327192186460969266318759898303380115440527301308371350230131664E-0002_r8,  & ! 46
       4.019722865021845930606200422921333952729661347532428877805797534E-0002_r8,  & ! 47
       3.752960638850576574606117818254821602507161134468271480347284945E-0002_r8,  & ! 48
       3.519337782493083756637297489953351970142777219206202563404219860E-0002_r8,  & ! 49
       3.313049999972553669977277294801872035208122227669309138887114781E-0002_r8,  & ! 50
       3.129571781590520988566477909161383437317081613057417169179835202E-0002_r8,  & ! 51
       2.965323064126216352519623531435461785871688420417814149329632135E-0002_r8,  & ! 52
       2.817434874105131931864915453447075835251573225296480740389798765E-0002_r8,  & ! 53
       2.683581315864795664219687801521825439579034377858379753676716914E-0002_r8,  & ! 54
       2.561857000587945266761757947455846538023360563274733504207114582E-0002_r8,  & ! 55
       2.450686208928260590649277351694335889447765285330176577636292659E-0002_r8,  & ! 56
       2.348754606368264051860302591666840906927531002911224978936485192E-0002_r8,  & ! 57
       2.254957243264135894360458395939187726006906701640961131274410338E-0002_r8,  & ! 58
       2.168358485056290661617299168729320269579531998199312854310461437E-0002_r8,  & ! 59
       2.123104471385057441351983960547894758287566770903743534120367476E-0002_r8,  & ! 60
       2.088160799042094067409449019293500894020380993699978983587139087E-0002_r8,  & ! 61
       1.175140054577856629795290707654389501031917423007629905837252269E-0002_r8,  & ! 62
       5.641613782989432903556457006951550718706021244402940434087137078E-0003_r8,  & ! 63
       3.761180312247991929988893646909876170332109011097388096342584477E-0003_r8,  & ! 64
       5.641893014533876541997450280616957271664021150069653916736389707E-0004_r8,  & ! 65
       5.641895807268084115235157250466647220428627387517711960423140320E-0005_r8,  & ! 66
       3.761263881960011101335456515271453406623074429520676259864805054E-0005_r8,  & ! 67
       1.175394965469415232148936972038803256109546890520404832346625762E-0005_r8,  & ! 68
       9.403159724489943709984941567757250737905257587667931986806605262E-0006_r8,  & ! 69
       8.059851193539374705395911191748479743861879040456803369569588902E-0009_r8,  & ! 70
       8.059851193539375527829706450868179797772151847557126526344081738E-0072_r8,  & ! 71
       1.253754630106125082106843225690605746320112509619997459653523826E-0151_r8,  & ! 72  (1/sqrt(pi)) / (4.500E+0150)
       2.350789931448984528950331048169885774350210955537495236850357173E-0301_r8,  & ! 73  (1/sqrt(pi)) / (2.400E+0300)
       2.350789931448984528950331048169885774350210955537495236850357173E-0307_r8,  & ! 74  (1/sqrt(pi)) / (2.400E+0306)
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 75  (1/sqrt(pi)) / (2.550E+0307)
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 76  +∞ => +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 77  +∞ => +0.00
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8  /) ! 78  +∞ => +0.00
!
!   erfcx: REAL*10/16
!
!   random variable x for y = erfcx(x), kind=3 = real*16 = REAL128 (Silverfrost real*10)
    real   (kind=r16), parameter, dimension(derfcx   )   :: dxerfcxr16 =         (/ & ! 00
        -HUGE(1.000E+0000_r16),       -106.000E+0000_r16,        -49.000E+0000_r16, & ! 03
            -27.000E+0000_r16,         -26.000E+0000_r16,        -10.000E+0000_r16, & ! 06
             -9.000E+0000_r16,          -8.000E+0000_r16,         -7.000E+0000_r16, & ! 09
             -6.000E+0000_r16,          -5.000E+0000_r16,         -4.000E+0000_r16, & ! 12
             -3.000E+0000_r16,          -2.000E+0000_r16,         -1.500E+0000_r16, & ! 15
             -1.000E+0000_r16,          -0.750E+0000_r16,         -0.500E+0000_r16, & ! 18
             -0.250E+0000_r16,          -0.030E+0000_r16,         -1.150E-0016_r16, & ! 21
             -2.220E-0020_r16,           0.000E+0000_r16,          2.220E-0020_r16, & ! 24
              1.150E-0016_r16,           0.030E+0000_r16,          0.250E+0000_r16, & ! 27
              0.500E+0000_r16,           0.750E+0000_r16,          1.000E+0000_r16, & ! 30
              1.500E+0000_r16,           2.000E+0000_r16,          2.500E+0000_r16, & ! 33
              3.000E+0000_r16,           3.500E+0000_r16,          4.000E+0000_r16, & ! 36
              4.500E+0000_r16,           5.000E+0000_r16,          6.000E+0000_r16, & ! 39
              7.000E+0000_r16,           8.000E+0000_r16,          9.000E+0000_r16, & ! 42
             10.000E+0000_r16,          11.000E+0000_r16,         12.000E+0000_r16, & ! 45
             13.000E+0000_r16,          14.000E+0000_r16,         15.000E+0000_r16, & ! 48
             16.000E+0000_r16,          17.000E+0000_r16,         18.000E+0000_r16, & ! 51
             19.000E+0000_r16,          20.000E+0000_r16,         21.000E+0000_r16, & ! 54
             22.000E+0000_r16,          23.000E+0000_r16,         24.000E+0000_r16, & ! 57
             25.000E+0000_r16,          26.000E+0000_r16,         26.555E+0000_r16, & ! 60
             27.000E+0000_r16,          48.000E+0000_r16,          1.000E+0002_r16, & ! 63
              1.500E+0002_r16,           1.000E+0003_r16,          1.000E+0004_r16, & ! 66
              1.500E+0004_r16,           4.800E+0004_r16,          6.000E+0004_r16, & ! 69
              7.000E+0007_r16,           7.000E+0070_r16,          4.500E+0150_r16, & ! 72
              2.400E+0300_r16,           2.400E+0306_r16,          2.550E+0307_r16, & ! 75
              5.000E+1045_r16,           2.000E+3300_r16,    +HUGE(1.000E+0000_r16)/) ! 78
!
!   scaled complementary error function, y = erfcx(x), kind=3 = real*16 = REAL128
    real   (kind=r16), parameter, dimension(derfcx   )   :: dwerfcxr16 =         (/ & ! 00
                                                               +HUGE(1.0E+0000_r16),& ! 01  -∞ => +∞
       1.081007384292911774009323562529646498192417673036703432246297354E+4880_r16, & ! 02
       1.101744891230519013281062545490362618657575418039039594609620564E+1043_r16, & ! 03
       7.974570524085193127093722094668700648898179096534145424698741091E+0316_r16, & ! 04
       7.657724931490568351526773657706215529330480507562346940485035732E+0293_r16, & ! 05
       5.376234283632270896825251103160027174722223749134285208656063137E+0043_r16, & ! 06
       3.012194629170061096705188260335349963175715743878784001794360069E+0035_r16, & ! 07
       1.247029816162323376581847741778695432346190276474884757933513822E+0028_r16, & ! 08
       3.814693144990199381050119881853367741842889596589865930650911766E+0021_r16, & ! 09
       8.622463094230390361450276785175496392091021908830770647053340636E+0015_r16, & ! 10
       1.440097986746610404105896343058821037439553145762016106106219209E+0011_r16, & ! 11
       1.777222090401628764846465759211772930179995731346404213150261392E+0007_r16, & ! 12
       1.620598885399958662546957408405020630903572419029912007078465534E+0004_r16, & ! 13
       1.089409043899779724123554338248132140422788747719728953862052389E+0002_r16, & ! 14
       1.865388625626273393874641550130021127851120319405663701880037332E+0001_r16, & ! 15
       5.008980080762283466309824598214809814694334684235666486188395484E+0000_r16, & ! 16
       3.003171663627452308696662251659107601538747047728341647273491144E+0000_r16, & ! 17
       1.952360489182557093276047713441130979890255339297294572744188004E+0000_r16, & ! 18
       1.358642370104722115210042016948988220013808502272091357355815027E+0000_r16, & ! 19
       1.034772098273178019832013143072154484345890514100185925367614013E+0000_r16, & ! 20
       1.000000000000000129763604215983959223058273858978838826575482337E+0000_r16, & ! 21
       1.000000000000000000025050017509520379140987567649298302811484078E+0000_r16, & ! 22
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 23
       9.999999999999999999749499824904796208599981123507016971885159216E-0001_r16, & ! 24
       9.999999999999998702363957840160672269417261410211611734245176626E-0001_r16, & ! 25
       9.670287119698766650109632717495573341360302147775554874047279156E-0001_r16, & ! 26
       7.703465477309967439167391723367911261876423850266152456827999940E-0001_r16, & ! 27
       6.156903441929258748707934226837419367823063912656316056908265890E-0001_r16, & ! 28
       5.069376502931448057914318215388277759377280127631554128297343480E-0001_r16, & ! 29
       4.275835761558070044107503444905151808201595031642526637455397707E-0001_r16, & ! 30
       3.215854164543175023543225877232655690292467412681370014065243116E-0001_r16, & ! 31
       2.553956763105057438650885809085427633025993052552420654479489936E-0001_r16, & ! 32
       2.108063640611435806471120484067023193383079571604570144749339935E-0001_r16, & ! 33
       1.790011511813899504192948153136209872279853641068542156627588395E-0001_r16, & ! 34
       1.552936556088942974027264975818789317720293465561794469954531758E-0001_r16, & ! 35
       1.369994576250613898894451713998054823302492355480653627115098493E-0001_r16, & ! 36
       1.224848042738414175492254544846529190937124368407817995771874189E-0001_r16, & ! 37
       1.107046377330686263702120864917530505889468618208519881185565845E-0001_r16, & ! 38
       9.277656780053835438948670532472992481448699505136245821506569383E-0002_r16, & ! 39
       7.980005432915293348986449770087429955765181389059070298608217360E-0002_r16, & ! 40
       6.998516620088092772275224944204516970748441002661746054466808797E-0002_r16, & ! 41
       6.230772403777468414653749202684764733993894726072122845426847610E-0002_r16, & ! 42
       5.614099274382258585751738722046831156515725665507548351903492024E-0002_r16, & ! 43
       5.108059475808844370998558301402977200031441462396825283402806716E-0002_r16, & ! 44
       4.685422101489376261958841339939665904323931915718803315921704839E-0002_r16, & ! 45
       4.327192186460969266318759898303380115440527301308371350230131664E-0002_r16, & ! 46
       4.019722865021845930606200422921333952729661347532428877805797534E-0002_r16, & ! 47
       3.752960638850576574606117818254821602507161134468271480347284945E-0002_r16, & ! 48
       3.519337782493083756637297489953351970142777219206202563404219860E-0002_r16, & ! 49
       3.313049999972553669977277294801872035208122227669309138887114781E-0002_r16, & ! 50
       3.129571781590520988566477909161383437317081613057417169179835202E-0002_r16, & ! 51
       2.965323064126216352519623531435461785871688420417814149329632135E-0002_r16, & ! 52
       2.817434874105131931864915453447075835251573225296480740389798765E-0002_r16, & ! 53
       2.683581315864795664219687801521825439579034377858379753676716914E-0002_r16, & ! 54
       2.561857000587945266761757947455846538023360563274733504207114582E-0002_r16, & ! 55
       2.450686208928260590649277351694335889447765285330176577636292659E-0002_r16, & ! 56
       2.348754606368264051860302591666840906927531002911224978936485192E-0002_r16, & ! 57
       2.254957243264135894360458395939187726006906701640961131274410338E-0002_r16, & ! 58
       2.168358485056290661617299168729320269579531998199312854310461437E-0002_r16, & ! 59
       2.123104471385057441351983960547894758287566770903743534120367476E-0002_r16, & ! 60
       2.088160799042094067409449019293500894020380993699978983587139087E-0002_r16, & ! 61
       1.175140054577856629795290707654389501031917423007629905837252269E-0002_r16, & ! 62
       5.641613782989432903556457006951550718706021244402940434087137078E-0003_r16, & ! 63
       3.761180312247991929988893646909876170332109011097388096342584477E-0003_r16, & ! 64
       5.641893014533876541997450280616957271664021150069653916736389707E-0004_r16, & ! 65
       5.641895807268084115235157250466647220428627387517711960423140320E-0005_r16, & ! 66
       3.761263881960011101335456515271453406623074429520676259864805054E-0005_r16, & ! 67
       1.175394965469415232148936972038803256109546890520404832346625762E-0005_r16, & ! 68
       9.403159724489943709984941567757250737905257587667931986806605262E-0006_r16, & ! 69
       8.059851193539374705395911191748479743861879040456803369569588902E-0009_r16, & ! 70
       8.059851193539375527829706450868179797772151847557126526344081738E-0072_r16, & ! 71
       1.253754630106125082106843225690605746320112509619997459653523826E-0151_r16, & ! 72  (1/sqrt(pi)) / (4.500E+0150)
       2.350789931448984528950331048169885774350210955537495236850357173E-0301_r16, & ! 73  (1/sqrt(pi)) / (2.400E+0300)
       2.350789931448984528950331048169885774350210955537495236850357173E-0307_r16, & ! 74  (1/sqrt(pi)) / (2.400E+0306)
       2.212508170775514850776782162983421905270786781682348458212100869E-0308_r16, & ! 75  (1/sqrt(pi)) / (2.550E+0307)
       1.128379167095512573896158903121545171688101258657997713688171443E-1046_r16, & ! 76  (1/sqrt(pi)) / (5.000E+1045)
       2.820947917738781434740397257803862929220253146644994284220428608E-3301_r16, & ! 77  (1/sqrt(pi)) / (2.000E+3300)
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r16 /) ! 78  +∞ => +0.00
!
!   erfcx: table
!
!      k   x                 y = erfcx(x), 64 digits exact
!          123.456E+0000     1.234567890123456789012345678901234567890123456789012345678901234E+0000
!
!      1      -HUGE( x )                                                                  +HUGE( x )  all, -∞ => +∞
!      2  -106.000E+0000     1.081007384292911774009323562529646498192417673036703432246297354E+4880  r*10/16
!      3   -49.000E+0000     1.101744891230519013281062545490362618657575418039039594609620564E+1043  r*10/16
!      4   -27.000E+0000     7.974570524085193127093722094668700648898179096534145424698741091E+0316  r*10/16
!      5   -26.000E+0000     7.657724931490568351526773657706215529330480507562346940485035732E+0293  r*10/16, r*8
!      6   -10.000E+0000     5.376234283632270896825251103160027174722223749134285208656063137E+0043  r*10/16, r*8
!      7    -9.000E+0000     3.012194629170061096705188260335349963175715743878784001794360069E+0035  r*10/16, r*8, r*4
!      8    -8.000E+0000     1.247029816162323376581847741778695432346190276474884757933513822E+0028  r*10/16, r*8, r*4
!      9    -7.000E+0000     3.814693144990199381050119881853367741842889596589865930650911766E+0021  ...
!     10    -6.000E+0000     8.622463094230390361450276785175496392091021908830770647053340636E+0015
!     11    -5.000E+0000     1.440097986746610404105896343058821037439553145762016106106219209E+0011
!     12    -4.000E+0000     1.777222090401628764846465759211772930179995731346404213150261392E+0007
!     13    -3.000E+0000     1.620598885399958662546957408405020630903572419029912007078465534E+0004
!     14    -2.000E+0000     1.089409043899779724123554338248132140422788747719728953862052389E+0002
!     15    -1.500E+0000     1.865388625626273393874641550130021127851120319405663701880037332E+0001
!     16    -1.000E+0000     5.008980080762283466309824598214809814694334684235666486188395484E+0000
!     17    -0.750E+0000     3.003171663627452308696662251659107601538747047728341647273491144E+0000
!     18    -0.500E+0000     1.952360489182557093276047713441130979890255339297294572744188004E+0000
!     19    -0.250E+0000     1.358642370104722115210042016948988220013808502272091357355815027E+0000
!     20    -0.030E+0000     1.034772098273178019832013143072154484345890514100185925367614013E+0000
!     21    -1.150E-0016     1.000000000000000129763604215983959223058273858978838826575482337E+0000
!     22    -2.220E-0020     1.000000000000000000025050017509520379140987567649298302811484078E+0000
!     23     0.000E+0000     1.000000000000000000000000000000000000000000000000000000000000000E+0000
!     24     2.220E-0020     9.999999999999999999749499824904796208599981123507016971885159216E-0001
!     25     1.150E-0016     9.999999999999998702363957840160672269417261410211611734245176626E-0001
!     26     0.030E+0000     9.670287119698766650109632717495573341360302147775554874047279156E-0001
!     27     0.250E+0000     7.703465477309967439167391723367911261876423850266152456827999940E-0001
!     28     0.500E+0000     6.156903441929258748707934226837419367823063912656316056908265890E-0001
!     29     0.750E+0000     5.069376502931448057914318215388277759377280127631554128297343480E-0001
!     30     1.000E+0000     4.275835761558070044107503444905151808201595031642526637455397707E-0001
!     31     1.500E+0000     3.215854164543175023543225877232655690292467412681370014065243116E-0001
!     32     2.000E+0000     2.553956763105057438650885809085427633025993052552420654479489936E-0001
!     33     2.500E+0000     2.108063640611435806471120484067023193383079571604570144749339935E-0001
!     34     3.000E+0000     1.790011511813899504192948153136209872279853641068542156627588395E-0001
!     35     3.500E+0000     1.552936556088942974027264975818789317720293465561794469954531758E-0001
!     36     4.000E+0000     1.369994576250613898894451713998054823302492355480653627115098493E-0001
!     37     4.500E+0000     1.224848042738414175492254544846529190937124368407817995771874189E-0001
!     38     5.000E+0000     1.107046377330686263702120864917530505889468618208519881185565845E-0001
!     39     6.000E+0000     9.277656780053835438948670532472992481448699505136245821506569383E-0002
!     40     7.000E+0000     7.980005432915293348986449770087429955765181389059070298608217360E-0002
!     41     8.000E+0000     6.998516620088092772275224944204516970748441002661746054466808797E-0002
!     42     9.000E+0000     6.230772403777468414653749202684764733993894726072122845426847610E-0002
!     43    10.000E+0000     5.614099274382258585751738722046831156515725665507548351903492024E-0002
!     44    11.000E+0000     5.108059475808844370998558301402977200031441462396825283402806716E-0002
!     45    12.000E+0000     4.685422101489376261958841339939665904323931915718803315921704839E-0002
!     46    13.000E+0000     4.327192186460969266318759898303380115440527301308371350230131664E-0002
!     47    14.000E+0000     4.019722865021845930606200422921333952729661347532428877805797534E-0002
!     48    15.000E+0000     3.752960638850576574606117818254821602507161134468271480347284945E-0002
!     49    16.000E+0000     3.519337782493083756637297489953351970142777219206202563404219860E-0002
!     50    17.000E+0000     3.313049999972553669977277294801872035208122227669309138887114781E-0002
!     51    18.000E+0000     3.129571781590520988566477909161383437317081613057417169179835202E-0002
!     52    19.000E+0000     2.965323064126216352519623531435461785871688420417814149329632135E-0002
!     53    20.000E+0000     2.817434874105131931864915453447075835251573225296480740389798765E-0002
!     54    21.000E+0000     2.683581315864795664219687801521825439579034377858379753676716914E-0002
!     55    22.000E+0000     2.561857000587945266761757947455846538023360563274733504207114582E-0002
!     56    23.000E+0000     2.450686208928260590649277351694335889447765285330176577636292659E-0002
!     57    24.000E+0000     2.348754606368264051860302591666840906927531002911224978936485192E-0002
!     58    25.000E+0000     2.254957243264135894360458395939187726006906701640961131274410338E-0002
!     59    26.000E+0000     2.168358485056290661617299168729320269579531998199312854310461437E-0002
!     60    26.555E+0000     2.123104471385057441351983960547894758287566770903743534120367476E-0002
!     61    27.000E+0000     2.088160799042094067409449019293500894020380993699978983587139087E-0002  ...
!     62    48.000E+0000     1.175140054577856629795290707654389501031917423007629905837252269E-0002  r*10/16, r*8, r*4
!     63     1.000E+0002     5.641613782989432903556457006951550718706021244402940434087137078E-0003
!     64     1.500E+0002     3.761180312247991929988893646909876170332109011097388096342584477E-0003
!     65     1.000E+0003     5.641893014533876541997450280616957271664021150069653916736389707E-0004
!     66     1.000E+0004     5.641895807268084115235157250466647220428627387517711960423140320E-0005
!     67     1.500E+0004     3.761263881960011101335456515271453406623074429520676259864805054E-0005
!     68     4.800E+0004     1.175394965469415232148936972038803256109546890520404832346625762E-0005
!     69     6.000E+0004     9.403159724489943709984941567757250737905257587667931986806605262E-0006
!     70     7.000E+0007     8.059851193539374705395911191748479743861879040456803369569588902E-0009  r*10/16, r*8, r*4
!     71     7.000E+0070     8.059851193539375527829706450868179797772151847557126526344081738E-0072  r*10/16, r*8
!     72     4.500E+0150     1.253754630106125082106843225690605746320112509619997459653523826E-0151  r*10/16, r*8       (1/sqrt(pi)) / (4.500E+0150)
!     73     2.400E+0300     2.350789931448984528950331048169885774350210955537495236850357173E-0301  r*10/16, r*8       (1/sqrt(pi)) / (2.400E+0300)
!     74     2.400E+0306     2.350789931448984528950331048169885774350210955537495236850357173E-0307  r*10/16, r*8       (1/sqrt(pi)) / (2.400E+0306)
!     75     2.550E+0307     2.212508170775514850776782162983421905270786781682348458212100869E-0308  r*10/16            (1/sqrt(pi)) / (2.550E+0307)
!     76     5.000E+1045     1.128379167095512573896158903121545171688101258657997713688171443E-1046  r*10/16            (1/sqrt(pi)) / (5.000E+1045)
!     77     2.000E+3300     2.820947917738781434740397257803862929220253146644994284220428608E-3301  r*10/16            (1/sqrt(pi)) / (2.000E+3300)
!     78      +HUGE( x )     0.000000000000000000000000000000000000000000000000000000000000000E+0000  all, +∞ => +0.00
!
!------------------------------------ ERFINV -----------------------------------------------------------------------------------
!
!   Inverse Error Function: x = erfinv(y)
!
!   erfinv: REAL*4
!
!   random variable y für x = erfinv(y), kind=1 = real*4 = REAL32
    real   (kind=r4 ), parameter, dimension(derfinv  )   :: dyerfinvr4 =         (/ & ! 00
                                                             -HUGE(1.000E+0000_r4), & ! 01  not defined => -HUGE(y)
      -1.500000000000000000000000000000000000000000000000000000000000000E+0001_r4,  & ! 02  not defined => -HUGE(y)
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 03  -1.00       => -∞
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 04
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 05
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 06
      -0.999977909503001414558627223870417679620152292912600750342761045E+0000_r4,  & ! 07
      -0.995322265018952734162069256367252928610891797040060076738352326E+0000_r4,  & ! 08
      -0.966105146475310727066976261645947858681410479257636780449967846E+0000_r4,  & ! 09
      -0.842700792949714869341220635082609259296066997966302908459937897E+0000_r4,  & ! 10
      -0.520499877813046537682746653891964528736451575757963700058805725E+0000_r4,  & ! 11
      -0.276326390168236932985068267764815712065353977892311254082471931E+0000_r4,  & ! 12
      -0.033841222341735433302216654256875465225733912120593527612714135E+0000_r4,  & ! 13
      -1.297636042159839459980582738589771227029097259497744623005824543E-0016_r4,  & ! 14
      -2.505001750952037914049472764929830281147173272533108523599220557E-0020_r4,  & ! 15
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 16
       2.505001750952037914049472764929830281147173272533108523599220557E-0020_r4,  & ! 17
       1.297636042159839459980582738589771227029097259497744623005824543E-0016_r4,  & ! 18
       0.033841222341735433302216654256875465225733912120593527612714135E+0000_r4,  & ! 19
       0.276326390168236932985068267764815712065353977892311254082471931E+0000_r4,  & ! 20
       0.520499877813046537682746653891964528736451575757963700058805725E+0000_r4,  & ! 21
       0.842700792949714869341220635082609259296066997966302908459937897E+0000_r4,  & ! 22
       0.966105146475310727066976261645947858681410479257636780449967846E+0000_r4,  & ! 23
       0.995322265018952734162069256367252928610891797040060076738352326E+0000_r4,  & ! 24
       0.999977909503001414558627223870417679620152292912600750342761045E+0000_r4,  & ! 25
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 26
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 27
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 28
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 29  +1.00       => +∞
       1.500000000000000000000000000000000000000000000000000000000000000E+0001_r4,  & ! 30  not defined => +HUGE(y)
                                                             +HUGE(1.000E+0000_r4) /) ! 31  not defined => +HUGE(y)
!
!   inverse error function, x = erfinv(y), kind=1 = real*4 = REAL32
    real   (kind=r4 ), parameter, dimension(derfinv  )   :: dwerfinvr4 =         (/ & ! 00
        -HUGE(1.000E+0000_r4),     -HUGE(1.000E+0000_r4),    -HUGE(1.000E+0000_r4), & ! 03
        -HUGE(1.000E+0000_r4),     -HUGE(1.000E+0000_r4),    -HUGE(1.000E+0000_r4), & ! 06
             -3.000E+0000_r4,           -2.000E+0000_r4,          -1.500E+0000_r4,  & ! 09
             -1.000E+0000_r4,           -0.500E+0000_r4,          -0.250E+0000_r4,  & ! 12
             -0.030E+0000_r4,           -1.150E-0016_r4,          -2.220E-0020_r4,  & ! 15
              0.000E+0000_r4,            2.220E-0020_r4,           1.150E-0016_r4,  & ! 18
              0.030E+0000_r4,            0.250E+0000_r4,           0.500E+0000_r4,  & ! 21
              1.000E+0000_r4,            1.500E+0000_r4,           2.000E+0000_r4,  & ! 24
              3.000E+0000_r4,      +HUGE(1.000E+0000_r4),    +HUGE(1.000E+0000_r4), & ! 27
        +HUGE(1.000E+0000_r4),     +HUGE(1.000E+0000_r4),    +HUGE(1.000E+0000_r4), & ! 30
        +HUGE(1.000E+0000_r4)                                                      /) ! 31
!
!   erfinv: REAL*8
!
!   random variable y für x = erfinv(y), kind=2 = real*8 = REAL64
    real   (kind=r8 ), parameter, dimension(derfinv  )   :: dyerfinvr8 =         (/ & ! 00
                                                             -HUGE(1.000E+0000_r8), & ! 01  not defined => -HUGE(y)
      -1.500000000000000000000000000000000000000000000000000000000000000E+0001_r8,  & ! 02  not defined => -HUGE(y)
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 03  -1.00       => -∞
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 04
      -0.999999999998462540205571965149811656514616621109881949685276620E+0000_r8,  & ! 05
      -0.999999984582742099719981147840326513115951427854746410808831657E+0000_r8,  & ! 06
      -0.999977909503001414558627223870417679620152292912600750342761045E+0000_r8,  & ! 07
      -0.995322265018952734162069256367252928610891797040060076738352326E+0000_r8,  & ! 08
      -0.966105146475310727066976261645947858681410479257636780449967846E+0000_r8,  & ! 09
      -0.842700792949714869341220635082609259296066997966302908459937897E+0000_r8,  & ! 10
      -0.520499877813046537682746653891964528736451575757963700058805725E+0000_r8,  & ! 11
      -0.276326390168236932985068267764815712065353977892311254082471931E+0000_r8,  & ! 12
      -0.033841222341735433302216654256875465225733912120593527612714135E+0000_r8,  & ! 13
      -1.297636042159839459980582738589771227029097259497744623005824543E-0016_r8,  & ! 14
      -2.505001750952037914049472764929830281147173272533108523599220557E-0020_r8,  & ! 15
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 16
       2.505001750952037914049472764929830281147173272533108523599220557E-0020_r8,  & ! 17
       1.297636042159839459980582738589771227029097259497744623005824543E-0016_r8,  & ! 18
       0.033841222341735433302216654256875465225733912120593527612714135E+0000_r8,  & ! 19
       0.276326390168236932985068267764815712065353977892311254082471931E+0000_r8,  & ! 20
       0.520499877813046537682746653891964528736451575757963700058805725E+0000_r8,  & ! 21
       0.842700792949714869341220635082609259296066997966302908459937897E+0000_r8,  & ! 22
       0.966105146475310727066976261645947858681410479257636780449967846E+0000_r8,  & ! 23
       0.995322265018952734162069256367252928610891797040060076738352326E+0000_r8,  & ! 24
       0.999977909503001414558627223870417679620152292912600750342761045E+0000_r8,  & ! 25
       0.999999984582742099719981147840326513115951427854746410808831657E+0000_r8,  & ! 26
       0.999999999998462540205571965149811656514616621109881949685276620E+0000_r8,  & ! 27
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 28
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 29  +1.00       => +∞
       1.500000000000000000000000000000000000000000000000000000000000000E+0001_r8,  & ! 30  not defined => +HUGE(y)
                                                             +HUGE(1.000E+0000_r8) /) ! 31  not defined => +HUGE(y)
!
!   inverse error function, x = erfinv(y), kind=2 = real*8 = REAL64
    real   (kind=r8 ), parameter, dimension(derfinv  )   :: dwerfinvr8 =         (/ & ! 00
        -HUGE(1.000E+0000_r8),     -HUGE(1.000E+0000_r8),    -HUGE(1.000E+0000_r8), & ! 03
        -HUGE(1.000E+0000_r8),          -5.000E+0000_r8,          -4.000E+0000_r8,  & ! 06
             -3.000E+0000_r8,           -2.000E+0000_r8,          -1.500E+0000_r8,  & ! 09
             -1.000E+0000_r8,           -0.500E+0000_r8,          -0.250E+0000_r8,  & ! 12
             -0.030E+0000_r8,           -1.150E-0016_r8,          -2.220E-0020_r8,  & ! 15
              0.000E+0000_r8,            2.220E-0020_r8,           1.150E-0016_r8,  & ! 18
              0.030E+0000_r8,            0.250E+0000_r8,           0.500E+0000_r8,  & ! 21
              1.000E+0000_r8,            1.500E+0000_r8,           2.000E+0000_r8,  & ! 24
              3.000E+0000_r8,            4.000E+0000_r8,           5.000E+0000_r8,  & ! 27
        +HUGE(1.000E+0000_r8),     +HUGE(1.000E+0000_r8),    +HUGE(1.000E+0000_r8), & ! 30
        +HUGE(1.000E+0000_r8)                                                      /) ! 31
!
!   erfinv: REAL*10/16
!
!   random variable y für x = erfinv(y), kind=3 = real*16 = REAL128 (Silverfrost real*10)
    real   (kind=r16), parameter, dimension(derfinv  )   :: dyerfinvr16 =        (/ & ! 00
                                                             -HUGE(1.000E+0000_r16),& ! 01  not defined => -HUGE(y)
      -1.500000000000000000000000000000000000000000000000000000000000000E+0001_r16, & ! 02  not defined => -HUGE(y)
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 03  -1.00       => -∞
      -0.999999999999999978480263287501086883406649600812615369522485938E+0000_r16, & ! 04
      -0.999999999998462540205571965149811656514616621109881949685276620E+0000_r16, & ! 05
      -0.999999984582742099719981147840326513115951427854746410808831657E+0000_r16, & ! 06
      -0.999977909503001414558627223870417679620152292912600750342761045E+0000_r16, & ! 07
      -0.995322265018952734162069256367252928610891797040060076738352326E+0000_r16, & ! 08
      -0.966105146475310727066976261645947858681410479257636780449967846E+0000_r16, & ! 09
      -0.842700792949714869341220635082609259296066997966302908459937897E+0000_r16, & ! 10
      -0.520499877813046537682746653891964528736451575757963700058805725E+0000_r16, & ! 11
      -0.276326390168236932985068267764815712065353977892311254082471931E+0000_r16, & ! 12
      -0.033841222341735433302216654256875465225733912120593527612714135E+0000_r16, & ! 13
      -1.297636042159839459980582738589771227029097259497744623005824543E-0016_r16, & ! 14
      -2.505001750952037914049472764929830281147173272533108523599220557E-0020_r16, & ! 15
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 16
       2.505001750952037914049472764929830281147173272533108523599220557E-0020_r16, & ! 17
       1.297636042159839459980582738589771227029097259497744623005824543E-0016_r16, & ! 18
       0.033841222341735433302216654256875465225733912120593527612714135E+0000_r16, & ! 19
       0.276326390168236932985068267764815712065353977892311254082471931E+0000_r16, & ! 20
       0.520499877813046537682746653891964528736451575757963700058805725E+0000_r16, & ! 21
       0.842700792949714869341220635082609259296066997966302908459937897E+0000_r16, & ! 22
       0.966105146475310727066976261645947858681410479257636780449967846E+0000_r16, & ! 23
       0.995322265018952734162069256367252928610891797040060076738352326E+0000_r16, & ! 24
       0.999977909503001414558627223870417679620152292912600750342761045E+0000_r16, & ! 25
       0.999999984582742099719981147840326513115951427854746410808831657E+0000_r16, & ! 26
       0.999999999998462540205571965149811656514616621109881949685276620E+0000_r16, & ! 27
       0.999999999999999978480263287501086883406649600812615369522485938E+0000_r16, & ! 28
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 29  +1.00       => +∞
       1.500000000000000000000000000000000000000000000000000000000000000E+0001_r16, & ! 30  not defined => +HUGE(y)
                                                             +HUGE(1.000E+0000_r16)/) ! 31  not defined => +HUGE(y)
!
!   inverse error function, x = erfinv(y), kind=3 = real*16 = REAL128 (Silverfrost real*10)
    real   (kind=r16), parameter, dimension(derfinv  )   :: dwerfinvr16 =        (/ & ! 00
        -HUGE(1.000E+0000_r16),    -HUGE(1.000E+0000_r16),   -HUGE(1.000E+0000_r16),& ! 03
             -6.000E+0000_r16,          -5.000E+0000_r16,         -4.000E+0000_r16, & ! 06
             -3.000E+0000_r16,          -2.000E+0000_r16,         -1.500E+0000_r16, & ! 09
             -1.000E+0000_r16,          -0.500E+0000_r16,         -0.250E+0000_r16, & ! 12
             -0.030E+0000_r16,          -1.150E-0016_r16,         -2.220E-0020_r16, & ! 15
              0.000E+0000_r16,           2.220E-0020_r16,          1.150E-0016_r16, & ! 18
              0.030E+0000_r16,           0.250E+0000_r16,          0.500E+0000_r16, & ! 21
              1.000E+0000_r16,           1.500E+0000_r16,          2.000E+0000_r16, & ! 24
              3.000E+0000_r16,           4.000E+0000_r16,          5.000E+0000_r16, & ! 27
              6.000E+0000_r16,     +HUGE(1.000E+0000_r16),   +HUGE(1.000E+0000_r16),& ! 30
        +HUGE(1.000E+0000_r16)                                                     /) ! 31
!
!   erfinv: table
!
!      k    y = erf(x), 64 digits exact                                                x = erfinv(y)
!           1.234567890123456789012345678901234567890123456789012345678901234E+0000    123.456E+0000
!
!      1                                                                 -HUGE( y )       -HUGE( y )  not defined => -HUGE(y)
!      2   -1.500000000000000000000000000000000000000000000000000000000000000E+0001       -HUGE( y )  not defined => -HUGE(y)
!      3   -1.000000000000000000000000000000000000000000000000000000000000000E+0000       -HUGE( y )  -1.00       => -∞
!      x   -0.999999999999999999999999999999999999999999999999999998559133862E+0000    -11.000E+0000  only here
!      x   -0.999999999999999999999999999999999999999999997911512416237455242E+0000    -10.000E+0000  only here
!      x   -0.999999999999999999999999999999999999586296825348618976194609653E+0000     -9.000E+0000  only here
!      x   -0.999999999999999999999999999988775702827017072920032111556829720E+0000     -8.000E+0000  only here
!      x   -0.999999999999999999999958161743922205856013859897761000677499704E+0000     -7.000E+0000  only here
!      4   -0.999999999999999978480263287501086883406649600812615369522485938E+0000     -6.000E+0000  r*10/16
!      5   -0.999999999998462540205571965149811656514616621109881949685276620E+0000     -5.000E+0000  r*8
!      6   -0.999999984582742099719981147840326513115951427854746410808831657E+0000     -4.000E+0000
!      7   -0.999977909503001414558627223870417679620152292912600750342761045E+0000     -3.000E+0000  r*4
!      8   -0.995322265018952734162069256367252928610891797040060076738352326E+0000     -2.000E+0000
!      9   -0.966105146475310727066976261645947858681410479257636780449967846E+0000     -1.500E+0000
!     10   -0.842700792949714869341220635082609259296066997966302908459937897E+0000     -1.000E+0000
!     11   -0.520499877813046537682746653891964528736451575757963700058805725E+0000     -0.500E+0000
!     12   -0.276326390168236932985068267764815712065353977892311254082471931E+0000     -0.250E+0000
!     13   -0.033841222341735433302216654256875465225733912120593527612714135E+0000     -0.030E+0000
!     14   -1.297636042159839459980582738589771227029097259497744623005824543E-0016     -1.150E-0016
!     15   -2.505001750952037914049472764929830281147173272533108523599220557E-0020     -2.220E-0020
!      x   -1.692568750643268860844238354682317757532151887986996570532257165E-0301     -1.500E-0301  r10/16, r8, r4=0, only table
!     16    0.000000000000000000000000000000000000000000000000000000000000000E+0000      0.000E+0000
!      x    1.692568750643268860844238354682317757532151887986996570532257165E-0301      1.500E-0301  r10/16, r8, r4=0, only table
!     17    2.505001750952037914049472764929830281147173272533108523599220557E-0020      2.220E-0020
!     18    1.297636042159839459980582738589771227029097259497744623005824543E-0016      1.150E-0016
!     19    0.033841222341735433302216654256875465225733912120593527612714135E+0000      0.030E+0000
!     20    0.276326390168236932985068267764815712065353977892311254082471931E+0000      0.250E+0000
!     21    0.520499877813046537682746653891964528736451575757963700058805725E+0000      0.500E+0000
!     22    0.842700792949714869341220635082609259296066997966302908459937897E+0000      1.000E+0000
!     23    0.966105146475310727066976261645947858681410479257636780449967846E+0000      1.500E+0000
!     24    0.995322265018952734162069256367252928610891797040060076738352326E+0000      2.000E+0000
!     25    0.999977909503001414558627223870417679620152292912600750342761045E+0000      3.000E+0000
!     26    0.999999984582742099719981147840326513115951427854746410808831657E+0000      4.000E+0000
!     27    0.999999999998462540205571965149811656514616621109881949685276620E+0000      5.000E+0000  r*8
!     28    0.999999999999999978480263287501086883406649600812615369522485938E+0000      6.000E+0000  r*10/16
!      x    0.999999999999999999999958161743922205856013859897761000677499704E+0000      7.000E+0000  only here
!      x    0.999999999999999999999999999988775702827017072920032111556829720E+0000      8.000E+0000  only here
!      x    0.999999999999999999999999999999999999586296825348618976194609653E+0000      9.000E+0000  only here
!      x    0.999999999999999999999999999999999999999999997911512416237455242E+0000     10.000E+0000  only here
!      x    0.999999999999999999999999999999999999999999999999999998559133862E+0000     11.000E+0000  only here
!     29    1.000000000000000000000000000000000000000000000000000000000000000E+0000       +HUGE( y )  +1.00       => +∞
!     30    1.500000000000000000000000000000000000000000000000000000000000000E+0001       +HUGE( y )  not defined => +HUGE(y)
!     31                                                                 +HUGE( y )       +HUGE( y )  not defined => +HUGE(y)
!
!------------------------------------ ERFCINV ----------------------------------------------------------------------------------
!
!   Inverse Complementary Error Function: x = erfcinv(y)
!
!   erfcinv: REAL*4
!
!   random variable y für x = erfcinv(y), kind=1 = real*4 = REAL32
    real   (kind=r4 ), parameter, dimension(derfcinv )   :: dyerfcinvr4 =        (/ & ! 00
                                                             +HUGE(1.000E+0000_r4), & ! 01  not defined => -HUGE(y)
       1.500000000000000000000000000000000000000000000000000000000000000E+0001_r4,  & ! 02  not defined => -HUGE(y)
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 03  +2.00       => -∞
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 04
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 05 
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 06  r8
       1.999977909503001414558627223870417679620152292912600750342761045E+0000_r4,  & ! 07  r4
       1.995322265018952734162069256367252928610891797040060076738352326E+0000_r4,  & ! 08
       1.966105146475310727066976261645947858681410479257636780449967846E+0000_r4,  & ! 09
       1.842700792949714869341220635082609259296066997966302908459937897E+0000_r4,  & ! 10
       1.711155633653515131598937834591410777374205954096537232278133397E+0000_r4,  & ! 11
       1.520499877813046537682746653891964528736451575757963700058805725E+0000_r4,  & ! 12
       1.276326390168236932985068267764815712065353977892311254082471931E+0000_r4,  & ! 13
       1.033841222341735433302216654256875465225733912120593527612714135E+0000_r4,  & ! 14
       1.000000000000000129763604215983945998058273858977122702909725949E+0000_r4,  & ! 15
       1.000000000000000000025050017509520379140494727649298302811471732E+0000_r4,  & ! 16
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 17  +1.00       => 0.00
       9.999999999999999999749499824904796208595052723507016971885282672E-0001_r4,  & ! 18
       9.999999999999998702363957840160540019417261410228772970902740502E-0001_r4,  & ! 19
       9.661587776582645666977833457431245347742660878794064723872858648E-0001_r4,  & ! 20
       7.236736098317630670149317322351842879346460221076887459175280687E-0001_r4,  & ! 21
       4.795001221869534623172533461080354712635484242420362999411942743E-0001_r4,  & ! 22
       2.888443663464848684010621654085892226257940459034627677218666028E-0001_r4,  & ! 23
       1.572992070502851306587793649173907407039330020336970915400621021E-0001_r4,  & ! 24
       3.389485352468927293302373835405214131858952074236321955003215355E-0002_r4,  & ! 25
       4.677734981047265837930743632747071389108202959939923261647673799E-0003_r4,  & ! 26
       4.069520174449589395642157399749127203486774037134201391207787690E-0004_r4,  & ! 27
       2.209049699858544137277612958232037984770708739924965723895484294E-0005_r4,  & ! 28
       7.430983723414127455236837560956357206600921727974625911095699435E-0007_r4,  & ! 29
       1.541725790028001885215967348688404857214525358919116834290499421E-0008_r4,  & ! 30
       1.966160441542887476279160367664332660577670599611646975681185960E-0010_r4,  & ! 31
       1.537459794428034850188343485383378890118050314723379930687914055E-0012_r4,  & ! 32
       2.151973671249891311659335039918738463047751406168854210052789205E-0017_r4,  & ! 33
       4.183825607779414398614010223899932250029617413812460152041634353E-0023_r4,  & ! 34
       1.122429717298292707996788844317027909343192916447896338591267524E-0029_r4,  & ! 35
       4.137031746513810238053903467362524595710191985947962459805909901E-0037_r4,  & ! 36  r4
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 37  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 38  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 39  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 40  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 41  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 42  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 43  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 44  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 45  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 46  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 47  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 48  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 49  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 50  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 51  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 52  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 53  r8, r4=0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 54  r8 = 0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 55  r8 = 0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 56  r8 = 0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 57  r8 = 0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 58  +HUGE( y )   +0.00       => +∞
      -1.000000000000000000000000000000000000000000000000000000000000000E+0001_r4  /) ! 59  +HUGE( y )   not defined => +HUGE(y)
!
!   inverse complementary error function, x = erfcinv(y), kind=1 = real*4 = REAL32
    real   (kind=r4 ), parameter, dimension(derfcinv )   :: dwerfcinvr4 =        (/ & ! 00
        -HUGE(1.000E+0000_r4),      -HUGE(1.000E+0000_r4),   -HUGE(1.000E+0000_r4), & ! 03
        -HUGE(1.000E+0000_r4),      -HUGE(1.000E+0000_r4),   -HUGE(1.000E+0000_r4), & ! 06
             -3.000E+0000_r4,            -2.000E+0000_r4,         -1.500E+0000_r4,  & ! 09
             -1.000E+0000_r4,            -0.750E+0000_r4,         -0.500E+0000_r4,  & ! 12
             -0.250E+0000_r4,            -0.030E+0000_r4,         -1.150E-0016_r4,  & ! 15
             -2.220E-0020_r4,             0.000E+0000_r4,          2.220E-0020_r4,  & ! 18
              1.150E-0016_r4,             0.030E+0000_r4,          0.250E+0000_r4,  & ! 21
              0.500E+0000_r4,             0.750E+0000_r4,          1.000E+0000_r4,  & ! 24
              1.500E+0000_r4,             2.000E+0000_r4,          2.500E+0000_r4,  & ! 27
              3.000E+0000_r4,             3.500E+0000_r4,          4.000E+0000_r4,  & ! 30
              4.500E+0000_r4,             5.000E+0000_r4,          6.000E+0000_r4,  & ! 33
              7.000E+0000_r4,             8.000E+0000_r4,          9.000E+0000_r4,  & ! 36
        +HUGE(1.000E+0000_r4),      +HUGE(1.000E+0000_r4),   +HUGE(1.000E+0000_r4), & ! 39
        +HUGE(1.000E+0000_r4),      +HUGE(1.000E+0000_r4),   +HUGE(1.000E+0000_r4), & ! 42
        +HUGE(1.000E+0000_r4),      +HUGE(1.000E+0000_r4),   +HUGE(1.000E+0000_r4), & ! 45
        +HUGE(1.000E+0000_r4),      +HUGE(1.000E+0000_r4),   +HUGE(1.000E+0000_r4), & ! 48
        +HUGE(1.000E+0000_r4),      +HUGE(1.000E+0000_r4),   +HUGE(1.000E+0000_r4), & ! 51
        +HUGE(1.000E+0000_r4),      +HUGE(1.000E+0000_r4),   +HUGE(1.000E+0000_r4), & ! 54
        +HUGE(1.000E+0000_r4),      +HUGE(1.000E+0000_r4),   +HUGE(1.000E+0000_r4), & ! 57
        +HUGE(1.000E+0000_r4),      +HUGE(1.000E+0000_r4)                          /) ! 59
!
!   erfcinv: REAL*8
!
!   random variable y für x = erfcinv(y), kind=2 = real*8 = REAL64
    real   (kind=r8 ), parameter, dimension(derfcinv )   :: dyerfcinvr8 =        (/ & ! 00
                                                             +HUGE(1.000E+0000_r8), & ! 01  not defined => -HUGE(y)
       1.500000000000000000000000000000000000000000000000000000000000000E+0001_r8,  & ! 02  not defined => -HUGE(y)
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 03  +2.00       => -∞
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 04
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 05 
       1.999999984582742099719981147840326513115951427854746410808831657E+0000_r8,  & ! 06  r8
       1.999977909503001414558627223870417679620152292912600750342761045E+0000_r8,  & ! 07  r4
       1.995322265018952734162069256367252928610891797040060076738352326E+0000_r8,  & ! 08
       1.966105146475310727066976261645947858681410479257636780449967846E+0000_r8,  & ! 09
       1.842700792949714869341220635082609259296066997966302908459937897E+0000_r8,  & ! 10
       1.711155633653515131598937834591410777374205954096537232278133397E+0000_r8,  & ! 11
       1.520499877813046537682746653891964528736451575757963700058805725E+0000_r8,  & ! 12
       1.276326390168236932985068267764815712065353977892311254082471931E+0000_r8,  & ! 13
       1.033841222341735433302216654256875465225733912120593527612714135E+0000_r8,  & ! 14
       1.000000000000000129763604215983945998058273858977122702909725949E+0000_r8,  & ! 15
       1.000000000000000000025050017509520379140494727649298302811471732E+0000_r8,  & ! 16
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 17  +1.00       => 0.00
       9.999999999999999999749499824904796208595052723507016971885282672E-0001_r8,  & ! 18
       9.999999999999998702363957840160540019417261410228772970902740502E-0001_r8,  & ! 19
       9.661587776582645666977833457431245347742660878794064723872858648E-0001_r8,  & ! 20
       7.236736098317630670149317322351842879346460221076887459175280687E-0001_r8,  & ! 21
       4.795001221869534623172533461080354712635484242420362999411942743E-0001_r8,  & ! 22
       2.888443663464848684010621654085892226257940459034627677218666028E-0001_r8,  & ! 23
       1.572992070502851306587793649173907407039330020336970915400621021E-0001_r8,  & ! 24
       3.389485352468927293302373835405214131858952074236321955003215355E-0002_r8,  & ! 25
       4.677734981047265837930743632747071389108202959939923261647673799E-0003_r8,  & ! 26
       4.069520174449589395642157399749127203486774037134201391207787690E-0004_r8,  & ! 27
       2.209049699858544137277612958232037984770708739924965723895484294E-0005_r8,  & ! 28
       7.430983723414127455236837560956357206600921727974625911095699435E-0007_r8,  & ! 29
       1.541725790028001885215967348688404857214525358919116834290499421E-0008_r8,  & ! 30
       1.966160441542887476279160367664332660577670599611646975681185960E-0010_r8,  & ! 31
       1.537459794428034850188343485383378890118050314723379930687914055E-0012_r8,  & ! 32
       2.151973671249891311659335039918738463047751406168854210052789205E-0017_r8,  & ! 33
       4.183825607779414398614010223899932250029617413812460152041634353E-0023_r8,  & ! 34
       1.122429717298292707996788844317027909343192916447896338591267524E-0029_r8,  & ! 35
       4.137031746513810238053903467362524595710191985947962459805909901E-0037_r8,  & ! 36  r4
       2.088487583762544757000786294957788611560818119321163727012213713E-0045_r8,  & ! 37  r4 = 0
       1.440866137943694680339809702856082753964395230973580405075797246E-0054_r8,  & ! 38
       1.356261169205904212780306156590417572666782233288104929912379537E-0064_r8,  & ! 39
       1.739557315466724521804198548243026985088574583449008314936058740E-0075_r8,  & ! 40
       3.037229847750311665115172806783328447911896669343422595150070601E-0087_r8,  & ! 41
       7.212994172451206666565066558692927109934090929825383240467341996E-0100_r8,  & ! 42
       2.328485751571530693364872854573442597534396948094948021516495095E-0113_r8,  & ! 43
       1.021228015094260881145599235077652994401543730587571172553506555E-0127_r8,  & ! 44
       6.082369231816399307668466715702274949587610076867576851592785891E-0143_r8,  & ! 45
       4.917722839256475446413297625239608170930672249989259036315216983E-0159_r8,  & ! 46
       5.395865611607900928934999167905345604088272670923605283470103784E-0176_r8,  & ! 47
       8.032453871022455669021356947138268888967875692513378852786023784E-0194_r8,  & ! 48
       1.621905860933472513052034647026123265177675596121851615118740214E-0212_r8,  & ! 49
       4.441265948088057244074884428946738565068812405566605852511837779E-0232_r8,  & ! 50
       1.648982583151933514218512437543746903943061790303826174801039526E-0252_r8,  & ! 51
       8.300172571196522752044012769513722768714223191886699596054448582E-0274_r8,  & ! 52
       5.663192408856142846475727896926092580328777560925807368646253847E-0296_r8,  & ! 53  r4 = 0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 54  r8 = 0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 55  r8 = 0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 56  r8 = 0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 57  r8 = 0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 58  +HUGE( y )   +0.00       => +∞
      -1.000000000000000000000000000000000000000000000000000000000000000E+0001_r8  /) ! 59  +HUGE( y )   not defined => +HUGE(y)
!
!   inverse complementary error function, x = erfcinv(y), kind=2 = real*8 = REAL64
    real   (kind=r8 ), parameter, dimension(derfcinv )   :: dwerfcinvr8 =        (/ & ! 00
        -HUGE(1.000E+0000_r8),      -HUGE(1.000E+0000_r8),   -HUGE(1.000E+0000_r8), & ! 03
        -HUGE(1.000E+0000_r8),      -HUGE(1.000E+0000_r8),        -4.000E+0000_r8,  & ! 06
             -3.000E+0000_r8,            -2.000E+0000_r8,         -1.500E+0000_r8,  & ! 09
             -1.000E+0000_r8,            -0.750E+0000_r8,         -0.500E+0000_r8,  & ! 12
             -0.250E+0000_r8,            -0.030E+0000_r8,         -1.150E-0016_r8,  & ! 15
             -2.220E-0020_r8,             0.000E+0000_r8,          2.220E-0020_r8,  & ! 18
              1.150E-0016_r8,             0.030E+0000_r8,          0.250E+0000_r8,  & ! 21
              0.500E+0000_r8,             0.750E+0000_r8,          1.000E+0000_r8,  & ! 24
              1.500E+0000_r8,             2.000E+0000_r8,          2.500E+0000_r8,  & ! 27
              3.000E+0000_r8,             3.500E+0000_r8,          4.000E+0000_r8,  & ! 30
              4.500E+0000_r8,             5.000E+0000_r8,          6.000E+0000_r8,  & ! 33
              7.000E+0000_r8,             8.000E+0000_r8,          9.000E+0000_r8,  & ! 36
             10.000E+0000_r8,            11.000E+0000_r8,         12.000E+0000_r8,  & ! 39
             13.000E+0000_r8,            14.000E+0000_r8,         15.000E+0000_r8,  & ! 42
             16.000E+0000_r8,            17.000E+0000_r8,         18.000E+0000_r8,  & ! 45
             19.000E+0000_r8,            20.000E+0000_r8,         21.000E+0000_r8,  & ! 48
             22.000E+0000_r8,            23.000E+0000_r8,         24.000E+0000_r8,  & ! 51
             25.000E+0000_r8,            26.000E+0000_r8,    +HUGE(1.000E+0000_r8), & ! 54
        +HUGE(1.000E+0000_r8),      +HUGE(1.000E+0000_r8),   +HUGE(1.000E+0000_r8), & ! 57
        +HUGE(1.000E+0000_r8),      +HUGE(1.000E+0000_r8)                          /) ! 59
!
!   erfcinv: REAL*10/16
!
!   random variable y für x = erfcinv(y), kind=3 = real*16 = REAL128 (Silverfrost real*10)
    real   (kind=r16), parameter, dimension(derfcinv )   :: dyerfcinvr16 =       (/ & ! 00
                                                             +HUGE(1.000E+0000_r16),& ! 01  not defined => -HUGE(y)
       1.500000000000000000000000000000000000000000000000000000000000000E+0001_r16, & ! 02  not defined => -HUGE(y)
       2.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 03  +2.00       => -∞
       1.999999999999999978480263287501086883406649600812615369522485938E+0000_r16, & ! 04
       1.999999999998462540205571965149811656514616621109881949685276620E+0000_r16, & ! 05 
       1.999999984582742099719981147840326513115951427854746410808831657E+0000_r16, & ! 06  r4 = 0
       1.999977909503001414558627223870417679620152292912600750342761045E+0000_r16, & ! 07  r4
       1.995322265018952734162069256367252928610891797040060076738352326E+0000_r16, & ! 08
       1.966105146475310727066976261645947858681410479257636780449967846E+0000_r16, & ! 09
       1.842700792949714869341220635082609259296066997966302908459937897E+0000_r16, & ! 10
       1.711155633653515131598937834591410777374205954096537232278133397E+0000_r16, & ! 11
       1.520499877813046537682746653891964528736451575757963700058805725E+0000_r16, & ! 12
       1.276326390168236932985068267764815712065353977892311254082471931E+0000_r16, & ! 13
       1.033841222341735433302216654256875465225733912120593527612714135E+0000_r16, & ! 14
       1.000000000000000129763604215983945998058273858977122702909725949E+0000_r16, & ! 15
       1.000000000000000000025050017509520379140494727649298302811471732E+0000_r16, & ! 16
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 17  +1.00       => 0.00
       9.999999999999999999749499824904796208595052723507016971885282672E-0001_r16, & ! 18
       9.999999999999998702363957840160540019417261410228772970902740502E-0001_r16, & ! 19
       9.661587776582645666977833457431245347742660878794064723872858648E-0001_r16, & ! 20
       7.236736098317630670149317322351842879346460221076887459175280687E-0001_r16, & ! 21
       4.795001221869534623172533461080354712635484242420362999411942743E-0001_r16, & ! 22
       2.888443663464848684010621654085892226257940459034627677218666028E-0001_r16, & ! 23
       1.572992070502851306587793649173907407039330020336970915400621021E-0001_r16, & ! 24
       3.389485352468927293302373835405214131858952074236321955003215355E-0002_r16, & ! 25
       4.677734981047265837930743632747071389108202959939923261647673799E-0003_r16, & ! 26
       4.069520174449589395642157399749127203486774037134201391207787690E-0004_r16, & ! 27
       2.209049699858544137277612958232037984770708739924965723895484294E-0005_r16, & ! 28
       7.430983723414127455236837560956357206600921727974625911095699435E-0007_r16, & ! 29
       1.541725790028001885215967348688404857214525358919116834290499421E-0008_r16, & ! 30
       1.966160441542887476279160367664332660577670599611646975681185960E-0010_r16, & ! 31
       1.537459794428034850188343485383378890118050314723379930687914055E-0012_r16, & ! 32
       2.151973671249891311659335039918738463047751406168854210052789205E-0017_r16, & ! 33
       4.183825607779414398614010223899932250029617413812460152041634353E-0023_r16, & ! 34
       1.122429717298292707996788844317027909343192916447896338591267524E-0029_r16, & ! 35
       4.137031746513810238053903467362524595710191985947962459805909901E-0037_r16, & ! 36  r4
       2.088487583762544757000786294957788611560818119321163727012213713E-0045_r16, & ! 37  r4 = 0
       1.440866137943694680339809702856082753964395230973580405075797246E-0054_r16, & ! 38
       1.356261169205904212780306156590417572666782233288104929912379537E-0064_r16, & ! 39
       1.739557315466724521804198548243026985088574583449008314936058740E-0075_r16, & ! 40
       3.037229847750311665115172806783328447911896669343422595150070601E-0087_r16, & ! 41
       7.212994172451206666565066558692927109934090929825383240467341996E-0100_r16, & ! 42
       2.328485751571530693364872854573442597534396948094948021516495095E-0113_r16, & ! 43
       1.021228015094260881145599235077652994401543730587571172553506555E-0127_r16, & ! 44
       6.082369231816399307668466715702274949587610076867576851592785891E-0143_r16, & ! 45
       4.917722839256475446413297625239608170930672249989259036315216983E-0159_r16, & ! 46
       5.395865611607900928934999167905345604088272670923605283470103784E-0176_r16, & ! 47
       8.032453871022455669021356947138268888967875692513378852786023784E-0194_r16, & ! 48
       1.621905860933472513052034647026123265177675596121851615118740214E-0212_r16, & ! 49
       4.441265948088057244074884428946738565068812405566605852511837779E-0232_r16, & ! 50
       1.648982583151933514218512437543746903943061790303826174801039526E-0252_r16, & ! 51
       8.300172571196522752044012769513722768714223191886699596054448582E-0274_r16, & ! 52
       5.663192408856142846475727896926092580328777560925807368646253847E-0296_r16, & ! 53  r4 = 0
       1.192310292190008994893575123620341321165641961929944346694965152E-0308_r16, & ! 54  r8 = 0
       5.237048923789255685016067682849547090933912547968670799599215945E-0319_r16, & ! 55  r8 = 0
       6.563215840328784152380910481606274774134162689948282618021777806E-0343_r16, & ! 56  r8 = 0
       2.089717312149417570964225632118560593923169250964555005070936907E-1045_r16, & ! 57  r8 = 0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 58  +HUGE( y )   +0.00       => +∞
      -1.000000000000000000000000000000000000000000000000000000000000000E+0001_r16 /) ! 59  +HUGE( y )   not defined => +HUGE(y)
!
!   inverse complementary error function, x = erfcinv(y), kind=3 = real*16 = REAL128
    real   (kind=r16), parameter, dimension(derfcinv )   :: dwerfcinvr16 =       (/ & ! 00
        -HUGE(1.000E+0000_r16),    -HUGE(1.000E+0000_r16),   -HUGE(1.000E+0000_r16),& ! 03
             -6.000E+0000_r16,           -5.000E+0000_r16,        -4.000E+0000_r16, & ! 06
             -3.000E+0000_r16,           -2.000E+0000_r16,        -1.500E+0000_r16, & ! 09
             -1.000E+0000_r16,           -0.750E+0000_r16,        -0.500E+0000_r16, & ! 12
             -0.250E+0000_r16,           -0.030E+0000_r16,        -1.150E-0016_r16, & ! 15
             -2.220E-0020_r16,            0.000E+0000_r16,         2.220E-0020_r16, & ! 18
              1.150E-0016_r16,            0.030E+0000_r16,         0.250E+0000_r16, & ! 21
              0.500E+0000_r16,            0.750E+0000_r16,         1.000E+0000_r16, & ! 24
              1.500E+0000_r16,            2.000E+0000_r16,         2.500E+0000_r16, & ! 27
              3.000E+0000_r16,            3.500E+0000_r16,         4.000E+0000_r16, & ! 30
              4.500E+0000_r16,            5.000E+0000_r16,         6.000E+0000_r16, & ! 33
              7.000E+0000_r16,            8.000E+0000_r16,         9.000E+0000_r16, & ! 36
             10.000E+0000_r16,           11.000E+0000_r16,        12.000E+0000_r16, & ! 39
             13.000E+0000_r16,           14.000E+0000_r16,        15.000E+0000_r16, & ! 42
             16.000E+0000_r16,           17.000E+0000_r16,        18.000E+0000_r16, & ! 45
             19.000E+0000_r16,           20.000E+0000_r16,        21.000E+0000_r16, & ! 48
             22.000E+0000_r16,           23.000E+0000_r16,        24.000E+0000_r16, & ! 51
             25.000E+0000_r16,           26.000E+0000_r16,        26.555E+0000_r16, & ! 54
             27.000E+0000_r16,           28.000E+0000_r16,        49.000E+0000_r16, & ! 57
        +HUGE(1.000E+0000_r16),     +HUGE(1.000E+0000_r16)                         /) ! 59
!
!   erfcinv: table
!
!      k    y = erfc(x), 64 digits exact                                               x = erfcinv(y)
!           1.234567890123456789012345678901234567890123456789012345678901234E+0000    123.456E+0000
!
!      1                                                                 +HUGE( y )       -HUGE( y )  not defined => -HUGE(y)
!      2    1.500000000000000000000000000000000000000000000000000000000000000E+0001       -HUGE( y )  not defined => -HUGE(y)
!      3    2.000000000000000000000000000000000000000000000000000000000000000E+0000       -HUGE( y )  +2.00       => -∞
!      x    1.999999999999999999999999999999999999999999999999999999999999999E+0000    -12.000E+0000  only here
!      x    1.999999999999999999999999999999999999999999997911512416237455242E+0000    -10.000E+0000  only here
!      x    1.999999999999999999999999999999999999586296825348618976194609653E+0000     -9.000E+0000  only here
!      x    1.999999999999999999999999999988775702827017072920032111556829720E+0000     -8.000E+0000  only here
!      x    1.999999999999999999999958161743922205856013859897761000677499703E+0000     -7.000E+0000  only here
!      4    1.999999999999999978480263287501086883406649600812615369522485938E+0000     -6.000E+0000  r10/16
!      5    1.999999999998462540205571965149811656514616621109881949685276620E+0000     -5.000E+0000  r10/16
!      6    1.999999984582742099719981147840326513115951427854746410808831657E+0000     -4.000E+0000  r8,  r4 = 0
!      7    1.999977909503001414558627223870417679620152292912600750342761045E+0000     -3.000E+0000  r4
!      8    1.995322265018952734162069256367252928610891797040060076738352326E+0000     -2.000E+0000
!      9    1.966105146475310727066976261645947858681410479257636780449967846E+0000     -1.500E+0000
!     10    1.842700792949714869341220635082609259296066997966302908459937897E+0000     -1.000E+0000
!     11    1.711155633653515131598937834591410777374205954096537232278133397E+0000     -0.750E+0000
!     12    1.520499877813046537682746653891964528736451575757963700058805725E+0000     -0.500E+0000
!     13    1.276326390168236932985068267764815712065353977892311254082471931E+0000     -0.250E+0000
!     14    1.033841222341735433302216654256875465225733912120593527612714135E+0000     -0.030E+0000
!     15    1.000000000000000129763604215983945998058273858977122702909725949E+0000     -1.150E-0016
!     16    1.000000000000000000025050017509520379140494727649298302811471732E+0000     -2.220E-0020
!     17    1.000000000000000000000000000000000000000000000000000000000000000E+0000      0.000E+0000  +1.00       => 0.00
!     18    9.999999999999999999749499824904796208595052723507016971885282672E-0001      2.220E-0020
!     19    9.999999999999998702363957840160540019417261410228772970902740502E-0001      1.150E-0016
!     20    9.661587776582645666977833457431245347742660878794064723872858648E-0001      0.030E+0000
!     21    7.236736098317630670149317322351842879346460221076887459175280687E-0001      0.250E+0000
!     22    4.795001221869534623172533461080354712635484242420362999411942743E-0001      0.500E+0000
!     23    2.888443663464848684010621654085892226257940459034627677218666028E-0001      0.750E+0000
!     24    1.572992070502851306587793649173907407039330020336970915400621021E-0001      1.000E+0000
!     25    3.389485352468927293302373835405214131858952074236321955003215355E-0002      1.500E+0000
!     26    4.677734981047265837930743632747071389108202959939923261647673799E-0003      2.000E+0000
!     27    4.069520174449589395642157399749127203486774037134201391207787690E-0004      2.500E+0000
!     28    2.209049699858544137277612958232037984770708739924965723895484294E-0005      3.000E+0000
!     29    7.430983723414127455236837560956357206600921727974625911095699435E-0007      3.500E+0000
!     30    1.541725790028001885215967348688404857214525358919116834290499421E-0008      4.000E+0000
!     31    1.966160441542887476279160367664332660577670599611646975681185960E-0010      4.500E+0000
!     32    1.537459794428034850188343485383378890118050314723379930687914055E-0012      5.000E+0000
!     33    2.151973671249891311659335039918738463047751406168854210052789205E-0017      6.000E+0000
!     34    4.183825607779414398614010223899932250029617413812460152041634353E-0023      7.000E+0000
!     35    1.122429717298292707996788844317027909343192916447896338591267524E-0029      8.000E+0000
!     36    4.137031746513810238053903467362524595710191985947962459805909901E-0037      9.000E+0000  r4
!     37    2.088487583762544757000786294957788611560818119321163727012213713E-0045     10.000E+0000  r8,  r4 = 0
!     38    1.440866137943694680339809702856082753964395230973580405075797246E-0054     11.000E+0000
!     39    1.356261169205904212780306156590417572666782233288104929912379537E-0064     12.000E+0000
!     40    1.739557315466724521804198548243026985088574583449008314936058740E-0075     13.000E+0000
!     41    3.037229847750311665115172806783328447911896669343422595150070601E-0087     14.000E+0000
!     42    7.212994172451206666565066558692927109934090929825383240467341996E-0100     15.000E+0000
!     43    2.328485751571530693364872854573442597534396948094948021516495095E-0113     16.000E+0000
!     44    1.021228015094260881145599235077652994401543730587571172553506555E-0127     17.000E+0000
!     45    6.082369231816399307668466715702274949587610076867576851592785891E-0143     18.000E+0000
!     46    4.917722839256475446413297625239608170930672249989259036315216983E-0159     19.000E+0000
!     47    5.395865611607900928934999167905345604088272670923605283470103784E-0176     20.000E+0000
!     48    8.032453871022455669021356947138268888967875692513378852786023784E-0194     21.000E+0000
!     49    1.621905860933472513052034647026123265177675596121851615118740214E-0212     22.000E+0000
!     50    4.441265948088057244074884428946738565068812405566605852511837779E-0232     23.000E+0000
!     51    1.648982583151933514218512437543746903943061790303826174801039526E-0252     24.000E+0000
!     52    8.300172571196522752044012769513722768714223191886699596054448582E-0274     25.000E+0000
!     53    5.663192408856142846475727896926092580328777560925807368646253847E-0296     26.000E+0000  r8,      r4 = 0
!     54    1.192310292190008994893575123620341321165641961929944346694965152E-0308     26.555E+0000  r10/16,  r8 = 0
!     55    5.237048923789255685016067682849547090933912547968670799599215945E-0319     27.000E+0000  r10/16,  r8 = 0
!     56    6.563215840328784152380910481606274774134162689948282618021777806E-0343     28.000E+0000  r10/16,  r8 = 0
!     57    2.089717312149417570964225632118560593923169250964555005070936907E-1045     49.000E+0000  r10/16,  r8 = 0
!     58    0.000000000000000000000000000000000000000000000000000000000000000E+0000       +HUGE( y )  +0.00       => +∞
!     59   -1.000000000000000000000000000000000000000000000000000000000000000E+0001       +HUGE( y )  not defined => +HUGE(y)
!     xx                                                                 -HUGE( y )       +HUGE( y )  not defined => +HUGE(y)
!
!------------------------------------ ERFCXINV ---------------------------------------------------------------------------------
!
!   Inverse Scaled Complementary Error Function: x = erfcxinv(y)
!
!   erfcxinv: REAL*4
!
!   random variable y für x = erfcxinv(y) kind=1 = real*4 = REAL32
    real   (kind=r4 ), parameter, dimension(derfcxinv)   :: dyerfcxinvr4 =       (/ & ! 00
                                                             +HUGE(1.000E+0000_r4), & ! 01  +∞ => -∞ all
                                                             +HUGE(1.000E+0000_r4), & ! 02
                                                             +HUGE(1.000E+0000_r4), & ! 03
                                                             +HUGE(1.000E+0000_r4), & ! 04
                                                             +HUGE(1.000E+0000_r4), & ! 05
                                                             +HUGE(1.000E+0000_r4), & ! 06
                                                             +HUGE(1.000E+0000_r4), & ! 07  r*8
       3.012194629170061096705188260335349963175715743878784001794360069E+0035_r4,  & ! 08
       1.247029816162323376581847741778695432346190276474884757933513822E+0028_r4,  & ! 09
       3.814693144990199381050119881853367741842889596589865930650911766E+0021_r4,  & ! 10
       8.622463094230390361450276785175496392091021908830770647053340636E+0015_r4,  & ! 11
       1.440097986746610404105896343058821037439553145762016106106219209E+0011_r4,  & ! 12
       1.777222090401628764846465759211772930179995731346404213150261392E+0007_r4,  & ! 13
       1.620598885399958662546957408405020630903572419029912007078465534E+0004_r4,  & ! 14
       1.089409043899779724123554338248132140422788747719728953862052389E+0002_r4,  & ! 15
       1.865388625626273393874641550130021127851120319405663701880037332E+0001_r4,  & ! 16
       5.008980080762283466309824598214809814694334684235666486188395484E+0000_r4,  & ! 17
       3.003171663627452308696662251659107601538747047728341647273491144E+0000_r4,  & ! 18
       1.952360489182557093276047713441130979890255339297294572744188004E+0000_r4,  & ! 19
       1.358642370104722115210042016948988220013808502272091357355815027E+0000_r4,  & ! 20
       1.034772098273178019832013143072154484345890514100185925367614013E+0000_r4,  & ! 21
       1.000000000000000129763604215983959223058273858978838826575482337E+0000_r4,  & ! 22
       1.000000000000000000025050017509520379140987567649298302811484078E+0000_r4,  & ! 23
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 24
       9.999999999999999999749499824904796208599981123507016971885159216E-0001_r4,  & ! 25
       9.999999999999998702363957840160672269417261410211611734245176626E-0001_r4,  & ! 26
       9.670287119698766650109632717495573341360302147775554874047279156E-0001_r4,  & ! 27
       7.703465477309967439167391723367911261876423850266152456827999940E-0001_r4,  & ! 28
       6.156903441929258748707934226837419367823063912656316056908265890E-0001_r4,  & ! 29
       5.069376502931448057914318215388277759377280127631554128297343480E-0001_r4,  & ! 30
       4.275835761558070044107503444905151808201595031642526637455397707E-0001_r4,  & ! 31
       3.215854164543175023543225877232655690292467412681370014065243116E-0001_r4,  & ! 32
       2.553956763105057438650885809085427633025993052552420654479489936E-0001_r4,  & ! 33
       2.108063640611435806471120484067023193383079571604570144749339935E-0001_r4,  & ! 34
       1.790011511813899504192948153136209872279853641068542156627588395E-0001_r4,  & ! 35
       1.552936556088942974027264975818789317720293465561794469954531758E-0001_r4,  & ! 36
       1.369994576250613898894451713998054823302492355480653627115098493E-0001_r4,  & ! 37
       1.224848042738414175492254544846529190937124368407817995771874189E-0001_r4,  & ! 38
       1.107046377330686263702120864917530505889468618208519881185565845E-0001_r4,  & ! 39
       9.277656780053835438948670532472992481448699505136245821506569383E-0002_r4,  & ! 40
       7.980005432915293348986449770087429955765181389059070298608217360E-0002_r4,  & ! 41
       6.998516620088092772275224944204516970748441002661746054466808797E-0002_r4,  & ! 42
       6.230772403777468414653749202684764733993894726072122845426847610E-0002_r4,  & ! 43
       5.614099274382258585751738722046831156515725665507548351903492024E-0002_r4,  & ! 44
       5.108059475808844370998558301402977200031441462396825283402806716E-0002_r4,  & ! 45
       4.685422101489376261958841339939665904323931915718803315921704839E-0002_r4,  & ! 46
       4.327192186460969266318759898303380115440527301308371350230131664E-0002_r4,  & ! 47
       4.019722865021845930606200422921333952729661347532428877805797534E-0002_r4,  & ! 48
       3.752960638850576574606117818254821602507161134468271480347284945E-0002_r4,  & ! 49
       3.519337782493083756637297489953351970142777219206202563404219860E-0002_r4,  & ! 50
       3.313049999972553669977277294801872035208122227669309138887114781E-0002_r4,  & ! 51
       3.129571781590520988566477909161383437317081613057417169179835202E-0002_r4,  & ! 52
       2.965323064126216352519623531435461785871688420417814149329632135E-0002_r4,  & ! 53
       2.817434874105131931864915453447075835251573225296480740389798765E-0002_r4,  & ! 54
       2.683581315864795664219687801521825439579034377858379753676716914E-0002_r4,  & ! 55
       2.561857000587945266761757947455846538023360563274733504207114582E-0002_r4,  & ! 56
       2.450686208928260590649277351694335889447765285330176577636292659E-0002_r4,  & ! 57
       2.348754606368264051860302591666840906927531002911224978936485192E-0002_r4,  & ! 58
       2.254957243264135894360458395939187726006906701640961131274410338E-0002_r4,  & ! 59
       2.168358485056290661617299168729320269579531998199312854310461437E-0002_r4,  & ! 60
       2.123104471385057441351983960547894758287566770903743534120367476E-0002_r4,  & ! 61
       2.088160799042094067409449019293500894020380993699978983587139087E-0002_r4,  & ! 62
       1.175140054577856629795290707654389501031917423007629905837252269E-0002_r4,  & ! 63
       8.059851193539374705395911191748479743861879040456803369569588902E-0009_r4,  & ! 64  r4
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 65  r4
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 66
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 67  r*8
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 68  r*10/16
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 69  r*10/16  r8 = 0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 70  r*10/16
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 71  r*10/16
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 72  +0.00       => +∞
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r4,  & ! 73  not defined => +HUGE(y)
                                                             -HUGE(1.000E+0000_r4) /) ! 74  not defined => +HUGE(y)
!
!   inverse scaled complementary error function, x = erfcxinv(y), kind=1 = real*4 = REAL32
    real   (kind=r4 ), parameter, dimension(derfcxinv)   :: dwerfcxinvr4 =       (/ & ! 00
                              -SQRT( LOG( HUGE(1.000E+0000_r4)/2.000E+0000_r4 ) ),  & ! 01
                              -SQRT( LOG( HUGE(1.000E+0000_r4)/2.000E+0000_r4 ) ),  & ! 02
                              -SQRT( LOG( HUGE(1.000E+0000_r4)/2.000E+0000_r4 ) ),  & ! 03
                              -SQRT( LOG( HUGE(1.000E+0000_r4)/2.000E+0000_r4 ) ),  & ! 04
                              -SQRT( LOG( HUGE(1.000E+0000_r4)/2.000E+0000_r4 ) ),  & ! 05
                              -SQRT( LOG( HUGE(1.000E+0000_r4)/2.000E+0000_r4 ) ),  & ! 06
                              -SQRT( LOG( HUGE(1.000E+0000_r4)/2.000E+0000_r4 ) ),  & ! 07
             -9.000E+0000_r4,            -8.000E+0000_r4,         -7.000E+0000_r4,  & ! 10
             -6.000E+0000_r4,            -5.000E+0000_r4,         -4.000E+0000_r4,  & ! 13
             -3.000E+0000_r4,            -2.000E+0000_r4,         -1.500E+0000_r4,  & ! 16
             -1.000E+0000_r4,            -0.750E+0000_r4,         -0.500E+0000_r4,  & ! 19
             -0.250E+0000_r4,            -0.030E+0000_r4,         -1.150E-0016_r4,  & ! 22
             -2.220E-0020_r4,             0.000E+0000_r4,          2.220E-0020_r4,  & ! 25
              1.150E-0016_r4,             0.030E+0000_r4,          0.250E+0000_r4,  & ! 28
              0.500E+0000_r4,             0.750E+0000_r4,          1.000E+0000_r4,  & ! 31
              1.500E+0000_r4,             2.000E+0000_r4,          2.500E+0000_r4,  & ! 34
              3.000E+0000_r4,             3.500E+0000_r4,          4.000E+0000_r4,  & ! 37
              4.500E+0000_r4,             5.000E+0000_r4,          6.000E+0000_r4,  & ! 40
              7.000E+0000_r4,             8.000E+0000_r4,          9.000E+0000_r4,  & ! 43
             10.000E+0000_r4,            11.000E+0000_r4,         12.000E+0000_r4,  & ! 46
             13.000E+0000_r4,            14.000E+0000_r4,         15.000E+0000_r4,  & ! 49
             16.000E+0000_r4,            17.000E+0000_r4,         18.000E+0000_r4,  & ! 52
             19.000E+0000_r4,            20.000E+0000_r4,         21.000E+0000_r4,  & ! 55
             22.000E+0000_r4,            23.000E+0000_r4,         24.000E+0000_r4,  & ! 58
             25.000E+0000_r4,            26.000E+0000_r4,         26.555E+0000_r4,  & ! 61
             27.000E+0000_r4,            48.000E+0000_r4,          7.000E+0007_r4,  & ! 64
        +HUGE(1.000E+0000_r4),      +HUGE(1.000E+0000_r4),   +HUGE(1.000E+0000_r4), & ! 67
        +HUGE(1.000E+0000_r4),      +HUGE(1.000E+0000_r4),   +HUGE(1.000E+0000_r4), & ! 70
        +HUGE(1.000E+0000_r4),      +HUGE(1.000E+0000_r4),   +HUGE(1.000E+0000_r4), & ! 73
        +HUGE(1.000E+0000_r4)                                                      /) ! 74
!
!   erfcxinv: REAL*8
!
!   random variable y für x = erfcxinv(y) kind=2 = real*8 = REAL64
    real   (kind=r8 ), parameter, dimension(derfcxinv)   :: dyerfcxinvr8 =       (/ & ! 00
                                                             +HUGE(1.000E+0000_r8), & ! 01  +∞ => -∞ all
                                                             +HUGE(1.000E+0000_r8), & ! 02
                                                             +HUGE(1.000E+0000_r8), & ! 03
                                                             +HUGE(1.000E+0000_r8), & ! 04
       3.561328767003070022312731489880608884148753896142635966149398407E+0306_r8,  & ! 05
       7.657724931490568351526773657706215529330480507562346940485035732E+0293_r8,  & ! 06
       5.376234283632270896825251103160027174722223749134285208656063137E+0043_r8,  & ! 07  r*8
       3.012194629170061096705188260335349963175715743878784001794360069E+0035_r8,  & ! 08
       1.247029816162323376581847741778695432346190276474884757933513822E+0028_r8,  & ! 09
       3.814693144990199381050119881853367741842889596589865930650911766E+0021_r8,  & ! 10
       8.622463094230390361450276785175496392091021908830770647053340636E+0015_r8,  & ! 11
       1.440097986746610404105896343058821037439553145762016106106219209E+0011_r8,  & ! 12
       1.777222090401628764846465759211772930179995731346404213150261392E+0007_r8,  & ! 13
       1.620598885399958662546957408405020630903572419029912007078465534E+0004_r8,  & ! 14
       1.089409043899779724123554338248132140422788747719728953862052389E+0002_r8,  & ! 15
       1.865388625626273393874641550130021127851120319405663701880037332E+0001_r8,  & ! 16
       5.008980080762283466309824598214809814694334684235666486188395484E+0000_r8,  & ! 17
       3.003171663627452308696662251659107601538747047728341647273491144E+0000_r8,  & ! 18
       1.952360489182557093276047713441130979890255339297294572744188004E+0000_r8,  & ! 19
       1.358642370104722115210042016948988220013808502272091357355815027E+0000_r8,  & ! 20
       1.034772098273178019832013143072154484345890514100185925367614013E+0000_r8,  & ! 21
       1.000000000000000129763604215983959223058273858978838826575482337E+0000_r8,  & ! 22
       1.000000000000000000025050017509520379140987567649298302811484078E+0000_r8,  & ! 23
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 24
       9.999999999999999999749499824904796208599981123507016971885159216E-0001_r8,  & ! 25
       9.999999999999998702363957840160672269417261410211611734245176626E-0001_r8,  & ! 26
       9.670287119698766650109632717495573341360302147775554874047279156E-0001_r8,  & ! 27
       7.703465477309967439167391723367911261876423850266152456827999940E-0001_r8,  & ! 28
       6.156903441929258748707934226837419367823063912656316056908265890E-0001_r8,  & ! 29
       5.069376502931448057914318215388277759377280127631554128297343480E-0001_r8,  & ! 30
       4.275835761558070044107503444905151808201595031642526637455397707E-0001_r8,  & ! 31
       3.215854164543175023543225877232655690292467412681370014065243116E-0001_r8,  & ! 32
       2.553956763105057438650885809085427633025993052552420654479489936E-0001_r8,  & ! 33
       2.108063640611435806471120484067023193383079571604570144749339935E-0001_r8,  & ! 34
       1.790011511813899504192948153136209872279853641068542156627588395E-0001_r8,  & ! 35
       1.552936556088942974027264975818789317720293465561794469954531758E-0001_r8,  & ! 36
       1.369994576250613898894451713998054823302492355480653627115098493E-0001_r8,  & ! 37
       1.224848042738414175492254544846529190937124368407817995771874189E-0001_r8,  & ! 38
       1.107046377330686263702120864917530505889468618208519881185565845E-0001_r8,  & ! 39
       9.277656780053835438948670532472992481448699505136245821506569383E-0002_r8,  & ! 40
       7.980005432915293348986449770087429955765181389059070298608217360E-0002_r8,  & ! 41
       6.998516620088092772275224944204516970748441002661746054466808797E-0002_r8,  & ! 42
       6.230772403777468414653749202684764733993894726072122845426847610E-0002_r8,  & ! 43
       5.614099274382258585751738722046831156515725665507548351903492024E-0002_r8,  & ! 44
       5.108059475808844370998558301402977200031441462396825283402806716E-0002_r8,  & ! 45
       4.685422101489376261958841339939665904323931915718803315921704839E-0002_r8,  & ! 46
       4.327192186460969266318759898303380115440527301308371350230131664E-0002_r8,  & ! 47
       4.019722865021845930606200422921333952729661347532428877805797534E-0002_r8,  & ! 48
       3.752960638850576574606117818254821602507161134468271480347284945E-0002_r8,  & ! 49
       3.519337782493083756637297489953351970142777219206202563404219860E-0002_r8,  & ! 50
       3.313049999972553669977277294801872035208122227669309138887114781E-0002_r8,  & ! 51
       3.129571781590520988566477909161383437317081613057417169179835202E-0002_r8,  & ! 52
       2.965323064126216352519623531435461785871688420417814149329632135E-0002_r8,  & ! 53
       2.817434874105131931864915453447075835251573225296480740389798765E-0002_r8,  & ! 54
       2.683581315864795664219687801521825439579034377858379753676716914E-0002_r8,  & ! 55
       2.561857000587945266761757947455846538023360563274733504207114582E-0002_r8,  & ! 56
       2.450686208928260590649277351694335889447765285330176577636292659E-0002_r8,  & ! 57
       2.348754606368264051860302591666840906927531002911224978936485192E-0002_r8,  & ! 58
       2.254957243264135894360458395939187726006906701640961131274410338E-0002_r8,  & ! 59
       2.168358485056290661617299168729320269579531998199312854310461437E-0002_r8,  & ! 60
       2.123104471385057441351983960547894758287566770903743534120367476E-0002_r8,  & ! 61
       2.088160799042094067409449019293500894020380993699978983587139087E-0002_r8,  & ! 62
       1.175140054577856629795290707654389501031917423007629905837252269E-0002_r8,  & ! 63
       8.059851193539374705395911191748479743861879040456803369569588902E-0009_r8,  & ! 64  r4
       8.059851193539375527829706450868179797772151847557126526344081738E-0072_r8,  & ! 65  r8
       1.253754630106125082106843225690605746320112509619997459653523826E-0151_r8,  & ! 66
       2.350789931448984528950331048169885774350210955537495236850357173E-0301_r8,  & ! 67  r*8
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 68  r*10/16
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 69  r*10/16  r8 = 0
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 70  r*10/16
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 71  r*10/16
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 72  +0.00       => +∞
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r8,  & ! 73  not defined => +HUGE(y)
                                                             -HUGE(1.000E+0000_r8) /) ! 74  not defined => +HUGE(y)
!
!   inverse scaled complementary error function, x = erfcxinv(y), kind=2 = real*8 = REAL64
    real   (kind=r8 ), parameter, dimension(derfcxinv)   :: dwerfcxinvr8 =       (/ & ! 00
                              -SQRT( LOG( HUGE(1.000E+0000_r8)/2.000E+0000_r8 ) ),  & ! 01
                              -SQRT( LOG( HUGE(1.000E+0000_r8)/2.000E+0000_r8 ) ),  & ! 02
                              -SQRT( LOG( HUGE(1.000E+0000_r8)/2.000E+0000_r8 ) ),  & ! 03
                              -SQRT( LOG( HUGE(1.000E+0000_r8)/2.000E+0000_r8 ) ),  & ! 04
            -26.555E+0000_r8,           -26.000E+0000_r8,        -10.000E+0000_r8,  & ! 07
             -9.000E+0000_r8,            -8.000E+0000_r8,         -7.000E+0000_r8,  & ! 10
             -6.000E+0000_r8,            -5.000E+0000_r8,         -4.000E+0000_r8,  & ! 13
             -3.000E+0000_r8,            -2.000E+0000_r8,         -1.500E+0000_r8,  & ! 16
             -1.000E+0000_r8,            -0.750E+0000_r8,         -0.500E+0000_r8,  & ! 19
             -0.250E+0000_r8,            -0.030E+0000_r8,         -1.150E-0016_r8,  & ! 22
             -2.220E-0020_r8,             0.000E+0000_r8,          2.220E-0020_r8,  & ! 25
              1.150E-0016_r8,             0.030E+0000_r8,          0.250E+0000_r8,  & ! 28
              0.500E+0000_r8,             0.750E+0000_r8,          1.000E+0000_r8,  & ! 31
              1.500E+0000_r8,             2.000E+0000_r8,          2.500E+0000_r8,  & ! 34
              3.000E+0000_r8,             3.500E+0000_r8,          4.000E+0000_r8,  & ! 37
              4.500E+0000_r8,             5.000E+0000_r8,          6.000E+0000_r8,  & ! 40
              7.000E+0000_r8,             8.000E+0000_r8,          9.000E+0000_r8,  & ! 43
             10.000E+0000_r8,            11.000E+0000_r8,         12.000E+0000_r8,  & ! 46
             13.000E+0000_r8,            14.000E+0000_r8,         15.000E+0000_r8,  & ! 49
             16.000E+0000_r8,            17.000E+0000_r8,         18.000E+0000_r8,  & ! 52
             19.000E+0000_r8,            20.000E+0000_r8,         21.000E+0000_r8,  & ! 55
             22.000E+0000_r8,            23.000E+0000_r8,         24.000E+0000_r8,  & ! 58
             25.000E+0000_r8,            26.000E+0000_r8,         26.555E+0000_r8,  & ! 61
             27.000E+0000_r8,            48.000E+0000_r8,          7.000E+0007_r8,  & ! 64
              7.000E+0070_r8,             4.500E+0150_r8,          2.400E+0300_r8,  & ! 67
        +HUGE(1.000E+0000_r8),      +HUGE(1.000E+0000_r8),   +HUGE(1.000E+0000_r8), & ! 70
        +HUGE(1.000E+0000_r8),      +HUGE(1.000E+0000_r8),   +HUGE(1.000E+0000_r8), & ! 73
        +HUGE(1.000E+0000_r8)                                                      /) ! 74
!
!   erfcxinv: REAL*10/16
!
!   random variable y für x = erfcxinv(y), kind=3 = real*16 = REAL128 (Silverfrost real*10)
    real   (kind=r16), parameter, dimension(derfcxinv)   :: dyerfcxinvr16 =      (/ & ! 00
                                                             +HUGE(1.000E+0000_r16),& ! 01  +∞ => -∞ all
       1.081007384292911774009323562529646498192417673036703432246297354E+4880_r16, & ! 02
       1.101744891230519013281062545490362618657575418039039594609620564E+1043_r16, & ! 03
       7.974570524085193127093722094668700648898179096534145424698741091E+0316_r16, & ! 04
       3.561328767003070022312731489880608884148753896142635966149398407E+0306_r16, & ! 05
       7.657724931490568351526773657706215529330480507562346940485035732E+0293_r16, & ! 06
       5.376234283632270896825251103160027174722223749134285208656063137E+0043_r16, & ! 07  r*8
       3.012194629170061096705188260335349963175715743878784001794360069E+0035_r16, & ! 08
       1.247029816162323376581847741778695432346190276474884757933513822E+0028_r16, & ! 09
       3.814693144990199381050119881853367741842889596589865930650911766E+0021_r16, & ! 10
       8.622463094230390361450276785175496392091021908830770647053340636E+0015_r16, & ! 11
       1.440097986746610404105896343058821037439553145762016106106219209E+0011_r16, & ! 12
       1.777222090401628764846465759211772930179995731346404213150261392E+0007_r16, & ! 13
       1.620598885399958662546957408405020630903572419029912007078465534E+0004_r16, & ! 14
       1.089409043899779724123554338248132140422788747719728953862052389E+0002_r16, & ! 15
       1.865388625626273393874641550130021127851120319405663701880037332E+0001_r16, & ! 16
       5.008980080762283466309824598214809814694334684235666486188395484E+0000_r16, & ! 17
       3.003171663627452308696662251659107601538747047728341647273491144E+0000_r16, & ! 18
       1.952360489182557093276047713441130979890255339297294572744188004E+0000_r16, & ! 19
       1.358642370104722115210042016948988220013808502272091357355815027E+0000_r16, & ! 20
       1.034772098273178019832013143072154484345890514100185925367614013E+0000_r16, & ! 21
       1.000000000000000129763604215983959223058273858978838826575482337E+0000_r16, & ! 22
       1.000000000000000000025050017509520379140987567649298302811484078E+0000_r16, & ! 23
       1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 24
       9.999999999999999999749499824904796208599981123507016971885159216E-0001_r16, & ! 25
       9.999999999999998702363957840160672269417261410211611734245176626E-0001_r16, & ! 26
       9.670287119698766650109632717495573341360302147775554874047279156E-0001_r16, & ! 27
       7.703465477309967439167391723367911261876423850266152456827999940E-0001_r16, & ! 28
       6.156903441929258748707934226837419367823063912656316056908265890E-0001_r16, & ! 29
       5.069376502931448057914318215388277759377280127631554128297343480E-0001_r16, & ! 30
       4.275835761558070044107503444905151808201595031642526637455397707E-0001_r16, & ! 31
       3.215854164543175023543225877232655690292467412681370014065243116E-0001_r16, & ! 32
       2.553956763105057438650885809085427633025993052552420654479489936E-0001_r16, & ! 33
       2.108063640611435806471120484067023193383079571604570144749339935E-0001_r16, & ! 34
       1.790011511813899504192948153136209872279853641068542156627588395E-0001_r16, & ! 35
       1.552936556088942974027264975818789317720293465561794469954531758E-0001_r16, & ! 36
       1.369994576250613898894451713998054823302492355480653627115098493E-0001_r16, & ! 37
       1.224848042738414175492254544846529190937124368407817995771874189E-0001_r16, & ! 38
       1.107046377330686263702120864917530505889468618208519881185565845E-0001_r16, & ! 39
       9.277656780053835438948670532472992481448699505136245821506569383E-0002_r16, & ! 40
       7.980005432915293348986449770087429955765181389059070298608217360E-0002_r16, & ! 41
       6.998516620088092772275224944204516970748441002661746054466808797E-0002_r16, & ! 42
       6.230772403777468414653749202684764733993894726072122845426847610E-0002_r16, & ! 43
       5.614099274382258585751738722046831156515725665507548351903492024E-0002_r16, & ! 44
       5.108059475808844370998558301402977200031441462396825283402806716E-0002_r16, & ! 45
       4.685422101489376261958841339939665904323931915718803315921704839E-0002_r16, & ! 46
       4.327192186460969266318759898303380115440527301308371350230131664E-0002_r16, & ! 47
       4.019722865021845930606200422921333952729661347532428877805797534E-0002_r16, & ! 48
       3.752960638850576574606117818254821602507161134468271480347284945E-0002_r16, & ! 49
       3.519337782493083756637297489953351970142777219206202563404219860E-0002_r16, & ! 50
       3.313049999972553669977277294801872035208122227669309138887114781E-0002_r16, & ! 51
       3.129571781590520988566477909161383437317081613057417169179835202E-0002_r16, & ! 52
       2.965323064126216352519623531435461785871688420417814149329632135E-0002_r16, & ! 53
       2.817434874105131931864915453447075835251573225296480740389798765E-0002_r16, & ! 54
       2.683581315864795664219687801521825439579034377858379753676716914E-0002_r16, & ! 55
       2.561857000587945266761757947455846538023360563274733504207114582E-0002_r16, & ! 56
       2.450686208928260590649277351694335889447765285330176577636292659E-0002_r16, & ! 57
       2.348754606368264051860302591666840906927531002911224978936485192E-0002_r16, & ! 58
       2.254957243264135894360458395939187726006906701640961131274410338E-0002_r16, & ! 59
       2.168358485056290661617299168729320269579531998199312854310461437E-0002_r16, & ! 60
       2.123104471385057441351983960547894758287566770903743534120367476E-0002_r16, & ! 61
       2.088160799042094067409449019293500894020380993699978983587139087E-0002_r16, & ! 62
       1.175140054577856629795290707654389501031917423007629905837252269E-0002_r16, & ! 63
       8.059851193539374705395911191748479743861879040456803369569588902E-0009_r16, & ! 64  r4
       8.059851193539375527829706450868179797772151847557126526344081738E-0072_r16, & ! 65  r8
       1.253754630106125082106843225690605746320112509619997459653523826E-0151_r16, & ! 66
       2.350789931448984528950331048169885774350210955537495236850357173E-0301_r16, & ! 67  r*8
       2.350789931448984528950331048169885774350210955537495236850357173E-0307_r16, & ! 68  r*10/16
       2.212508170775514850776782162983421905270786781682348458212100869E-0308_r16, & ! 69  r*10/16  r8 = 0
       1.128379167095512573896158903121545171688101258657997713688171443E-1046_r16, & ! 70  r*10/16
       2.820947917738781434740397257803862929220253146644994284220428608E-3301_r16, & ! 71  r*10/16
       0.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 72  +0.00       => +∞
      -1.000000000000000000000000000000000000000000000000000000000000000E+0000_r16, & ! 73  not defined => +HUGE(y)
                                                             -HUGE(1.000E+0000_r16)/) ! 74  not defined => +HUGE(y)
!
!   inverse scaled complementary erf, x = erfcxinv(y), kind=3 = real*16 = REAL128
    real   (kind=r16), parameter, dimension(derfcxinv)   :: dwerfcxinvr16 =      (/ & ! 00
                             -SQRT( LOG( HUGE(1.000E+0000_r16)/2.000E+0000_r16 ) ), & ! 01
           -106.000E+0000_r16,          -49.000E+0000_r16,       -27.000E+0000_r16, & ! 04
            -26.555E+0000_r16,          -26.000E+0000_r16,       -10.000E+0000_r16, & ! 07
             -9.000E+0000_r16,           -8.000E+0000_r16,        -7.000E+0000_r16, & ! 10
             -6.000E+0000_r16,           -5.000E+0000_r16,        -4.000E+0000_r16, & ! 13
             -3.000E+0000_r16,           -2.000E+0000_r16,        -1.500E+0000_r16, & ! 16
             -1.000E+0000_r16,           -0.750E+0000_r16,        -0.500E+0000_r16, & ! 19
             -0.250E+0000_r16,           -0.030E+0000_r16,        -1.150E-0016_r16, & ! 22
             -2.220E-0020_r16,            0.000E+0000_r16,         2.220E-0020_r16, & ! 25
              1.150E-0016_r16,            0.030E+0000_r16,         0.250E+0000_r16, & ! 28
              0.500E+0000_r16,            0.750E+0000_r16,         1.000E+0000_r16, & ! 31
              1.500E+0000_r16,            2.000E+0000_r16,         2.500E+0000_r16, & ! 34
              3.000E+0000_r16,            3.500E+0000_r16,         4.000E+0000_r16, & ! 37
              4.500E+0000_r16,            5.000E+0000_r16,         6.000E+0000_r16, & ! 40
              7.000E+0000_r16,            8.000E+0000_r16,         9.000E+0000_r16, & ! 43
             10.000E+0000_r16,           11.000E+0000_r16,        12.000E+0000_r16, & ! 46
             13.000E+0000_r16,           14.000E+0000_r16,        15.000E+0000_r16, & ! 49
             16.000E+0000_r16,           17.000E+0000_r16,        18.000E+0000_r16, & ! 52
             19.000E+0000_r16,           20.000E+0000_r16,        21.000E+0000_r16, & ! 55
             22.000E+0000_r16,           23.000E+0000_r16,        24.000E+0000_r16, & ! 58
             25.000E+0000_r16,           26.000E+0000_r16,        26.555E+0000_r16, & ! 61
             27.000E+0000_r16,           48.000E+0000_r16,         7.000E+0007_r16, & ! 64
              7.000E+0070_r16,            4.500E+0150_r16,         2.400E+0300_r16, & ! 67
              2.400E+0306_r16,            2.550E+0307_r16,         5.000E+1045_r16, & ! 70
              2.000E+3300_r16,      +HUGE(1.000E+0000_r16),  +HUGE(1.000E+0000_r16),& ! 73
        +HUGE(1.000E+0000_r16)                                                     /) ! 74
!
!   erfcxinv: table
!
!      k    y = erfcx(x), 64 digits exact                                             x = erfcxinv(y)
!           1.234567890123456789012345678901234567890123456789012345678901234E+0000   123.456E+0000    
!
!      x                                                             +∞ (+infinity)  -∞ (-infinity)             only table
!      x                                                                 +HUGE( y )      -HUGE( y )    +∞ => -∞ only table
!      1                                                                 +HUGE( y ) -√(LOG(HUGE(y)/2)) +∞ => -∞ all
!      2    1.081007384292911774009323562529646498192417673036703432246297354E+4880  -106.000E+0000
!      3    1.101744891230519013281062545490362618657575418039039594609620564E+1043   -49.000E+0000
!      4    7.974570524085193127093722094668700648898179096534145424698741091E+0316   -27.000E+0000   r*10
!      5    3.561328767003070022312731489880608884148753896142635966149398407E+0306   -26.555E+0000
!      6    7.657724931490568351526773657706215529330480507562346940485035732E+0293   -26.000E+0000
!      7    5.376234283632270896825251103160027174722223749134285208656063137E+0043   -10.000E+0000   r*8
!      8    3.012194629170061096705188260335349963175715743878784001794360069E+0035    -9.000E+0000
!      9    1.247029816162323376581847741778695432346190276474884757933513822E+0028    -8.000E+0000
!     10    3.814693144990199381050119881853367741842889596589865930650911766E+0021    -7.000E+0000
!     11    8.622463094230390361450276785175496392091021908830770647053340636E+0015    -6.000E+0000
!     12    1.440097986746610404105896343058821037439553145762016106106219209E+0011    -5.000E+0000
!     13    1.777222090401628764846465759211772930179995731346404213150261392E+0007    -4.000E+0000
!     14    1.620598885399958662546957408405020630903572419029912007078465534E+0004    -3.000E+0000
!     15    1.089409043899779724123554338248132140422788747719728953862052389E+0002    -2.000E+0000
!     16    1.865388625626273393874641550130021127851120319405663701880037332E+0001    -1.500E+0000
!     17    5.008980080762283466309824598214809814694334684235666486188395484E+0000    -1.000E+0000
!     18    3.003171663627452308696662251659107601538747047728341647273491144E+0000    -0.750E+0000
!     19    1.952360489182557093276047713441130979890255339297294572744188004E+0000    -0.500E+0000
!     20    1.358642370104722115210042016948988220013808502272091357355815027E+0000    -0.250E+0000
!     21    1.034772098273178019832013143072154484345890514100185925367614013E+0000    -0.030E+0000
!     22    1.000000000000000129763604215983959223058273858978838826575482337E+0000    -1.150E-0016
!     23    1.000000000000000000025050017509520379140987567649298302811484078E+0000    -2.220E-0020
!     24    1.000000000000000000000000000000000000000000000000000000000000000E+0000     0.000E+0000
!     25    9.999999999999999999749499824904796208599981123507016971885159216E-0001     2.220E-0020
!     26    9.999999999999998702363957840160672269417261410211611734245176626E-0001     1.150E-0016
!     27    9.670287119698766650109632717495573341360302147775554874047279156E-0001     0.030E+0000
!     28    7.703465477309967439167391723367911261876423850266152456827999940E-0001     0.250E+0000
!     29    6.156903441929258748707934226837419367823063912656316056908265890E-0001     0.500E+0000
!     30    5.069376502931448057914318215388277759377280127631554128297343480E-0001     0.750E+0000
!     31    4.275835761558070044107503444905151808201595031642526637455397707E-0001     1.000E+0000
!     32    3.215854164543175023543225877232655690292467412681370014065243116E-0001     1.500E+0000
!     33    2.553956763105057438650885809085427633025993052552420654479489936E-0001     2.000E+0000
!     34    2.108063640611435806471120484067023193383079571604570144749339935E-0001     2.500E+0000
!     35    1.790011511813899504192948153136209872279853641068542156627588395E-0001     3.000E+0000
!     36    1.552936556088942974027264975818789317720293465561794469954531758E-0001     3.500E+0000
!     37    1.369994576250613898894451713998054823302492355480653627115098493E-0001     4.000E+0000
!     38    1.224848042738414175492254544846529190937124368407817995771874189E-0001     4.500E+0000
!     39    1.107046377330686263702120864917530505889468618208519881185565845E-0001     5.000E+0000
!     40    9.277656780053835438948670532472992481448699505136245821506569383E-0002     6.000E+0000
!     41    7.980005432915293348986449770087429955765181389059070298608217360E-0002     7.000E+0000
!     42    6.998516620088092772275224944204516970748441002661746054466808797E-0002     8.000E+0000
!     43    6.230772403777468414653749202684764733993894726072122845426847610E-0002     9.000E+0000
!     44    5.614099274382258585751738722046831156515725665507548351903492024E-0002    10.000E+0000
!     45    5.108059475808844370998558301402977200031441462396825283402806716E-0002    11.000E+0000
!     46    4.685422101489376261958841339939665904323931915718803315921704839E-0002    12.000E+0000
!     47    4.327192186460969266318759898303380115440527301308371350230131664E-0002    13.000E+0000
!     48    4.019722865021845930606200422921333952729661347532428877805797534E-0002    14.000E+0000
!     49    3.752960638850576574606117818254821602507161134468271480347284945E-0002    15.000E+0000
!     50    3.519337782493083756637297489953351970142777219206202563404219860E-0002    16.000E+0000
!     51    3.313049999972553669977277294801872035208122227669309138887114781E-0002    17.000E+0000
!     52    3.129571781590520988566477909161383437317081613057417169179835202E-0002    18.000E+0000
!     53    2.965323064126216352519623531435461785871688420417814149329632135E-0002    19.000E+0000
!     54    2.817434874105131931864915453447075835251573225296480740389798765E-0002    20.000E+0000
!     55    2.683581315864795664219687801521825439579034377858379753676716914E-0002    21.000E+0000
!     56    2.561857000587945266761757947455846538023360563274733504207114582E-0002    22.000E+0000
!     57    2.450686208928260590649277351694335889447765285330176577636292659E-0002    23.000E+0000
!     58    2.348754606368264051860302591666840906927531002911224978936485192E-0002    24.000E+0000
!     59    2.254957243264135894360458395939187726006906701640961131274410338E-0002    25.000E+0000
!     60    2.168358485056290661617299168729320269579531998199312854310461437E-0002    26.000E+0000
!     61    2.123104471385057441351983960547894758287566770903743534120367476E-0002    26.555E+0000
!     62    2.088160799042094067409449019293500894020380993699978983587139087E-0002    27.000E+0000
!     63    1.175140054577856629795290707654389501031917423007629905837252269E-0002    48.000E+0000
!     64    8.059851193539374705395911191748479743861879040456803369569588902E-0009     7.000E+0007   r4
!     65    8.059851193539375527829706450868179797772151847557126526344081738E-0072     7.000E+0070   r8
!     66    1.253754630106125082106843225690605746320112509619997459653523826E-0151     4.500E+0150   (1/sqrt(pi)) / y
!     67    2.350789931448984528950331048169885774350210955537495236850357173E-0301     2.400E+0300   (1/sqrt(pi)) / y r*8
!     68    2.350789931448984528950331048169885774350210955537495236850357173E-0307     2.400E+0306   (1/sqrt(pi)) / y r*10/16
!     69    2.212508170775514850776782162983421905270786781682348458212100869E-0308     2.550E+0307   (1/sqrt(pi)) / y r*10/16, r8 = 0
!     70    1.128379167095512573896158903121545171688101258657997713688171443E-1046     5.000E+1045   (1/sqrt(pi)) / y r*10/16
!     71    2.820947917738781434740397257803862929220253146644994284220428608E-3301     2.000E+3300   (1/sqrt(pi)) / y r*10/16
!     72    0.000000000000000000000000000000000000000000000000000000000000000E+0000      +HUGE( y )   +0.00       => +∞
!     73   -1.000000000000000000000000000000000000000000000000000000000000000E+0000      +HUGE( y )   not defined => +HUGE(y)
!     74                                                                 -HUGE( y )      +HUGE( y )   not defined => +HUGE(y)
!
!-------------------------------------------------------------------------------------------------------------------------------
!
!   store the value results from the tables
!
    if( k > 0_ik ) then                              ! next i or k
        i = k                                        ! choose k
    else
        i = i + 1_ik                                 ! next i
    end if
!
    h = rki                                          ! variab. rk to 1,2 or 3
!
         if( func == 1_ik ) then                     ! y = erf(x)
             if( i > derf      ) i = 1_ik            ! check max. i
                  if( h == 1_ik ) then               ! real*4
                      xy          = dxerfr4(i)       ! x
                      taberf      = dwerfr4(i)       ! y
             else if( h == 2_ik ) then               ! real*8
                      xy          = dxerfr8(i)       ! x
                      taberf      = dwerfr8(i)       ! y
             else if( h == 3_ik ) then               ! real*10/16
                      xy          = dxerfr16(i)      ! x
                      taberf      = dwerfr16(i)      ! y
             else
                      write( *,'( t2,"DatError mismatch at func = 1" )' )
             end  if
!
    else if( func == 2_ik ) then                     ! y = erfc(x)
             if( i > derfc     ) i = 1_ik            ! check max. i
                  if( h == 1_ik ) then               ! real*4
                      xy          = dxerfcr4(i)      ! x
                      taberf      = dwerfcr4(i)      ! y
             else if( h == 2_ik ) then               ! real*8
                      xy          = dxerfcr8(i)      ! x
                      taberf      = dwerfcr8(i)      ! y
             else if( h == 3_ik ) then               ! real*10/16
                      xy          = dxerfcr16(i)     ! x
                      taberf      = dwerfcr16(i)     ! y
             else
                      write( *,'( t2,"DatError mismatch at func = 2" )' )
             end  if
!
    else if( func == 3_ik ) then                     ! y = erfcx(x)
             if( i > derfcx    ) i = 1_ik            ! check max. i
                  if( h == 1_ik ) then               ! real*4
                      xy          = dxerfcxr4(i)     ! x
                      taberf      = dwerfcxr4(i)     ! y
             else if( h == 2_ik ) then               ! real*8
                      xy          = dxerfcxr8(i)     ! x
                      taberf      = dwerfcxr8(i)     ! y
             else if( h == 3_ik ) then               ! real*10/16
                      xy          = dxerfcxr16(i)    ! x
                      taberf      = dwerfcxr16(i)    ! y
             else
                      write( *,'( t2,"DatError mismatch at func = 3" )' )
             end  if
!
!
    else if( func == 4_ik ) then                     ! x = erfinv(y)
             if( i > derfinv   ) i = 1_ik            ! check max. i
                  if( h == 1_ik ) then               ! real*4
                      xy          = dyerfinvr4(i)    ! y
                      taberf      = dwerfinvr4(i)    ! x
             else if( h == 2_ik ) then               ! real*8
                      xy          = dyerfinvr8(i)    ! y
                      taberf      = dwerfinvr8(i)    ! x
             else if( h == 3_ik ) then               ! real*10/16
                      xy          = dyerfinvr16(i)   ! y
                      taberf      = dwerfinvr16(i)   ! x
             else
                      write( *,'( t2,"DatError mismatch at func = 4" )' )
             end  if
!
    else if( func == 5_ik ) then                     ! x = erfcinv(y)
             if( i > derfcinv  ) i = 1_ik            ! check max. i
                  if( h == 1_ik ) then               ! real*4
                      xy          = dyerfcinvr4(i)   ! y
                      taberf      = dwerfcinvr4(i)   ! x
             else if( h == 2_ik ) then               ! real*8
                      xy          = dyerfcinvr8(i)   ! y
                      taberf      = dwerfcinvr8(i)   ! x
             else if( h == 3_ik ) then               ! real*10/16
                      xy          = dyerfcinvr16(i)  ! y
                      taberf      = dwerfcinvr16(i)  ! x
             else
                      write( *,'( t2,"DatError mismatch at func = 5" )' )
             end  if
!
    else if( func == 6_ik ) then                     ! x = erfcxinv(y)
             if( i > derfcxinv ) i = 1_ik            ! check max. i
                  if( h == 1_ik ) then               ! real*4
                      xy          = dyerfcxinvr4(i)  ! y
                      taberf      = dwerfcxinvr4(i)  ! x
             else if( h == 2_ik ) then               ! real*8
                      xy          = dyerfcxinvr8(i)  ! y
                      taberf      = dwerfcxinvr8(i)  ! x
             else if( h == 3_ik ) then               ! real*10/16
                      xy          = dyerfcxinvr16(i) ! y
                      taberf      = dwerfcxinvr16(i) ! x
             else
                      write( *,'( t2,"DatError mismatch at func = 6" )' )
             end  if
    end  if
!
    return
    end subroutine DatError
!
!-------------------------------------------------------------------------------------------------------------------------------
!