!
!-------------------------------------------------------------------------------------------------------
!   Starting Information, Menu Selection and Screen Output
!
!   subroutine Startinfo (   )                                                     ! screen information
!
!   subroutine Algorithms( func, w, m )                                            ! list of algorithms
!
!   subroutine Menu      ( func, w, m, alg )                                       ! main menu
!   subroutine MenuErf   ( func, w, m, alg )                                       ! error function menu
!
!   subroutine Out1      ( func, algtxt                                          ) ! screen headlines
!   subroutine Out2      ( point, inputxy, tabulyx, fvalue, deviation, iteration ) ! screen calculations
!
!   subroutine ScrClear  (    )                                                    ! 60 new lines
!   subroutine ScrLine   ( ch )                                                    ! S/L line
!   subroutine ScrBreak  ( ch )                                                    ! S/L line + <RETURN>
!
!-------------------------------------------------------------------------------------------------------
!
!   Startinfo
!     Start Screen with important information
!   Reference:
!     ./.
!   Remark:
!     ./.
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    subroutine Startinfo ( )
!
    use kinds, only : compiler, iki, rki, lki, ik
    use const, only : eps, typerr
!
    implicit none
!
!   local variable
    character(len =50),dimension(4):: itxt       ! help data type integer
    character(len =50),dimension(3):: rtxt       ! help data type real
    character(len =50),dimension(3):: ltxt       ! help data type logical
    integer  (kind=ik)             :: Efmt       ! kind for format
    character(len = 9)             :: Eform      ! format of figures
    integer  (kind=ik)             :: h          ! variable for typerr
    character(len =71)             :: txt        ! help variable text
!
!
    write( *, '( 65(/), 105("-"),                            ///, &
    &t24,"Error functions and their Inverses",                //, &
    &t10,"Implementation, adjustments and/or extensions",         &
    &    "by Thomas Hoering",                                  /, &
    &t33, "version 1.00 (2025)",                               /, &
    &t23,"Copyright (C) 2025 Thomas Hoering",                 //)')
!
!   Data type setting
!
!   integer
    itxt(1) = ",  1 byte,  max. 127"
    itxt(2) = ",  2 bytes, max. 32767"
    itxt(3) = ",  4 bytes, max. 2147483647"
    itxt(4) = ",  8 bytes, max. 9223372036854775807"
!
!   real
    rtxt(1) = ",  4 bytes, single (6 digits) precision"
    rtxt(2) = ",  8 bytes, double (15 digits) precision"
    h       = compiler                           ! variabilize compiler
         if( h == 1 ) then                       ! GFortran
             rtxt(3) = ", 16 bytes, quadruple (32 digits) precision"
    else if( h == 2 ) then                       ! Silverfrost ftn95
             rtxt(3) = ", 10 bytes, extended (18 digits) precision"
    else                                         ! compiler not defined
             rtxt(3) = ", compiler and precision not defined"
    end  if
!
!   logical
    ltxt(1) = ",  1 byte"
    ltxt(2) = ",  2 bytes"
    ltxt(3) = ",  4 bytes"
!
    write( *,'( 105("-")                       //, &
    &t15,"Data Type Setting:"                   /, &
    &t20,"      INTEGER : kind = ", i2, a,      /, &
    &t20,"      REAL    : kind = ", i2, a,      /, &
    &t20,"      LOGICAL : kind = ", i2, a,      /)') iki, TRIM(itxt(iki)), &
                                                     rki, TRIM(rtxt(rki)), &
                                                     lki, TRIM(ltxt(lki))
!
!   Precision and/or Termination Accuracy
!
        Efmt =  rki                              ! variabilize rki
    if( Efmt ==  1_ik ) Eform = "ES17.8E4 "      ! E-Format kind = 1
    if( Efmt ==  2_ik ) Eform = "ES25.16E4"      ! E-Format kind = 2
    if( Efmt ==  3_ik ) Eform = "ES28.19E4"      ! E-Format kind = 3
    txt = "( t15, a, /, t20, a, " // Eform // ", / )"
!
    write( *, txt ) "Precision and/or Termination Accuracy:", &
      &             "      eps            = ", eps
!
!   Definition of deviations (error)
!
    h = typerr                                   ! variabilize typerr
         if( h == 1_ik ) then
             txt = " 1: deviation      := true value - measured value"
    else if( h == 2_ik ) then
             txt = " 2: absolute error := ABS(true value - measured value)"
    else if( h == 3_ik ) then
             txt = " 3: relative error := absolute error / true value"
    else
             txt = "?: wrong definition - output not reliable"
    end if
!
    write( *, '( t15, a, /, t20, a, a, / )' )  &
      &    "Definition of deviations (error):",  &
      &    "      typerr         = ", TRIM(txt)
!
!   Continue with return
!
    write( *, '( 105("-"),                     /, &
      &          "Continue with <RETURN> ..."     )' )
    read(*,*)
!
    end subroutine Startinfo
!
!---------------------------------------------------------------------
!
!   Algorithms
!     Generates for a function a list with text of avail. algorithms
!   Reference:
!     ./.
!   Remark:
!     ./.
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    subroutine Algorithms( func, w, m )
!
    use kinds, only : ik
!
    implicit none
!
!   interface
     integer  (kind=ik),intent(in   ) :: func    ! number of function
     character(len =56),intent(  out) :: w(1:)   ! text   of algorithm
     integer  (kind=ik),intent(  out) :: m       ! number of algorithm
!   end interface
!
!
    if( func == 1_ik ) then                      ! y = erf(x)
    w(  1 )='101  Intr  erf: Intrinsic           GFortran/Silv.(2025)'
    w(  2 )='102  High  erf: Coeff. A1n,A2n      Strecok       (1968)'
    w(  3 )='103  High  erf: Rational Chebyshev  Cody          (1969)'
    w(  4 )='104  High  erf: Chebyshev Approx.   Schonfelder   (1978)'
    w(  5 )='105  High  erf: Slatec exDERF       Fullerton     (1993)'
    w(  6 )='106  High  erf: Various Approx.     Sun           (1995)'
    w(  7 )='107  High  erf: Polynomials r*8a    Ooura         (1996)'
    w(  8 )='108  High  erf: Polynomials r*8b    Ooura         (1996)'
    w(  9 )='109  High  erf: Polynomials r*16    Ooura         (1996)'
    w( 10 )='110  High  erf: Mill Ratio          Dia           (2023)'
    w( 11 )='111  Low   erf: Alg. 7.1.26-7.1.28  Abram./Stegun (1972)'
    w( 12 )='112  Low   erf: One-Line Approx.    Menzel        (1975)'
    w( 13 )='113  Low   erf: TANH and/or ATAN    Ved./Vaz./San.(1987)'
    w( 14 )='114  Low   erf: Rational Approx.    Winitzki a.o. (2008)'
    w( 15 )='115  Low   erf: Pade Approx.        Bercu/Howard  (2020)'
    w( 16 )='116  Expa  erf: SE,  symp. Expan.   Hoering       (2024)'
    w( 17 )='117  Expa  erf: SE, asymp. Expan.   Hoering       (2024)'
    w( 18 )='118  CFra  erf: CF, Cont. Fraction  Hoering       (2024)'
    m= 18_ik
!
    else if( func == 2_ik ) then                 ! y = erfc(x)
    w(  1 )='201  Intr  erfc: Intrinsic          GFortran/Silv.(2025)'
    w(  2 )='202  High  erfc: Rat. Chebyshev     Cody          (1969)'
    w(  3 )='203  High  erfc: Chebyshev Approx.  Schonfelder   (1978)'
    w(  4 )='204  High  erfc: Chebyshev Approx.  Shepherd      (1981)'
    w(  5 )='205  High  erfc: Slatec exDERFC     Fullerton     (1993)'
    w(  6 )='206  High  erfc: Various Approx.    Sun           (1995)'
    w(  7 )='207  High  erfc: Polynomials r*8a   Ooura         (1996)'
    w(  8 )='208  High  erfc: Polynomials r*8b   Ooura         (1996)'
    w(  9 )='209  High  erfc: Polynomials r*16   Ooura         (1996)'
    w( 10 )='210  High  erfc: Mill Ratio         Dia           (2023)'
    w( 11 )='211  Expa  erfc: SE,  symp. Expan.  Hoering       (2024)'
    w( 12 )='212  Expa  erfc: SE, asymp. Expan.  Hoering       (2024)'
    w( 13 )='213  CF    erfc: CF, Cont. Fraction Hoering       (2024)'
    m= 13_ik
!
    else if( func == 3_ik ) then                 ! y = erfcx(x)
    w(  1 )='301  Intr  erfcx: Intrinsic         GFortran/Silv.(2025)'
    w(  2 )='302  High  erfcx: Rat. Chebyshev    Cody          (1969)'
    w(  3 )='303  High  erfcx: Chebyshev Expan.  Schonfelder   (1978)'
    w(  4 )='304  High  erfcx: Chebyshev Appr.   Shepherd/Laf. (1981)'
    w(  5 )='305  High  erfcx: Slatec Cheby Ser. Fullerton     (1993)'
    w(  6 )='306  High  erfcx: Polynomials r*8a  Ooura         (1996)'
    w(  7 )='307  High  erfcx: Polynomials r*8b  Ooura         (1996)'
    w(  8 )='308  High  erfcx: Polynomials r*16  Ooura         (1996)'
    w(  9 )='309  High  erfcx: Var. Approx.      Johnson/Wuttke(2013)'
    w( 10 )='310  High  erfcx: Var. Approx.      Zaghloul      (2024)'
    w( 11 )='311  High  erfcx: Var. Approx. opt. Zaghloul      (2024)'
    w( 12 )='312  High  erfcx: Var. Approx.      Hoering       (2024)'
    w( 13 )='313  Low   erfcx: Adj. Expression   Ren/MacKenzie (2007)'
    w( 14 )='314  Expa  erfcx: SE,  symp. Expan. Hoering       (2024)'
    w( 15 )='315  Expa  erfcx: SE, asymp. Expan. Hoering       (2024)'
    w( 16 )='316  CF    erfcx: CF, Cont.Fraction Hoering       (2024)'
    m= 16_ik
!
    else if( func == 4_ik ) then                 ! x = erfinv(y)
    w(  1 )='xxx  High  erfinv: in Volume II     Author        (yyyy)'
    m=  1_ik
!
    else if( func == 5_ik ) then                 ! x = erfcinv(y)
    w(  1 )='xxx  High  erfcinv: in Volume II    Author        (yyyy)'
    m=  1_ik
!
    else if( func == 6_ik ) then                 ! x = erfcxinv(y)
    w(  1 )='xxx  High  erfcxinv: in Volume II   Author        (yyyy)'
    m=  1_ik
!
    else;   m = 0_ik;   end if                   ! wrong input
!
!   the last character strings is always the same
    w( m+1_ik ) = "                                           " //  &
     &            "EXIT submenue"
!
!   number elements to choose
    m = m + 1_ik                                 ! number function + 1
!
    return
    end subroutine Algorithms
!
!--------------------------------------------------------------------------
!
!   Menu
!     Menu for the selection of an error function
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    subroutine Menu( func, w, m, alg )
!
    use kinds, only : ik, lk
!
    implicit none
!
!   interface
     integer  (kind=ik),intent(  out)  :: func   ! selected  function
     character(len =56),intent(inout)  :: w(1:)  ! text   of algorithm
     integer  (kind=ik),intent(  out)  :: m      ! number of algorithm
     integer  (kind=ik),intent(  out)  :: alg    ! selected  algorithm
!   end interface
!
!   local variables
    logical  (kind=lk)               :: xmenu    ! T/F overall menu
    logical  (kind=lk)               :: mainmenu = .true.
    logical  (kind=lk)               :: submenu  = .false.
!
    save mainmenu, submenu
!
!
    xmenu = .true.                               ! switch menu on
    do while( xmenu )                            ! do until xmenu
!
!      Main Menu
       do while ( mainmenu )                     ! do until main menu
          write( *, 101 )                        ! format -> end sub
          read ( *, *   ) func                   ! read func number
          if( func == 0 ) func = 7               ! 0 is also exit
          if( func >= 1  .and.  func <= 7 ) then ! 1:7 selection
              if( func < 7 ) then                ! 1:6 erf selected
                  call Algorithms( func, w, m      ) ! fill algorithms
                  call MenuErf   ( func, w, m, alg ) ! select erf alg
                  if( alg /= m ) then            ! no exit selected
                      mainmenu = .false.         ! no main menu
                      submenu  = .true.          ! sub menu selected
                      call ScrClear( )           ! clear screen
                      return                     ! return result
                  else                           ! exit selected
                      xmenu    = .true.          ! xmenu still on
                      mainmenu = .true.          ! go to main menu
                      submenu  = .false.         ! finish sub menu
                  end if                         ! end alg selection
               else                              ! Main exit selected
!                  func     = 7                  ! 7, exit program
                   xmenu    = .false.            ! xmenu off
                   mainmenu = .false.            ! main menu off
                   submenu  = .false.            ! sub menu off
               end if                            ! end func selection
           end if                                ! end func valid
       end do                                    ! end do main menu
!
!      Sub Menue
       do while( submenu )                       ! do unitl sub menu
          call MenuErf( func, w, m, alg )        ! select erf algor.
          if( alg /= m ) then                    ! no exit selected
              xmenu    = .false.                 ! finish xmenu
              mainmenu = .false.                 ! main menu off
              submenu  = .true.                  ! next is sub menu
              call ScrClear( )                   ! clear screen
              return                             ! return result
          else                                   ! leave for main menu
              xmenu    = .true.                  ! xmenu is on
              mainmenu = .true.                  ! goto main menu
              submenu  = .false.                 ! sub menu off
          end if                                 ! end alg selection
       end do                                    ! end do sub menu
!
    end do                                       ! end do xmenu
    call ScrClear( )                             ! clear screen
!
101 format(                                                60(/), &
     t2,  72('='),                                            //, &
     t15, 'Error Function Menue by Thomas Hoering',           //, &
     t2,  72('='),                                            //, &
     t3,  'y = erf     ( x ) =',                                  &
     t52,                              'Error Function    1',  /, &
     t3,  'y = erfc    ( x ) =',                                  &
     t38,                'Complementary Error Function    2',  /, &
     t3,  'y = erfcx   ( x ) =',                                  &
     t31,         'Scaled Complementary Error Function    3', //, &
     t3,  'x = erfinv  ( y ) =',                                  &
     t23, 'Inverse                      Error Function    4',  /, &
     t3,  'x = erfcinv ( y ) =',                                  &
     t23, 'Inverse        Complementary Error Function    5',  /, &
     t3,  'x = erfcxinv( y ) =',                                  &
     t23, 'Inverse Scaled Complementary Error Function    6', //, &
     t54,                                'EXIT program    7', //, &
     t2,  72('='),                                             /, &
     t2,  'Your input - choose 1,...,7 : '                        )
!
    return
    end subroutine Menu
!
!---------------------------------------------------------------------
!
!   MenuErf
!     Menu error function for the selection of an algorithm
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    subroutine MenuErf( func, w, m, alg )
!
    use kinds, only : ik, lk
!
    implicit none
!
!   interface
     integer  (kind=ik),intent(in   )  :: func   ! selected  function
     character(len =56),intent(in   )  :: w(1:)  ! text   of algorithm
     integer  (kind=ik),intent(in   )  :: m      ! number of algorithm
     integer  (kind=ik),intent(  out)  :: alg    ! selected  algorithm
!   end interface
!
!   local variable
    character(len =16)               :: t1       ! function text short
    character(len =48)               :: t2       ! function text long
    logical  (kind=lk)               :: ok       ! T/F check
    integer  (kind=ik)               :: i, j     ! do ...
!
!
!   fix text
         if( func == 1 ) then
             t1 = 'y = erf(x)'
             t2 = 'Error Function'
    else if( func == 2 ) then
             t1 = 'y = erfc(x)'
             t2 = 'Complementary Error Function'
    else if( func == 3 ) then
             t1 = 'y = erfcx(x)'
             t2 = 'Scaled Complementary Error Function'
    else if( func == 4 ) then
             t1 = 'x = erfinv(y)'
             t2 = 'Inverse Error Function'
    else if( func == 5 ) then
             t1 = 'x = erfcinv(y)'
             t2 = 'Inverse Complementary Error Function'
    else if( func == 6 ) then
             t1 = 'x = erfcxinv(y)'
             t2 = 'Inverse Scaled Complementary Error Function'
    end  if
!
    ok = .false.
    do while ( .not. ok )
       write( *, 102 ) TRIM(t1), TRIM(t2)
102    format(                                                60(/), &
        t2,  70('='),                                            //, &
        t3,  'Menue for Algorithms:',                             /, &
        t3,  a, ' : ', a,                                        //, &
        t2,  70('='),                                             /, &
        t3,  'No.  Type  Description              ',                 &
             'Reference      Year       Choose'                   /, &
        t2,  70('-'),                                             /  )
!
       j = 0
       do i = 1,m-1,1                           ! war m-2
          write( *, 103 ) w(i),  i
103       format( t3, a,                     t60, ' .... ', i3       )
          j = j + 1
          if( j == 5 ) then; write( *,'( " " )' ); j = 0; end if
       end do
!
       write( *, 104 ) w(m), m
104    format(                                                   //, &
        t3,  a,                              t60, ' .... ', i3       )
!
       write( *, 105 ) m
105    format(                                                    /, &
        t2,  70('='),                                             /, &
        t2,  'Your input - choose 1,...,', i2, ' : '                 )
!
       read( *,* ) alg
       if( alg == 0 ) alg = m                    ! 0 is also exit
       if( alg >= 1  .and.  alg <= m ) ok = .true.
    end do
!
    return
    end subroutine MenuErf
!
!---------------------------------------------------------------------
!
!   Out1
!     Output headline for a single function with a single algorithms
!     and all available data points for the function
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    subroutine Out1( func, algtxt )
!
    use kinds, only : ik
    use const, only : typerr
!
    implicit none
!
!   interface
     integer  (kind=ik),intent(in)   :: func     ! function number
     character(len = *),intent(in)   :: algtxt   ! algorithm text
!   end interface
!
!   local variable
    character(len =48)               :: t1       ! function text long
    character(len =16)               :: t2       ! function text short
    character(len = 1)               :: t3       ! "x" or "y"
    character(len =31),dimension(1:3):: t4   =   &
           (/ 'Deviation from tabulated result', &
              'Absolute Deviation from table  ', &
              'Relative Deviation from table  ' /)
!
!
!   fix texts
!
         if( func == 1 ) then
             t1 = 'Error Function with'
             t2 = ' y = erf(x)'
             t3 = 'x'
    else if( func == 2 ) then
             t1 = 'Complementary Error Function with'
             t2 = ' y = erfc(x)'
             t3 = 'x'
    else if( func == 3 ) then
             t1 = 'Scaled Complementary Error Function with'
             t2 = ' y = erfcx(x)'
             t3 = 'x'
    else if( func == 4 ) then
             t1 = 'Inverse Error Function with '
             t2 = ' x = erfinv(y)'
             t3 = 'y'
    else if( func == 5 ) then
             t1 = 'Inverse Complementary Error Function with'
             t2 = ' x = erfcinv(y)'
             t3 = 'y'
    else if( func == 6 ) then
             t1 = 'Inverse Scaled Complementary Error Function with'
             t2 = ' x = erfcxinv(y)'
             t3 = 'y'
    end  if
!
    write( *, 1 ) TRIM(t1), TRIM(t2), algtxt(1:LEN(algtxt)),   &
                       t3,  TRIM(t2), TRIM(t2), TRIM(t4(typerr))
  1 format(                                                         &
     t2,   137('='),                                             /, &
                                                 t2,   a, a,    //, &
     t40,  'No.',                                                   &
     t45,  'Type',                                                  &
     t51,  'Description',                                           &
     t76,  'Reference',                                             &
     t91,  'Year',                                               /, &
                                                 t40,  a,        /, &
     t2,   137('=')                                              /, &
     t2,   'No.',                                                   &
     t7,   'Input ',                                   a,           &
     t37,  'Result tabulated',                         a,           &
     t67,  'Result Algorithm',                         a,           &
     t97,                                              a,           &
     t130, 'Iteration',                                          /, &
     t2,   137('-')                                                 )
!
    return
    end subroutine Out1
!
!---------------------------------------------------------------------
!
!   Out2
!     Output values for a single function with a single algorithms
!     and all available data points for the function
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    subroutine Out2( point, inputxy, tabulyx, fvalue, iteration )
!
    use kinds, only : ik, rki, rk
    use const, only : zero, typerr
!
    implicit none
!
!   interface
     integer(kind=ik),intent( in ) :: point      ! data point number
     real   (kind=rk),intent( in ) :: inputxy    ! input x resp. y
     real   (kind=rk),intent( in ) :: tabulyx    ! tabul y resp. x
     real   (kind=rk),intent( in ) :: fvalue     ! f(x) resp. f(y)
     integer(kind=ik),intent( in ) :: iteration  ! num. of iteration
!   end interface
!
!   local variable
    real     (kind=rk)             :: deviation  ! deviation   typerr
    integer  (kind=ik)             :: h          ! variabilize typerr
!
!   E Format description
    integer  (kind=ik)             :: Efmt       ! kind for format
    character(len = 9)             :: Eform      ! format of figures
    character(len =71)             :: formx      ! entire format
!
!
!   calculation of the deviation, according typerr
    h = typerr
         if( h == 1_ik ) then
             deviation = tabulyx - fvalue
    else if( h == 2_ik ) then
             deviation = ABS( tabulyx - fvalue )
    else if( h == 3_ik ) then
             if( ABS(fvalue) <= zero .and. ABS(fvalue) >= zero ) then
                 deviation = zero
             else
                 deviation = ABS( tabulyx - fvalue ) / tabulyx
             end if
    else
             deviation = HUGE( fvalue )          ! error in typerr
    end if
!
!   format layout for real figures
        Efmt =  rki                              ! take "kind" over
    if( Efmt ==  1_ik ) Eform = "ES17.8E4 "      ! E-Format kind = 1
    if( Efmt ==  2_ik ) Eform = "ES25.16E4"      ! E-Format kind = 2
    if( Efmt ==  3_ik ) Eform = "ES28.19E4"      ! E-Format kind = 3
!
    formx = "(t2,i3,t7," // Eform // ",t37," // Eform //  &
      &     ",t67,"      // Eform // ",t97," // Eform //  &
      &     ",t132,i4 )"
!
    write( *, formx ) point,  inputxy,   tabulyx,   &
      &               fvalue, deviation, iteration
!
    return
    end subroutine Out2
!
!---------------------------------------------------------------------
!
!   ScrClear
!     Output (screen) of 65 lines (clear screen)
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    subroutine ScrClear(  )
!
    implicit none
!
!
    write( *, '(  65(/)  )' )
!
    return
    end subroutine ScrClear
!
!---------------------------------------------------------------------
!
!   ScrLine
!     Output (screen) of one line, consists of 125 times "-"
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    subroutine ScrLine( ch )
!
    implicit none
!
!   interface
     character(len = *), intent(in)  :: ch       ! "S" oder "L"
!   end interface
!
!
         if( ch == "s" .or. ch == "S" ) then
             write( *, '( t2, 137("-") )' )
    else if( ch == "l" .or. ch == "L" ) then
             write( *, '( t2, 153("-") )' )
    else
             write( *, '( t2, "wrong input parameter ScrBreak" )' )
    end  if
!
    return
    end subroutine ScrLine
!
!---------------------------------------------------------------------
!
!   ScrBreak
!     Output (screen) of a line and "return" request
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    subroutine ScrBreak( ch )
!
    implicit none
!
!   interface
     character(len = *), intent(in)  :: ch       ! "S" oder "L"
!   end interface
!
!
         if( ch == "s" .or. ch == "S" ) then
             write( *, '( t2, 137("-"),                     /, &
               &          t2, "Continue with <RETURN> ..."     )' )
             read( *,* )
!
    else if( ch == "l" .or. ch == "L" ) then
             write( *, '( t2, 153("-"),                     /, &
               &          t2, "Continue with <RETURN> ..."     )' )
             read( *,* )
!
    else
             write( *, '( t2, "wrong input parameter ScrBreak" )' )
    end  if
!
    return
    end subroutine ScrBreak
!
!---------------------------------------------------------------------
!