!
!   Remark:
!   1. ./.
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   Error Function (erf): High Precision
!
!   real(rk) function erfIntrinsic    ( x )       result( f )  GFortran/Silverfrost (2023),  Intrinsic ERF
!   real(rk) function erfStrecok      ( x )       result( f )  Strecok              (1968),  Coefficients
!   real(rk) function erfCody         ( x )       result( f )  Cody                 (1969),  Rat. Chebyshev Approximation
!   real(rk) function erfSchonfelder  ( x )       result( f )  Schonfelder          (1978),  Chebyshev Approximation
!   real(rk) function erfSlatec       ( x )       result( f )  Fullerton            (1993),  Chebyshev series, Slatec exDERF
!   real(rk) function erfSun          ( x )       result( f )  Sun Microsystems     (1993),  Various Polynomials
!   real(rk) function erfOoura8a      ( x )       result( f )  Oouara               (1996),  Polynomials degree 12 and 13
!   real(rk) function erfOoura8b      ( x )       result( f )  Oouara               (1996),  Polynomial 
!   real(rk) function erfOoura16      ( x )       result( f )  Oouara               (1996),  Polynomial - quadruple precision
!   real(rk) function erfDia          ( x )       result( f )  Dia                  (2023),  Mill Ratio
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   erfIntrinsic
!     Error Function f = erf(x) with intrinsic function referenced
!     GFortran or Silverfrost FTN95
!   Reference:
!     GFortran (GNU Fortran) version 6.3.0-1, part of GCC (GNU Compiler
!      Collection). “class: elemental function ERF”, 2024, http://gcc.gnu.org
!     Silverfrost FORTRAN 95, “class: elemental function ERF”, 2024, 
!      https://www.silverfrost.com/ftn95-help/intrin/erf.aspx
!   Remark:
!     For Silverfrost FTN95 this intrinsic function is not implemented for
!      extended precision and therefore set to "zero".
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfIntrinsic( x )   result( f )
!
    use kinds, only : compiler, ik, rki, rk
    use const, only : zero
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    integer(ik)                 :: h1            ! = variable "compiler"
    integer(ik)                 :: h2            ! = variable "kind-i"
!
!
    h1 = compiler                                ! variabilize compiler
!
!   Compiler = 1: GFortran part
         if( h1 == 1_ik ) then                   ! GFortran
             f = ERF( x )                        ! always intrinsic defined
!
!   Compiler = 2: Silverfrost FTN95 part
    else if( h1 == 2_ik ) then                   ! Silverfrost ftn 95
             h2 = rki                            ! variabilize rki
             if( h2 <= 2_ik ) then               ! kind = 1 or 2
                 f = ERF( x )                    ! intrinsic
             else
!                Silverfrost is not able to handle kind=3 => f = 0.0
                 f = zero                        ! f = zero
             end if
    end  if
!
    return
    end function erfIntrinsic
!
!-----------------------------------------------------------------------------
!
!   erfStrecok
!     Error Function f = erf(x) with coefficients referenced Strecok (1968)
!   Reference:
!     Anthony J. Strecok, "On the Calculation of the Inverse of the Error
!     Function", Mathematics of Computation, Jan 1968, Volume 22, Issue 101,
!     Page 144-158, https://doi.org/10.1090/S0025-5718-1968-0223070-2
!   Remark:
!     Realisation of formula (2) on page 145 and with the coefficients
!      A1n and A2n on page 146.
!     Parameter "sqrteps" is defined in "Constants" for erf,
!      where erf( |x| ≤ sqrteps )  := +2.00/SQRT(pi)*x
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfStrecok( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, two, five,  &
                      pi, twodivsqrtpi,      &
                      sqrteps, limiterf
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: C, S, a       ! help variable
    real   (rk)                 :: p             ! a**(n-1)
    integer(ik)                 :: n             ! do ...
!
!   coefficients for calculating erf(x) from formula (2) on page 146
    integer(ik), parameter      :: dim           = 19_ik
    real   (rk), parameter      :: A1n(1_ik:dim) =                      (/ &
     0.7032250027437753888217370E+0_rk, 0.3305015219166062231414836E+0_rk, &
     0.2013397472647062732206869E+0_rk, 0.1086302450227406906873823E+0_rk, &
     0.0467755234324848608231226E+0_rk, 0.0153985726157101996635576E+0_rk, &
     0.0038015076798529871151801E+0_rk, 0.0006971837924080287357368E+0_rk, &
     0.0000944909268810454980951E+0_rk, 0.0000094328116983836678903E+0_rk, &
     0.0000006919275203251400557E+0_rk, 0.0000000372253249369107964E+0_rk, &
     0.0000000014666061423380014E+0_rk, 0.0000000000422616144318049E+0_rk, &
     0.0000000000008897865267233E+0_rk, 0.0000000000000136760444757E+0_rk, &
     0.0000000000000001533423425E+0_rk, 0.0000000000000000012536751E+0_rk, &
     0.0000000000000000000074517E+0_rk                                    /)
    real   (rk), parameter      :: A2n(1_ik:dim) =                      (/ &
     0.2472551681400521469013060E+0_rk, 0.1442272263615747171484869E+0_rk, &
     0.0869894549959345538475537E+0_rk, 0.0439773381940833680821082E+0_rk, &
     0.0172439625886622621167913E+0_rk, 0.0050790696122025702812471E+0_rk, &
     0.0011086064534234065370328E+0_rk, 0.0001782280162548616973045E+0_rk, &
     0.0000210404583073251381878E+0_rk, 0.0000018206631636434076704E+0_rk, &
     0.0000001153309909443694132E+0_rk, 0.0000000053427502760308268E+0_rk, &
     0.0000000001808485878095127E+0_rk, 0.0000000000044696822924881E+0_rk, &
     0.0000000000000806068838945E+0_rk, 0.0000000000000010601364636E+0_rk, &
     0.0000000000000000101649277E+0_rk, 0.0000000000000000000710005E+0_rk, &
     0.0000000000000000000000000E+0_rk                                    /)
!
!
    z = ABS( x )                                 ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = twodivsqrtpi * x                ! 2.0/√pi * x
!
!   sqrteps < z=|x| ≤ limiterf
    else if( z <= limiterf ) then
             C = COS( two * z / five )           ! p. 144
             S = SIN( two * z / five )           ! p. 145
             a = two * ( C*C ) - one             ! p. 144
             f = zero                            ! function value
             p = one                             ! a**0 = 1
             do n = 1_ik, dim, 1_ik              ! do 1,...,19
!               p. 145, formula (2) with coefficients A1n and A2n from p. 146
                f = f + (A1n(n) + two*C * A2n(n)) * p
                p = p * a                        ! a**n for next iter
             end do
             f = (two * z)/(five * pi) + (S * f) ! finalize f. (2) on p. 145
             if( x < zero ) f = -f               ! erf(-x)=-erf(x)
!
!   limiterf < z=|x| ≤ +∞
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfStrecok
!
!-----------------------------------------------------------------------------
!
!   erfCody
!     Error Function f = erf(x) with Rational Chebyshev Approximations
!     referenced Cody (1969)
!   Reference:
!     W.J. Cody, "Rational Chebyshev Approximations for the Error Function",
!       Mathematics of Computation, 1969, Volume 23, Issue 107, Pages 631-637,
!       https://doi.org/10.1090/S0025-5718-1969-0247736-4
!     W.J. Cody, "SUBROUTINE CALERF(ARG,RESULT,JINT)", 1990, Latest modific.
!       19. March 1990, a fortran subroutine for error and scaled and unscaled
!       complementary error functions at http://www.netlib.org/specfun/erf
!   Remark:
!     Parameter "xsmall" is now defined 'EPSILON(x)/two' for erf,
!      where erf( z=|x| ≤ xsmall ) := 2/√pi * x
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfCody( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, half, one, two, four, sixteen, &
                      onedivsqrtpi, twodivsqrtpi, limiterf
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: y             ! help variable
    real   (rk)                 :: xnum          ! sum numerator
    real   (rk)                 :: xden          ! sum denominator
    integer(ik)                 :: i             ! do...
!
!   Cody mathematical & machine-dependent constants/parameters
    real   (rk), parameter      :: xsmall        = EPSILON(one)/two
    real   (rk), parameter      :: thresh        = 0.46875E+00_rk
!                                  thresh       := erf(x) == erfc(x)
!
!   coefficients erf(x), p. 633 for xsmall < |x| ≤ (thresh = 0.46875)
    real   (rk), parameter      :: p1(0_ik:4_ik) =                (/ &
     3.209377589138469472562E+03_rk, 3.774852376853020208137E+02_rk, &
     1.138641541510501556495E+02_rk, 3.161123743870565596947E+00_rk, &
     1.857777061846031526730E-01_rk                                 /)
    real   (rk), parameter      :: q1(0_ik:4_ik) =                (/ &
     2.844236833439170622273E+03_rk, 1.282616526077372275645E+03_rk, &
     2.440246379344441733056E+02_rk, 2.360129095234412093499E+01_rk, &
     1.000000000000000000000E+00_rk                                 /)
!
!   coefficients erfc(x), p. 635 for (thresh = 0.46875) < |x| ≤ (four = 4.0)
    real   (rk), parameter      :: p2(0_ik:8_ik) =                (/ &
     1.23033935479799725272E+03_rk,  2.05107837782607146533E+03_rk,  &
     1.71204761263407058314E+03_rk,  8.81952221241769090411E+02_rk,  &
     2.98635138197400131132E+02_rk,  6.61191906371416294775E+01_rk,  &
     8.88314979438837594118E+00_rk,  5.64188496988670089180E-01_rk,  &
     2.15311535474403846343E-08_rk                                  /)
    real   (rk), parameter      :: q2(0_ik:8_ik) =                (/ &
     1.23033935480374942043E+03_rk,  3.43936767414372163696E+03_rk,  &
     4.36261909014324715820E+03_rk,  3.29079923573345962678E+03_rk,  &
     1.62138957456669018874E+03_rk,  5.37181101862009857509E+02_rk,  &
     1.17693950891312499305E+02_rk,  1.57449261107098347253E+01_rk,  &
     1.00000000000000000000E+00_rk                                  /)
!
!   coefficients erfc(x), p. 636 for (four = 4.0) < |x| < (six = 6.0)
    real   (rk), parameter      :: p3(0_ik:5_ik) =                (/ &
     6.58749161529837803157E-04_rk,  1.60837851487422766278E-02_rk,  &
     1.25781726111229246204E-01_rk,  3.60344899949804439429E-01_rk,  &
     3.05326634961232344035E-01_rk,  1.63153871373020978498E-02_rk  /)
    real   (rk), parameter      :: q3(0_ik:5_ik) =                (/ &
     2.33520497626869185443E-03_rk,  6.05183413124413191178E-02_rk,  &
     5.27905102951428412248E-01_rk,  1.87295284992346042097E+00_rk,  &
     2.56852019228982242072E+00_rk,  1.00000000000000000000E+00_rk  /)
!
!
    z = ABS( x )                                 ! absolute x
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( xsmall = EPSILON(x) / two )
         if( z <= xsmall ) then
!            f = x * p1(0_ik) / q1(0_ik)         ! 2/√pi * x
             f = twodivsqrtpi * x                ! faster by Hoering
!
!   xsmall < z=|x| ≤ (thresh = 0.46875E+00)
    else if( z <= thresh ) then
             y    = z * z                        ! 1st formula p. 631
             xnum = p1(4_ik) * y
             xden = q1(4_ik) * y
             do i = 3_ik, 1_ik, -1_ik
                xnum = (xnum + p1(i)) * y
                xden = (xden + q1(i)) * y
             end do
             f = x * (xnum + p1(0_ik)) / (xden + q1(0_ik))
!
!   (thresh = 0.46875E+00) < z=|x| ≤ (four = 4.00E+00)
    else if( z <= four ) then
             xnum = p2(8_ik) * z                 ! 2nd formula p. 631
             xden = q2(8_ik) * z
             do i = 7_ik, 1_ik, -1_ik
                xnum = (xnum + p2(i)) * z
                xden = (xden + q2(i)) * z
             end do
             f = (xnum + p2(0_ik)) / (xden + q2(0_ik))
             y = INT( z * sixteen ) / sixteen    ! erf specific
             f = EXP( -y*y ) * EXP( -(z-y)*(z+y) ) * f
             if( x < zero ) then                 ! x is negative
                 f = (f - half) - half           ! erf(x) = erfc(x) -1
             else                                ! x is positive
                 f = (half - f) + half           ! erf(x) = 1 -erfc(x)
             end if                              ! end if for x
!
!   (four= 4.00E+00) < z=|x| ≤ limiterf
    else if( z <= limiterf ) then
             y    = one / (z * z)                ! 3rd formula p. 631
             xnum = p3(5_ik) * y
             xden = q3(5_ik) * y
             do i = 4_ik, 1_ik, -1_ik
                xnum = (xnum + p3(i)) * y
                xden = (xden + q3(i)) * y
             end do
             f = y * (xnum + p3(0_ik)) / (xden + q3(0_ik))
             y = INT( z * sixteen ) / sixteen
             f = EXP(-y*y) * EXP(-(z-y)*(z+y)) * ((onedivsqrtpi-f)/z)
             if( x < zero ) then                 ! x is negative
                 f = (f - half) - half           ! erf(x) = erfc(x) -1
             else                                ! x is positive
                 f = (half - f) + half           ! erf(x) = 1 -erfc(x)
             end if                              ! end if for x
!
!   limiterf < z=|x| ≤ +∞
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfCody
!
!-----------------------------------------------------------------------------
!
!   erfSchonfelder
!     Error Function f = erf(x) with Chebyshev Expansions referenced
!     Schonfelder (1978)
!   Reference:
!     J.L. Schonfelder, "Chebyshev Expansions for the Error and Related
!     Functions", Mathematics of Computation, 1978, Volume 32, Issue 144,
!     Pages 1232-1240, https://doi.org/10.1090/S0025-5718-1978-0494846-8,
!     here: page 1233 (formula), page 1234 (table 1 and table 2)
!   Remark:
!     Two Chebyhev Approximations for erf(x), including back transformation
!      to erf(x) and the extension for negative input of x.
!     Parameter "sqrteps" is defined in "Constants" for erf,
!      where erf( |x| ≤ sqrteps )  := +2.00/SQRT(pi)*x
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfSchonfelder( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, two, eight,  &
                      twodivsqrtpi,           &
                      sqrteps, limiterf
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! erf(x)
!   end interface
    real   (rk)                 :: f             ! function value
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: zz            ! (z=|x|)²
    real   (rk)                 :: t, u          ! var. Chebyshev
    real   (rk)                 :: sv, d, dd     ! var. Clenshaw
    integer(ik)                 :: j             ! do ...
!
!   parameter for y = erf(x)/x and t = 2*(x/a)**2 -1,
!   with a = 2 on page 1234, table 1: valid for 0 ≤ x < 2
    real   (rk), parameter      :: a             =   2.000E+00_rk
    integer(ik), parameter      :: m1            =  28_ik
    real   (rk), parameter      :: c1(1_ik:m1)   =                                  (/ &
    +1.483110564084803581889448079057E+00_rk, -3.01071073386594942470731046311E-01_rk, &
      +6.8994830689831566246603180718E-02_rk,  -1.3916271264722187682546525687E-02_rk, &
       +2.420799522433463662891678239E-03_rk,    -3.65863968584808644649382577E-04_rk, &
         +4.8620984432319048282887568E-05_rk,      -5.749256558035684835054215E-06_rk, &
           +6.11324357843476469706758E-07_rk,        -5.8991015312958434390846E-08_rk, &
             +5.207009092068648240455E-09_rk,          -4.23297587996554326810E-10_rk, &
               +3.1881135066491749748E-11_rk,            -2.236155018832684273E-12_rk, &
                 +1.46732984799108492E-13_rk,               -9.044001985381747E-15_rk, &
                    +5.25481371547092E-16_rk,                 -2.8874261222849E-17_rk, &
                      +1.504785187558E-18_rk,                    -7.4572892821E-20_rk, &
                         +3.522563810E-21_rk,                      -1.58944644E-22_rk, &
                            +6.864365E-24_rk,                         -2.84257E-25_rk, &
                              +1.1306E-26_rk,                            -4.33E-28_rk, &
                                 +1.6E-29_rk,                             -1.0E-30_rk /)
!
!   parameter for y = x*exp(x²)*(1-erf(x)) and t = (k-x²)/(k-8+x²),
!   with k = 10.5 on page 1234, table 2: valid for 2 ≤ x ≤ +∞
    real   (rk), parameter      :: k             =  10.500E+00_rk
    integer(ik), parameter      :: m2            =  44_ik
    real   (rk), parameter      :: c2(1_ik:m2)   =                                 (/  &
    +1.077977852072383151168335910348E+00_rk,  -2.6559890409148673372146500904E-02_rk, &
       -1.487073146698099509605046333E-03_rk,    -1.38040145414143859607708920E-04_rk, &
         -1.1280303332287491498507366E-05_rk,      -1.172869842743725224053739E-06_rk, &
           -1.03476150393304615537382E-07_rk,        -1.1899114085892438254447E-08_rk, &
             -1.016222544989498640476E-09_rk,          -1.37895716146965692169E-10_rk, &
                -9.369613033737303335E-12_rk,            -1.918809583959525349E-12_rk, &
                  -3.7573017201993707E-14_rk,              -3.7053726026983357E-14_rk, &
                   +2.627565423490371E-15_rk,               -1.121322876437933E-15_rk, &
                    +1.84136028922538E-16_rk,                 -4.9130256574886E-17_rk, &
                     +1.0704455167373E-17_rk,                  -2.671893662405E-18_rk, &
                       +6.49326867976E-19_rk,                   -1.65399353183E-19_rk, &
                        +4.2605626604E-20_rk,                    -1.1255840765E-20_rk, &
                         +3.025617448E-21_rk,                      -8.29042146E-22_rk, &
                          +2.31049558E-22_rk,                       -6.5469511E-23_rk, &
                           +1.8842314E-23_rk,                        -5.504341E-24_rk, &
                            +1.630950E-24_rk,                         -4.89860E-25_rk, &
                             +1.49054E-25_rk,                          -4.5922E-26_rk, &
                              +1.4318E-26_rk,                           -4.516E-27_rk, &
                               +1.440E-27_rk,                            -4.64E-28_rk, &
                                +1.51E-28_rk,                             -5.0E-29_rk, &
                                 +1.7E-29_rk,                             -6.0E-30_rk, &
                                 +2.0E-30_rk,                             -1.0E-30_rk /)
!
!
    z = ABS( x )                                 ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = twodivsqrtpi * x                ! 2.0/√pi * x
!
!   sqrteps < |x| < (two = 2.00E+00)
    else if( z < two ) then
!            Chebyshew Approximation, formula low on page 1233
             t  = two * ((z/a) * (z/a)) -one     ! Chebyshev t
             u  = two * t
             d  = zero
             dd = zero
             do j = m1, 2_ik, -1_ik              ! Clenshaw's recurr.
                sv = d; d = u*d - dd + c1(j); dd = sv
             end do
             f = t*d - dd + c1(1_ik)/two         ! finalize Chebyshew
             f = z * f                           ! finalize erf(z)
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
!   (two = 2.00E+00) ≤ z=|x| ≤ limiterf
    else if( z <= limiterf ) then
!            Chebyshew Approximation, formula top on page 1235
             zz = z * z                          ! z² = |x|²
             t  = (k - zz) / (k - eight + zz)    ! Chebyshev t
             u  = two * t
             d  = zero
             dd = zero
             do j = m2, 2_ik, -1_ik              ! Clenshaw's recurr.
                sv = d; d = u*d - dd + c2(j); dd = sv
             end do
             f = t*d - dd + c2(1_ik)/two         ! finalize Chebyshew
             f = one - f / ( z * EXP(zz) )       ! finalize erf(z)
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
!   limiterf < z=|x| ≤ +∞
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfSchonfelder
!
!-----------------------------------------------------------------------------
!
!   erfSlatec
!     Error Function f = erf(x) with Chebyshev series referenced Slatec (1993)
!   Reference:
!     Chebyshev series "c1, c2 and c3" and other coefficients. SLATEC Common 
!      Mathematical Library, Version 4.1, July 1993: A comprehensive software
!      library containing over 1400 general purpose mathematical and statist.
!      routines written in Fortran 77. Energy Science and Technology Software
!      Center, U.S. Department of Energy. The Library is in the public domain
!      and distributed by the Energy Science and Technology Software Center.
!      Energy Science and Technology Software Center, P.O. Box 1020 Oak Ridge,
!      TN 37831, Telephone 615-576-2606, 
!      E-mail: estsc%a1.adonis.mrouter@zeus.osti.gov . Quoted from
!      http://www.netlib.org/slatec/guide, Section 4. "Obtaining the Library"
!      http://www.netlib.org/slatec/ , Referenced function: "DERF" by
!      the author Fullerton
!     Evaluation of n-term Chebyshev series
!      R. Broucke, "Ten subroutines for the manipulation of Chebyshev series,
!      Algorithm 446", Communications of the ACM, Association for Computer
!      Machinery, 1973, Volume 16, Issue 4, Page 254-256,
!      https://doi.org/10.1145/362003.362037
!      here: recurrence formula (4) on page 254f
!   Remark:
!     Parameter "sqrteps" is defined in "Constants" for erf,
!      where erf( |x| ≤ sqrteps )  := +2.00/SQRT(pi)*x
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfSlatec( x )   result( f )
!
    use kinds, only : compiler, ik, rki, rkx, rk
    use const, only : zero, half, one, two, three, five, eight, &
                      twodivsqrtpi, sqrteps, limiterf
!
    implicit none
!
!   interface
     real  (rk),  intent(in )   :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: zz            ! (z=|x|)²
    real   (rk)                 :: t,  u         ! Chebyshev coefficients
    real   (rk)                 :: b0, b1, b2    ! help variable Broucke
    integer(ik)                 :: i             ! do ...
!
!   Chebyshev series coefficients (former "erfcs") for the 
!   interval 0.00E+00 to 1.00E+00, with weighted error 1.28...E-32
    integer(ik), parameter      :: d1            = 21_ik
    real   (rk), parameter      :: c1(1_ik:d1)   =                                       (/ &
    -0.49046121234691808039984544033376E-01_rk, -0.14226120510371364237824741899631E+00_rk, &
    +0.10035582187599795575754676712933E-01_rk, -0.57687646997674847650827025509167E-03_rk, &
    +0.27419931252196061034422160791471E-04_rk, -0.11043175507344507604135381295905E-05_rk, &
    +0.38488755420345036949961311498174E-07_rk, -0.11808582533875466969631751801581E-08_rk, &
    +0.32334215826050909646402930953354E-10_rk, -0.79910159470045487581607374708595E-12_rk, &
    +0.17990725113961455611967245486634E-13_rk, -0.37186354878186926382316828209493E-15_rk, &
    +0.71035990037142529711689908394666E-17_rk, -0.12612455119155225832495424853333E-18_rk, &
    +0.20916406941769294369170500266666E-20_rk, -0.32539731029314072982364160000000E-22_rk, &
    +0.47668672097976748332373333333333E-24_rk, -0.65980120782851343155199999999999E-26_rk, &
    +0.86550114699637626197333333333333E-28_rk, -0.10788925177498064213333333333333E-29_rk, &
    +0.12811883993017002666666666666666E-31_rk                                             /)
!
!   Chebyshev series coefficients (former "erc2cs") for the
!   interval 2.50E-01 to 1.00E+00, with weighted error 2.67...E-32
    integer(ik), parameter      :: d2            = 49_ik
    real   (rk), parameter      :: c2(1_ik:d2)   =                                       (/ &
     -0.6960134660230950112739150826197E-01_rk,  -0.4110133936262089348982212084666E-01_rk, &
     +0.3914495866689626881561143705244E-02_rk,  -0.4906395650548979161280935450774E-03_rk, &
     +0.7157479001377036380760894141825E-04_rk,  -0.1153071634131232833808232847912E-04_rk, &
     +0.1994670590201997635052314867709E-05_rk,  -0.3642666471599222873936118430711E-06_rk, &
     +0.6944372610005012589931277214633E-07_rk,  -0.1371220902104366019534605141210E-07_rk, &
     +0.2788389661007137131963860348087E-08_rk,  -0.5814164724331161551864791050316E-09_rk, &
     +0.1238920491752753181180168817950E-09_rk,  -0.2690639145306743432390424937889E-10_rk, &
     +0.5942614350847910982444709683840E-11_rk,  -0.1332386735758119579287754420570E-11_rk, &
     +0.3028046806177132017173697243304E-12_rk,  -0.6966648814941032588795867588954E-13_rk, &
     +0.1620854541053922969812893227628E-13_rk,  -0.3809934465250491999876913057729E-14_rk, &
     +0.9040487815978831149368971012975E-15_rk,  -0.2164006195089607347809812047003E-15_rk, &
     +0.5222102233995854984607980244172E-16_rk,  -0.1269729602364555336372415527780E-16_rk, &
     +0.3109145504276197583836227412951E-17_rk,  -0.7663762920320385524009566714811E-18_rk, &
     +0.1900819251362745202536929733290E-18_rk,  -0.4742207279069039545225655999965E-19_rk, &
     +0.1189649200076528382880683078451E-19_rk,  -0.3000035590325780256845271313066E-20_rk, &
     +0.7602993453043246173019385277098E-21_rk,  -0.1935909447606872881569811049130E-21_rk, &
     +0.4951399124773337881000042386773E-22_rk,  -0.1271807481336371879608621989888E-22_rk, &
     +0.3280049600469513043315841652053E-23_rk,  -0.8492320176822896568924792422399E-24_rk, &
     +0.2206917892807560223519879987199E-24_rk,  -0.5755617245696528498312819507199E-25_rk, &
     +0.1506191533639234250354144051199E-25_rk,  -0.3954502959018796953104285695999E-26_rk, &
     +0.1041529704151500979984645051733E-26_rk,  -0.2751487795278765079450178901333E-27_rk, &
     +0.7290058205497557408997703680000E-28_rk,  -0.1936939645915947804077501098666E-28_rk, &
     +0.5160357112051487298370054826666E-29_rk,  -0.1378419322193094099389644800000E-29_rk, &
     +0.3691326793107069042251093333333E-30_rk,  -0.9909389590624365420653226666666E-31_rk, &
     +0.2666491705195388413323946666666E-31_rk                                             /)
!
!   Chebyshev series coefficients (former "erfccs") for the
!   interval 0.00E+00 to 2.50E-01, with weighted error 1.53...E-31
    integer(ik), parameter      :: d3            = 59_ik
    real   (rk), parameter      :: c3(1_ik:d3)   =                                       (/ &
      +0.715179310202924774503697709496E-01_rk,   -0.265324343376067157558893386681E-01_rk, &
      +0.171115397792085588332699194606E-02_rk,   -0.163751663458517884163746404749E-03_rk, &
      +0.198712935005520364995974806758E-04_rk,   -0.284371241276655508750175183152E-05_rk, &
      +0.460616130896313036969379968464E-06_rk,   -0.822775302587920842057766536366E-07_rk, &
      +0.159214187277090112989358340826E-07_rk,   -0.329507136225284321486631665072E-08_rk, &
      +0.722343976040055546581261153890E-09_rk,   -0.166485581339872959344695966886E-09_rk, &
      +0.401039258823766482077671768814E-10_rk,   -0.100481621442573113272170176283E-10_rk, &
      +0.260827591330033380859341009439E-11_rk,   -0.699111056040402486557697812476E-12_rk, &
      +0.192949233326170708624205749803E-12_rk,   -0.547013118875433106490125085271E-13_rk, &
      +0.158966330976269744839084032762E-13_rk,   -0.472689398019755483920369584290E-14_rk, &
      +0.143587337678498478672873997840E-14_rk,   -0.444951056181735839417250062829E-15_rk, &
      +0.140481088476823343737305537466E-15_rk,   -0.451381838776421089625963281623E-16_rk, &
      +0.147452154104513307787018713262E-16_rk,   -0.489262140694577615436841552532E-17_rk, &
      +0.164761214141064673895301522827E-17_rk,   -0.562681717632940809299928521323E-18_rk, &
      +0.194744338223207851429197867821E-18_rk,   -0.682630564294842072956664144723E-19_rk, &
      +0.242198888729864924018301125438E-19_rk,   -0.869341413350307042563800861857E-20_rk, &
      +0.315518034622808557122363401262E-20_rk,   -0.115737232404960874261239486742E-20_rk, &
      +0.428894716160565394623737097442E-21_rk,   -0.160503074205761685005737770964E-21_rk, &
      +0.606329875745380264495069923027E-22_rk,   -0.231140425169795849098840801367E-22_rk, &
      +0.888877854066188552554702955697E-23_rk,   -0.344726057665137652230718495566E-23_rk, &
      +0.134786546020696506827582774181E-23_rk,   -0.531179407112502173645873201807E-24_rk, &
      +0.210934105861978316828954734537E-24_rk,   -0.843836558792378911598133256738E-25_rk, &
      +0.339998252494520890627359576337E-25_rk,   -0.137945238807324209002238377110E-25_rk, &
      +0.563449031183325261513392634811E-26_rk,   -0.231649043447706544823427752700E-26_rk, &
      +0.958446284460181015263158381226E-27_rk,   -0.399072288033010972624224850193E-27_rk, &
      +0.167212922594447736017228709669E-27_rk,   -0.704599152276601385638803782587E-28_rk, &
      +0.297976840286420635412357989444E-28_rk,   -0.126252246646061929722422632994E-28_rk, &
      +0.539543870454248793985299653154E-29_rk,   -0.238099288253145918675346190062E-29_rk, &
      +0.109905283010276157359726683750E-29_rk,   -0.486771374164496572732518677435E-30_rk, &
      +0.152587726411035756763200828211E-30_rk                                             /)
!
!   coefficients n1, n2 and n3 to achieve the required precision according kind specification
    integer(ik),parameter :: pn1(3,2) = reshape( (/13,13,21, 13,13,15/), shape(pn1) )
    integer(ik),parameter ::  n1      = pn1( rki, compiler )
    integer(ik),parameter :: pn2(3,2) = reshape( (/25,25,49, 25,25,30/), shape(pn2) )
    integer(ik),parameter ::  n2      = pn2( rki, compiler )
    integer(ik),parameter :: pn3(3,2) = reshape( (/26,26,59, 26,26,33/), shape(pn3) )
    integer(ik),parameter ::  n3      = pn3( rki, compiler )
!
!ac alternative calculation of n1, n2 and n3. Calculated via eta and backward
!ac summation of the coefficients, until the desired precision is achieved.
!ac Attention: eta is declared as single precision (sp, REAL32 or real*4).
!ac
!a  logical(lk)                 :: first         = .true.
!a  integer(ik), parameter      :: r4            = rkx(1_ik) ! kind for real*4
!a  real   (r4), parameter      :: eta           = 0.10E+00_r4*EPSILON(x)/two
!ac                                eta, sp:      = 5.96046456635690446..E-0009
!ac                                eta, dp:      = 1.11022302462515660..E-0017
!ac                                eta, ep:      = 5.42101086242752216..E-0021
!ac                                eta, qp:      = 9.62964986542914201..E-0036
!a  real   (rk)                 :: err
!a  integer(ik)                 :: n1, n2, n3
!a  save first, n1, n2, n3
!a  if( first ) then
!ac     for n1:
!a      err = zero
!a      do i = d1,1_ik,-1_ik
!a         err = err+ABS(REAL(c1(i))); if(err>eta) exit; end do; n1=i
!ac     for n2:
!a      err = zero
!a      do i = d2,1_ik,-1_ik
!a         err = err+ABS(REAL(c2(i))); if(err>eta) exit; end do; n2=i
!ac     for n3:
!a      err = zero
!a      do i = d3,1_ik,-1_ik
!a         err = err+ABS(REAL(c3(i))); if(err>eta) exit; end do; n3=i
!a      first = .false.
!a  end if
!ac end alterative calculation of n1, n2 and n3
!
!
    z = ABS( x )                                 ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
!            f = two * x / sqrtpi                ! original
             f = twodivsqrtpi * x                ! faster
!
!   sqrteps ≤ z=|x| ≤ (one = 1.00E+00)
    else if( z <= one ) then
             t  = two * x*x - one; u  = two * t  ! n1-term Chebyshev series
             b0 = zero;            b1 = zero
             do i = n1, 1_ik, -1_ik
                b2 = b1; b1 = b0; b0 = u*b1 - b2 + c1(i)
             end do
             f = half * (b0 - b2)                ! finalize Broucke
             f = x * (one + f)                   ! finalize calculation
!
!   (one = 1.00E+00) < z=|x| ≤ (two = 2.00E+00)
    else if( z <= two ) then
             zz = z * z                          ! (z=|x|)²
             t  = (eight/zz - five)/three        ! n2-term Chebyshev series
             u  = two * t
             b0 = zero;            b1 = zero
             do i = n2, 1_ik, -1_ik
                b2 = b1; b1 = b0; b0 = u*b1 - b2 + c2(i)
             end do
             f = half * (b0 - b2)                ! finalize Broucke
             f = one - EXP(-zz)/z * (half + f )  ! finalize calculation
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
!   (two = 2.00E+00) < z=|x| < limiterf
    else if( z <= limiterf ) then
             zz = z * z                          ! (z=|x|)²
             t  = eight/zz - one;  u  = two * t  ! n3-term Chebyshev series
             b0 = zero;            b1 = zero
             do i = n3, 1_ik, -1_ik
                b2 = b1; b1 = b0; b0 = u*b1 - b2 + c3(i)
             end do
             f = half * (b0 - b2)                ! finalize Broucke
             f = one - EXP(-zz)/z * (half + f )  ! finalize calculation
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
!   limiterf < z=|x| ≤ +∞
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfSlatec
!
!-----------------------------------------------------------------------------
!
!   erfSun
!     Error Function f = erf(x) with several Polynomials referenced 
!     Sun Microsystems (1993)
!   Reference:
!     Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
!     Developed at SunSoft, a Sun Microsystems, Inc. business. Permission to
!     use, copy, modify, and distribute this software is freely granted,
!     provided that this notice is preserved.
!     Original "C" routines at http://netlib.org/fdlibm/s_erf.c
!     @(#)s_erf.c 1.3 95/01/18
!   Remark:
!     Parameter "smal" is defined in the Sun function as "smal = 1.00E-300".
!      It is now redefined as "smal = TINY( x ) * ten**(four*rk)", with
!      kind = 1 (sp): +1.1754943508222875080...E-0034 (new)
!      kind = 2 (dp): +1.0000000000000000000...E-0300 (original Sun)
!               (dp): +2.2250738585072013827...E-0300 (new)
!      kind = 3 (ep): +3.3621031431120934968...E-4916 (new)
!               (qp): +3.3621031431120934968...E-4916 (new)
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!    last modification: 31. January 2025
!
    pure function erfSun( x )   result( f )
!
    use kinds, only : rk, rk2008
    use const, only : zero, half, one, ten, limiterf
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: u, y          ! help variable
    real   (rk)                 :: r, s, p, q    ! help variable
!
!   mathematical & machine-dependent constants/parameters
    real   (rk), parameter      :: loctiny     = TINY(x)
    real   (rk), parameter      :: smal        = TINY(x) * (ten ** rk2008)
!
!   coefficient for  approximation to erf in [0.00E+000, dp:1.00E-300]
    real   (rk), parameter      :: efx8        = 1.02703333676410069053E+00_rk
!                                              =  (16 / √π) - 8 = efx * 8
!
!   coefficient for  approximation to erf in [dp:1.00E-300, 2**(-28)]
    real   (rk), parameter      :: efx         = 1.28379167095512586316E-01_rk
!                                              =  ( 2 / √π) - 1
!
!   coefficients for  approximation to erf in [2**(-28), 0.84375=9.0/16.0]
    real   (rk), parameter      ::                                           &
    pp0 =  1.28379167095512558561E-1_rk,pp1 = -3.25042107247001499370E-1_rk, &
    pp2 = -2.84817495755985104766E-2_rk,pp3 = -5.77027029648944159157E-3_rk, &
    pp4 = -2.37630166566501626084E-5_rk,                                     &
    qq1 =  3.97917223959155352819E-1_rk,qq2 =  6.50222499887672944485E-2_rk, &
    qq3 =  5.08130628187576562776E-3_rk,qq4 =  1.32494738004321644526E-4_rk, &
    qq5 = -3.96022827877536812320E-6_rk
!
!   coefficients for approximation to erf in [0.84375=9.0/16., 1.25=5.0/4.0]
    real   (rk), parameter      ::      erx =  8.45062911510467529297E-1_rk
!                               =  0.84506291151 rounded to single (24 bits)
    real   (rk), parameter      ::                                           &
    pa0 = -2.36211856075265944077E-3_rk,pa1 =  4.14856118683748331666E-1_rk, &
    pa2 = -3.72207876035701323847E-1_rk,pa3 =  3.18346619901161753674E-1_rk, &
    pa4 = -1.10894694282396677476E-1_rk,pa5 =  3.54783043256182359371E-2_rk, &
    pa6 = -2.16637559486879084300E-3_rk,                                     &
    qa1 =  1.06420880400844228286E-1_rk,qa2 =  5.40397917702171048937E-1_rk, &
    qa3 =  7.18286544141962662868E-2_rk,qa4 =  1.26171219808761642112E-1_rk, &
    qa5 =  1.36370839120290507362E-2_rk,qa6 =  1.19844998467991074170E-2_rk
!
!   coefficients for approximation to erfc in [1.25=5.0/4.0, 1.00/0.35]
    real   (rk), parameter      ::                                           &
    ra0 = -9.86494403484714822705E-3_rk,ra1 = -6.93858572707181764372E-1_rk, &
    ra2 = -1.05586262253232909814E+1_rk,ra3 = -6.23753324503260060396E+1_rk, &
    ra4 = -1.62396669462573470355E+2_rk,ra5 = -1.84605092906711035994E+2_rk, &
    ra6 = -8.12874355063065934246E+1_rk,ra7 = -9.81432934416914548592E+0_rk, &
    sa1 =  1.96512716674392571292E+1_rk,sa2 =  1.37657754143519042600E+2_rk, &
    sa3 =  4.34565877475229228821E+2_rk,sa4 =  6.45387271733267880336E+2_rk, &
    sa5 =  4.29008140027567833386E+2_rk,sa6 =  1.08635005541779435134E+2_rk, &
    sa7 =  6.57024977031928170135E+0_rk,sa8 = -6.04244152148580987438E-2_rk
!
!   coefficients for approximation to erfc in [1.00/0.35, limiterf]
    real   (rk), parameter      ::                                           &
    rb0 = -9.86494292470009928597E-3_rk,rb1 = -7.99283237680523006574E-1_rk, &
    rb2 = -1.77579549177547519889E+1_rk,rb3 = -1.60636384855821916062E+2_rk, &
    rb4 = -6.37566443368389627722E+2_rk,rb5 = -1.02509513161107724954E+3_rk, &
    rb6 = -4.83519191608651397019E+2_rk,                                     &
    sb1 =  3.03380607434824582924E+1_rk,sb2 =  3.25792512996573918826E+2_rk, &
    sb3 =  1.53672958608443695994E+3_rk,sb4 =  3.19985821950859553908E+3_rk, &
    sb5 =  2.55305040643316442583E+3_rk,sb6 =  4.74528541206955367215E+2_rk, &
    sb7 = -2.24409524465858183362E+1_rk
!
!
    z = ABS( x )                                 ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| < ( smal= [TINY(x)*(ten**(four*rk))] )
         if( z < smal ) then                     ! avoid underflow
             f = 0.125E+00_rk * ( 8.0E0_rk*x + efx8*x ) ! original
!              = multiplied out: 2.0 / SQRT(pi) * x     ! equivalent

!
!   smal ≤ z=|x| < (2**(-28) = 3.72529...E-09)
    else if( z < 3.7252902984619140626E-0009_rk ) then
             f = x + efx*x                              ! original
!              = multiplied out: 2.0 / SQRT(pi) * x     ! equivalent
!
!   (2**(-28) = 3.72529...E-09) ≤ z=|x| < (0.84375E+00 = 27.0/32.0)
    else if( z < 0.84375E+00_rk ) then
             y = x * x
             r = pp0 + y*(pp1 + y*(pp2 + y*(pp3 +      &
                       y* pp4                        )))
             s = one + y*(qq1 + y*(qq2 + y*(qq3 +      &
                       y*(qq4 + y* qq5)              )))
             f = x + x * r/s
!
!   (0.84375E+00 = 27.0/32.0) ≤ z=|x| < (1.25E+00 = 5.0/4.0)
    else if( z < 1.25E+00_rk ) then
             s = z - one
             p = pa0 + s*(pa1 + s*(pa2 + s*(pa3 +      &
                       s*(pa4 + s*(pa5 + s* pa6    )))))
             q = one + s*(qa1 + s*(qa2 + s*(qa3 +      &
                       s*(qa4 + s*(qa5 + s* qa6)    ))))
             f = erx + p/q                      ! finalize erf(x)
             if( x < zero ) f = -f              ! erf(-x) = -erf(x)
!
!   (1.25E+00 = 5.0/4.0) ≤ z=|x| < (1.0/0.35 = 2.857142857...E+00)
    else if( z < 2.857142857E+00_rk ) then
             s = one /(z * z)
             r = ra0 + s*(ra1 + s*(ra2 + s*(ra3 +      &
                       s*(ra4 + s*(ra5 + s*(ra6 +      &
                       s* ra7                     ))))))
             s = one + s*(sa1 + s*(sa2 + s*(sa3 +      &
                       s*(sa4 + s*(sa5 + s*(sa6 +      &
                       s*(sa7 + s* sa8)           ))))))
             u = REAL( z )
             r = EXP(-u*u - 0.5625E+00_rk) * EXP((u-z)*(u+z) + r/s)
!                           0.5625E+00 = 9.0 / 16.0
!            finalize erfc(x) and conversion erfc(x) to neg/pos erf(x)
             if( x < zero ) then                 ! x is negative
                 f = ( r/z - half ) - half       ! erf(x) = erfc(x) - 1.0
             else                                ! x is positive
                 f = ( half - r/z ) + half       ! erf(x) = 1.0 - erfc(x)
             end if                              ! end if for x
!a           f = one - r/z; if(x < zero) f = -f  ! f and erf(-x) = -erf(x)
!
!   1.0/0.35 ≤ z=|x| ≤ limiterf
    else if( z <= limiterf ) then
             s = one /(z * z)
             r = rb0 + s*(rb1 + s*(rb2 + s*(rb3 +      &
                       s*(rb4 + s*(rb5 + s* rb6    )))))
             s = one + s*(sb1 + s*(sb2 + s*(sb3 +      &
                       s*(sb4 + s*(sb5 + s*(sb6 +      &
                       s* sb7                     ))))))
             u = REAL( z )
             r = EXP(-u*u - 0.5625E+00_rk) * EXP((u-z)*(u+z) + r/s)
!                           0.5625E+00 = 9.0 / 16.0
!            finalize erfc(x) and conversion erfc(x) to neg/pos erf(x)
             if( x < zero ) then                 ! x is negative
                 f = ( r/z - half ) - half       ! erf(x) = erfc(x) - 1.0
             else                                ! x is positive
                 f = ( half - r/z ) + half       ! erf(x) = 1.0 - erfc(x)
             end if                              ! end if for x
!a           f = one - r/z; if(x < zero) f = -f  ! f and erf(-x) = -erf(x)
!
!   limiterf < z=|x| ≤ +∞
    else
             f = one - loctiny                   ! f ≈ 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfSun
!
!-----------------------------------------------------------------------------
!
!   erfOoura8a
!     Error Function f = erf(x) with Rational Function referenced Ooura (1996)
!   Reference:
!     Masatake Mori, "A Method for Evaluation of the Error Function of Real
!      and Complex Variable with High Relative Accuracy", Publications of the
!      Research Institute for Mathematical Sciences (PRIMS), Kyoto University
!      1983, Volume 19, Issue 3, Page 1081–1094,
!      https://doi.org/10.2977/PRIMS/1195182021
!     Copyright(C) 1996 Takuya Ooura, email:ooura@mmm.t.u-tokyo.ac.jp
!      You may use, copy, modify this code for any purpose and without
!      fee. You may distribute this ORIGINAL package.
!      Internet: http://www.kurims.kyoto-u.ac.jp/~ooura/gamerf.html
!      here: referenced file derf.f, function derf
!   Remark:
!     Parameter "sqrteps" is defined in "Constants" for erf,
!      where erf( |x| ≤ sqrteps )  := +2.00/SQRT(pi)*x
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfOoura8a( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, twodivsqrtpi, sqrteps, limiterf
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: t             ! help variable
    integer(ik)                 :: k             ! help variable
    integer(ik)                 :: i             ! do ...
!
!   parameters for the rational functions: 0.0 < z=|x| < 2.2
    real   (rk), parameter      :: a(0_ik:64_ik) =                 (/ &
      0.00000000005958930743E+00_rk, -0.00000000113739022964E+00_rk,  &
      0.00000001466005199839E+00_rk, -0.00000016350354461960E+00_rk,  &
      0.00000164610044809620E+00_rk, -0.00001492559551950604E+00_rk,  &
      0.00012055331122299265E+00_rk, -0.00085483269811296660E+00_rk,  &
      0.00522397762482322257E+00_rk, -0.02686617064507733420E+00_rk,  &
      0.11283791670954881569E+00_rk, -0.37612638903183748117E+00_rk,  &
      1.12837916709551257377E+00_rk,  0.00000000002372510631E+00_rk,  &
     -0.00000000045493253732E+00_rk,  0.00000000590362766598E+00_rk,  &
     -0.00000006642090827576E+00_rk,  0.00000067595634268133E+00_rk,  &
     -0.00000621188515924000E+00_rk,  0.00005103883009709690E+00_rk,  &
     -0.00037015410692956173E+00_rk,  0.00233307631218880978E+00_rk,  &
     -0.01254988477182192210E+00_rk,  0.05657061146827041994E+00_rk,  &
     -0.21379664776456006580E+00_rk,  0.84270079294971486929E+00_rk,  &
      0.00000000000949905026E+00_rk, -0.00000000018310229805E+00_rk,  &
      0.00000000239463074000E+00_rk, -0.00000002721444369609E+00_rk,  &
      0.00000028045522331686E+00_rk, -0.00000261830022482897E+00_rk,  &
      0.00002195455056768781E+00_rk, -0.00016358986921372656E+00_rk,  &
      0.00107052153564110318E+00_rk, -0.00608284718113590151E+00_rk,  &
      0.02986978465246258244E+00_rk, -0.13055593046562267625E+00_rk,  &
      0.67493323603965504676E+00_rk,  0.00000000000382722073E+00_rk,  &
     -0.00000000007421598602E+00_rk,  0.00000000097930574080E+00_rk,  &
     -0.00000001126008898854E+00_rk,  0.00000011775134830784E+00_rk,  &
     -0.00000111992758382650E+00_rk,  0.00000962023443095201E+00_rk,  &
     -0.00007404402135070773E+00_rk,  0.00050689993654144881E+00_rk,  &
     -0.00307553051439272889E+00_rk,  0.01668977892553165586E+00_rk,  &
     -0.08548534594781312114E+00_rk,  0.56909076642393639985E+00_rk,  &
      0.00000000000155296588E+00_rk, -0.00000000003032205868E+00_rk,  &
      0.00000000040424830707E+00_rk, -0.00000000471135111493E+00_rk,  &
      0.00000005011915876293E+00_rk, -0.00000048722516178974E+00_rk,  &
      0.00000430683284629395E+00_rk, -0.00003445026145385764E+00_rk,  &
      0.00024879276133931664E+00_rk, -0.00162940941748079288E+00_rk,  &
      0.00988786373932350462E+00_rk, -0.05962426839442303805E+00_rk,  &
      0.49766113250947636708E+00_rk                                  /)
!
!   parameters for the rational functions: 2.2 ≤ z=|x| < 6.9
    real   (rk), parameter      :: b(0_ik:64_ik) =                (/  &
     -0.00000000029734388465E+00_rk,  0.00000000269776334046E+00_rk,  &
     -0.00000000640788827665E+00_rk, -0.00000001667820132100E+00_rk,  &
     -0.00000021854388148686E+00_rk,  0.00000266246030457984E+00_rk,  &
      0.00001612722157047886E+00_rk, -0.00025616361025506629E+00_rk,  &
      0.00015380842432375365E+00_rk,  0.00815533022524927908E+00_rk,  &
     -0.01402283663896319337E+00_rk, -0.19746892495383021487E+00_rk,  &
      0.71511720328842845913E+00_rk, -0.00000000001951073787E+00_rk,  &
     -0.00000000032302692214E+00_rk,  0.00000000522461866919E+00_rk,  &
      0.00000000342940918551E+00_rk, -0.00000035772874310272E+00_rk,  &
      0.00000019999935792654E+00_rk,  0.00002687044575042908E+00_rk,  &
     -0.00011843240273775776E+00_rk, -0.00080991728956032271E+00_rk,  & 
      0.00661062970502241174E+00_rk,  0.00909530922354827295E+00_rk,  &
     -0.20160072778491013140E+00_rk,  0.51169696718727644908E+00_rk,  &
      0.00000000003147682272E+00_rk, -0.00000000048465972408E+00_rk,  &
      0.00000000063675740242E+00_rk,  0.00000003377623323271E+00_rk,  &
     -0.00000015451139637086E+00_rk, -0.00000203340624738438E+00_rk,  &
      0.00001947204525295057E+00_rk,  0.00002854147231653228E+00_rk,  &
     -0.00101565063152200272E+00_rk,  0.00271187003520095655E+00_rk,  &
      0.02328095035422810727E+00_rk, -0.16725021123116877197E+00_rk,  &
      0.32490054966649436974E+00_rk,  0.00000000002319363370E+00_rk,  &
     -0.00000000006303206648E+00_rk, -0.00000000264888267434E+00_rk,  &
      0.00000002050708040581E+00_rk,  0.00000011371857327578E+00_rk,  &
     -0.00000211211337219663E+00_rk,  0.00000368797328322935E+00_rk,  &
      0.00009823686253424796E+00_rk, -0.00065860243990455368E+00_rk,  &
     -0.00075285814895230877E+00_rk,  0.02585434424202960464E+00_rk,  &
     -0.11637092784486193258E+00_rk,  0.18267336775296612024E+00_rk,  &
     -0.00000000000367789363E+00_rk,  0.00000000020876046746E+00_rk,  &
     -0.00000000193319027226E+00_rk, -0.00000000435953392472E+00_rk,  &
      0.00000018006992266137E+00_rk, -0.00000078441223763969E+00_rk,  &
     -0.00000675407647949153E+00_rk,  0.00008428418334440096E+00_rk,  &
     -0.00017604388937031815E+00_rk, -0.00239729611435071610E+00_rk,  &
      0.02064129023876022970E+00_rk, -0.06905562880005864105E+00_rk,  &
      0.09084526782065478489E+00_rk                                  /)
!
!
    z = ABS( x )                                 ! z = |x|
!
!   0.0 ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = twodivsqrtpi * x                ! 2.0/√pi * x
!
!   sqrteps < z=|x| < 2.2
    else if( z < 2.20E+00_rk ) then
             t = z * z
             k =     INT ( t, kind=ik )
             t = t - REAL( k, kind=rk )
             k = k * 13_ik
             f = zero
             do i = 0_ik, 11_ik, 1_ik
                f = (f + a(k+i)) * t
             end do
             f = (f + a(k+12_ik)) * z
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
!   2.2 ≤ z=|x| ≤ limiterf
    else if( z <= limiterf ) then
             k =     INT ( z, kind=ik )
             t = z - REAL( k, kind=rk )
             k = (k - 2_ik) * 13_ik
             f = zero
             do i = 0_ik, 11_ik, 1_ik
                f = (f + b(k+i)) * t
             end do
             f = f + b(k+12_ik)
!
             f =       f*f*f*f
             f = one - f*f*f*f
! or         f = one - f*f*f*f * f*f*f*f * f*f*f*f * f*f*f*f
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
!   limiterf < z=|x| ≤ +∞
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfOoura8a
!
!-----------------------------------------------------------------------------
!
!   erfOoura8b
!     Error Function f = erf(x) with Rational Function referenced Ooura (1996)
!   Reference:
!     Masatake Mori, "A Method for Evaluation of the Error Function of Real
!      and Complex Variable with High Relative Accuracy", Publications of the
!      Research Institute for Mathematical Sciences (PRIMS), Kyoto University
!      1983, Volume 19, Issue 3, Page 1081–1094,
!      https://doi.org/10.2977/PRIMS/1195182021
!     Copyright(C) 1996 Takuya Ooura, email:ooura@mmm.t.u-tokyo.ac.jp.
!      You may use, copy, modify this code for any purpose and without
!      fee. You may distribute this ORIGINAL package.
!      Internet: http://www.kurims.kyoto-u.ac.jp/~ooura/gamerf.html
!      here: referenced file gamerf2a.f, function derf and derfc
!   Remark:
!     Parameter "sqrteps" is defined in "Constants" for erf,
!      where erf( |x| ≤ sqrteps )  := +2.00/SQRT(pi)*x
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfOoura8b( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, half, one, two,  &
                      twodivsqrtpi, sqrteps, limiterf
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: y             ! y = z²
    integer(ik)                 :: i             ! do ...
!
!   Ooura parameters for f = erf(x) and x ≤ 0.125
    real   (rk), parameter      :: w(0_ik:5_ik)  =                (/ &
          1.12837916709551257E+00_rk,  -3.76126389031833602E-01_rk,  &
          1.12837916706621301E-01_rk,  -2.68661698447642378E-02_rk,  &
          5.22387877685618101E-03_rk,  -8.49202435186918470E-04_rk  /)
!
!   Ooura parameter for f = 1 - erfc(x) and x > 0.125
    real   (rk), parameter      :: pv = 1.26974899965115684E+01_rk,  &
                                   ph = 6.10399733098688199E+00_rk
    real   (rk), parameter      :: p(0_ik:7_ik)  =                (/ &
          2.96316885199227378E-01_rk,   1.81581125134637070E-01_rk,  &
          6.81866451424939493E-02_rk,   1.56907543161966709E-02_rk,  &
          2.21290116681517573E-03_rk,   1.91395813098742864E-04_rk,  &
          9.71013284010551623E-06_rk,   1.66642447174307753E-07_rk  /)
    real   (rk), parameter      :: q(0_ik:7_ik)  =                (/ &
          6.12158644495538758E-02_rk,   5.50942780056002085E-01_rk,  &
          1.53039662058770397E+00_rk,   2.99957952311300634E+00_rk,  &
          4.95867777128246701E+00_rk,   7.41471251099335407E+00_rk,  &
          1.04765104356545238E+01_rk,   1.48455557345597957E+01_rk  /)
!
!
    z = ABS( x )                                 ! z = |x|
!
!   0.000 ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = twodivsqrtpi * x                ! 2.0/√pi * x
!
!   sqrteps < z=|x| ≤ 0.125
    else if( z <= 0.125E+00_rk ) then
!            calculation of erf(x)
             y = x * x                           ! y = x²
             f = zero
             do i = 5_ik, 1_ik, -1_ik
                f = (f + w(i)) * y
             end do
             f = x * ( f + w(0_ik) )             ! finalize erf(x)
!
!   0.125 < z=|x| ≤ limiterf
    else if( z <= limiterf ) then
!            calculate erfc(z) and convert to erf(z)
             y = z * z                           ! y = z²
             f = zero
             do i = 0_ik, 7_ik, 1_ik
                f = f + p(i) / (y + q(i))
             end do
             f = EXP( -y ) * z * f
             if( z <  ph ) f = f + two / ( EXP(pv * z) + one )
!            conversion erfc(x) to erf(x) for negative and positive x
             if( x < zero ) then                 ! x is negative
                 f = ( f - half ) - half         ! erf(x) = erfc(x) - 1.0
             else                                ! x is positive
                 f = ( half - f ) + half         ! erf(x) = 1.0 - erfc(x)
             end if                              ! end if for x
!
!   limiterf < z=|x| ≤ +∞
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfOoura8b
!
!-----------------------------------------------------------------------------
!
!   erfOoura16
!     Error Function f = erf(x) with Rational Function referenced Ooura (1996)
!   Reference:
!     Masatake Mori, "A Method for Evaluation of the Error Function of Real
!      and Complex Variable with High Relative Accuracy", Publications of the
!      Research Institute for Mathematical Sciences (PRIMS), Kyoto University
!      1983, Volume 19, Issue 3, Page 1081–1094,
!      https://doi.org/10.2977/PRIMS/1195182021
!     Copyright(C) 1996 Takuya Ooura, email:ooura@mmm.t.u-tokyo.ac.jp
!      You may use, copy, modify this code for any purpose and without
!      fee. You may distribute this ORIGINAL package.
!      Internet: http://www.kurims.kyoto-u.ac.jp/~ooura/gamerf.html
!      here: referenced file gamerf2a.f, function qerf and qerfc
!   Remark:
!     This version is especially developed for quadruple precision!
!      Kind selection (see "kinds") should be "qp" to achieve good results.
!     Parameter "sqrteps" is defined in "Constants" for erf,
!      where erf( |x| ≤ sqrteps )  := +2.00/SQRT(pi)*x
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfOoura16( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, half, one, two,            &
                      twodivsqrtpi, sqrteps, limiterf
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: y             ! y = z²
    integer(ik)                 :: i             ! do ...
!
!   Ooura parameters for f = erf(x) and x ≤ 0.125
    real   (rk), parameter      :: w(0_ik:10_ik) =                                                (/ &
    +1.1283791670955125738961589031215451638E+00_rk,-3.7612638903183752463205296770705954644E-01_rk, &
    +1.1283791670955125738961588999935886077E-01_rk,-2.6866170645131251759432042514961432313E-02_rk, &
    +5.2239776254421878419519215904293747973E-03_rk,-8.5483270234508523548393236134016747257E-04_rk, &
    +1.2055332981788774687616425024649090688E-04_rk,-1.4925650357342418486127546759864765351E-05_rk, &
    +1.6462113548558693746378431523719715229E-06_rk,-1.6365454692461428822012524459668520250E-07_rk, &
    +1.4701966650809284594534032908988195691E-08_rk                                                 /)
!
!   Ooura parameter for f = 1 - erfc(x) and x > 0.125
    real   (rk), parameter      ::              pv =+1.8296570980424689847157930974106706835E+01_rk, &
                                                ph =+8.9588287394342176848213494031807385567E+00_rk
    real   (rk), parameter      :: p(0_ik:16_ik) =                                                (/ &
    +2.1226887418241545314975570224238841543E-01_rk,+1.6766968820663231170102487414107148110E-01_rk, &
    +1.0461429607758480243524362040994242137E-01_rk,+5.1557963860512142911764627378588661742E-02_rk, &
    +2.0070986488528139460346647533434778000E-02_rk,+6.1717726506718148117513762897928828534E-03_rk, &
    +1.4990611906920858646769185063310410160E-03_rk,+2.8760540416705806615617926157307107830E-04_rk, &
    +4.3585593590380741491013549969419946961E-05_rk,+5.2174364856655433775383935118049845471E-06_rk, &
    +4.9333351722974670085736982894474122277E-07_rk,+3.6846914376723888190666722894010079935E-08_rk, &
    +2.1729515092764086499231043367920037215E-09_rk,+9.9870022842895735663712411206346261651E-11_rk, &
    +3.1775163189596489863458236395414830880E-12_rk,+4.5657943993597540327708145643160878201E-14_rk, &
    +1.1940964427370412648558173558044106203E-16_rk                                                 /)
    real   (rk), parameter      :: q(0_ik:16_ik) =                                                (/ &
    +2.9482230394292049252878077330764031337E-02_rk,+2.6534007354862844327590269604581049764E-01_rk, &
    +7.3705575985730123132195272141160572532E-01_rk,+1.4446292893203104133929687855854497896E+00_rk, &
    +2.3880606619376559912235584857800710490E+00_rk,+3.5673498777093386979273977202889759348E+00_rk, &
    +4.9824969366355296879760903991854492762E+00_rk,+6.6335018387405633238409855625402006223E+00_rk, &
    +8.5203645862651289478197632097553870199E+00_rk,+1.0643085317662274170216548777166393329E+01_rk, &
    +1.3001669850030489723387515813223808078E+01_rk,+1.5596282517377690399267249728222735970E+01_rk, &
    +1.8429903207271748464995406180854691072E+01_rk,+2.1533907893494593530979123915138686107E+01_rk, &
    +2.5076752889217226137869837117288885077E+01_rk,+2.9515380437412601845256918753602002410E+01_rk, &
    +3.5792848810704122499184545805923520658E+01_rk                                                 /)
!
!
    z = ABS( x )                                 ! z = |x|
!
!   0.000 ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = twodivsqrtpi * x                ! 2.0/√pi * x
!
!   sqrteps < z=|x| ≤ 0.125
    else if( z <= 0.125E+00_rk ) then
!            calculation of erf(x)
             y = x * x                           ! y = x²
             f = zero
             do i = 10_ik, 1_ik, -1_ik
                f = (f + w(i)) * y
             end do
             f = x * ( f + w(0_ik) )             ! finalize erf(x)
!
!   0.125 < z=|x| ≤ limiterf
    else if( z <= limiterf ) then
!            calculation of erfc(z) and convert to erf(z)
             y = z * z                           ! y = x²
             f = zero
             do i = 0_ik, 16_ik, 1_ik
                f = f  +  p(i) / (y + q(i))
             end do
             f = EXP( -y ) * z * f
             if( z < ph ) f = f + two / (EXP(pv * z) + one)
!            conversion erfc(x) to erf(x) for negative and positive x
             if( x < zero ) then                 ! x is negative
                 f = ( f - half ) - half         ! erf(x) = erfc(x) - 1.0
             else                                ! x is positive
                 f = ( half - f ) + half         ! erf(x) = 1.0 - erfc(x)
             end if                              ! end if for x
!
!   limiterf < z=|x| ≤ +∞
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfOoura16
!
!-----------------------------------------------------------------------------
!
!   erfDia
!     Error Function f = erf(x) with Mills Ratio referenced Dia (2023)
!   Reference:
!     Yaya D. Dia, "Approximate Incomplete Integrals, Application to 
!     Complementary Error Function", 2023, June 21, paper, 15 pages,
!     available at SSRN: https://ssrn.com/abstract=4487559 or
!     https://doi.org/10.2139/ssrn.4487559
!   Remarks:
!     Parameter "sqrteps" is defined in "Constants" for erf,
!      where erf( |x| ≤ sqrteps )  := +2.00/SQRT(pi)*x
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfDia( x )   result ( f )
!
    use kinds, only : rk
    use const, only : zero, half, one, two,               &
                      sqrtpi, onedivsqrtpi, twodivsqrtpi, &
                      sqrteps, limiterf
!a  use const, only : twodivpi                   ! alternative only
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z,  zz        ! z=|x|, z*z
!a  real   (rk)                 :: z2, zz2       ! original z and zz
    real   (rk)                 :: M             ! Mills ratio (M)
!
!   calculation of M (Mills ratio), page 9, formula (69) with:
!   a) parameters for M on page 10 in table 1, beginning with "o..."
!   b) parameters eliminating "√2 * x" in the interface of M with modified
!      parameters, beginning with "m..."
    real   (rk), parameter      ::                                        &
      ob00 =  2.92678600515804815402E+00_rk,  mb00 = ob00 / SQRT( two ),  &
      ob11 =  8.97280659046817350354E+00_rk,  mb11 = ob11 /       two,    &
      ob12 = 10.27157061171363078863E+00_rk,  mb12 = ob12 /       two,    &
      ob13 = 12.72323261907760928036E+00_rk,  mb13 = ob13 /       two,    &
      ob14 = 16.88639562007936907786E+00_rk,  mb14 = ob14 /       two,    &
      ob15 = 24.12333774572479110372E+00_rk,  mb15 = ob15 /       two,    &
      ob21 =  5.81582518933527390512E+00_rk,  mb21 = ob21 / SQRT( two ),  &
      ob22 =  5.70347935898051436684E+00_rk,  mb22 = ob22 / SQRT( two ),  &
      ob23 =  5.51862483025707963145E+00_rk,  mb23 = ob23 / SQRT( two ),  &
      ob24 =  5.26184239579604207321E+00_rk,  mb24 = ob24 / SQRT( two ),  &
      ob25 =  4.92081346632882032881E+00_rk,  mb25 = ob25 / SQRT( two ),  &
      oc11 = 11.61511226260603247078E+00_rk,  mc11 = oc11 /       two,    &
      oc12 = 18.25323235347346524796E+00_rk,  mc12 = oc12 /       two,    &
      oc13 = 18.38871225773938486923E+00_rk,  mc13 = oc13 /       two,    &
      oc14 = 18.61193318971775795045E+00_rk,  mc14 = oc14 /       two,    &
      oc15 = 24.14804072812762821134E+00_rk,  mc15 = oc15 /       two,    &
      oc21 =  3.83362947800146179416E+00_rk,  mc21 = oc21 / SQRT( two ),  &
      oc22 =  7.30756258553673541139E+00_rk,  mc22 = oc22 / SQRT( two ),  &
      oc23 =  8.42742300458043240405E+00_rk,  mc23 = oc23 / SQRT( two ),  &
      oc24 =  5.66479518878470764762E+00_rk,  mc24 = oc24 / SQRT( two ),  &
      oc25 =  4.91396098895240075156E+00_rk,  mc25 = oc25 / SQRT( two )
!
!
    z = ABS( x )                                 ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = twodivsqrtpi * x                ! 2.0/√pi * x
!
!   sqrteps < z=|x| ≤ limiterf
    else if( z <= limiterf ) then
!            modified calculation M (Mills ratio), page 9, formula(69),
!            using z in the interface
             zz = z * z                         ! zz = z² = |x|²
             M  = (      onedivsqrtpi      / (      z + mb00      )   ) *  &
                  ( (zz  + mc21*z  + mc11) / (zz  + mb21*z  + mb11)   ) *  &
                  ( (zz  + mc22*z  + mc12) / (zz  + mb22*z  + mb12)   ) *  &
                  ( (zz  + mc23*z  + mc13) / (zz  + mb23*z  + mb13)   ) *  &
                  ( (zz  + mc24*z  + mc14) / (zz  + mb24*z  + mb14)   ) *  &
                  ( (zz  + mc25*z  + mc15) / (zz  + mb25*z  + mb15)   )
!            finalize erfc(x), page 7, formula (61) with modified parameters
             f = M  *  EXP( -zz )                ! f = erfc( x)
!
!alternative original M (Mills ratio), page 9, formula (69)
!a           z2  = SQRT( two ) * z               ! interface to M
!a           zz2 = z2 * z2                       ! square interface
!a           M   = (         one            / (           z2 + ob00)   ) *  &
!a                 ( (zz2 + oc21*z2 + oc11) / (zz2 + ob21*z2 + ob11)   ) *  &
!a                 ( (zz2 + oc22*z2 + oc12) / (zz2 + ob22*z2 + ob12)   ) *  &
!a                 ( (zz2 + oc23*z2 + oc13) / (zz2 + ob23*z2 + ob13)   ) *  &
!a                 ( (zz2 + oc24*z2 + oc14) / (zz2 + ob24*z2 + ob14)   ) *  &
!a                 ( (zz2 + oc25*z2 + oc15) / (zz2 + ob25*z2 + ob15)   )
!a           finalize erfc(x), page 7, formula (61)
!a           f = SQRT( twodivpi )  *  M  *  EXP( -(z*z) )
!
!            conversion erfc(x) to erf(x) for negative and positive x
             if( x < zero ) then                 ! x is negative
                 f = ( f - half ) - half         ! erf(x) = erfc(x) - 1
             else                                ! x is positive
                 f = ( half - f ) + half         ! erf(x) = 1 - erfc(x)
             end if                              ! end if for x
!
!   limiterf < z=|x| ≤ +∞
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfDia
!
!-----------------------------------------------------------------------------
!