!
!   Remark:
!   1. ./.
!
!------------------------------------------------------------------------------------------------------------------------------------------
!
!   Error Function (erf): Low Precision
!
!   real(rk) function erfAStegun      ( x )       result( f )  Hastings, A./Stegun  (1972),  p. 299, AS 7.1.26, AS 7.1.27, AS 7.1.28
!   real(rk) function erfMenzel       ( x )       result( f )  Menzel               (1975)   p. 367, direct calculation and erratum p. 923
!   real(rk) function erfTANH         ( x )       result( f )  Vedder               (1987),  p. 762, formula (5)
!                                                              Vazquez-Leal et al.  (2012),  p.   5, formula (3.1)
!                                                              Vazquez-Leal et al.  (2012),  p.  20, formula (7.1)
!                                                              Sandoval-He. et al.  (2019),  p.1781, formula (25)
!   real(rk) function erfWinitzki     ( x )       result( f )  Winitzki             (2008),  p.   1, formula (3) and (4)
!                                                              Soranzo and Epure    (2012),  p.   2, formula (2) converted
!                                                              Soranzo and Epure    (2014),  p.4330, 2nd formula
!                                                              Soranzo and Epure    (2014),  p.4330, 3rd formula
!   real(rk) function erfPade         ( x )       result( f )  Bercu                (2020),  p.   3, Padé [5,6] approximation
!                                                              Howard               (2022),  p.   3, formula (4) and (5), Padé [4,2] approx.
!
!------------------------------------------------------------------------------------------------------------------------------------------
!
!   erfAStegun
!     Error Function f = erf(x) with rational approximation referenced
!     Hastings (1955) and Abramowitz/Stegun (1972)
!   Reference:
!     Cecil Hastings, "Approximations for digital computers", Princeton, New
!      Jersey, Princeton University Press, 1955, page 169 sheet 45, page 185
!      sheet 61, page 187 sheet 63.
!     Milton Abramowitz and Irene A. Stegun, "Handbook of Mathematical
!      Functions with Formulas, Graphs, and Mathematical Tables", Ninth
!      Dover printing, 1972, page 299.
!   Remark:
!     Elimination of "pure function" due to write/read statement.
!     Selection ("selec") according input
!      1: Hastings ('55), p.169, sh.45 or Abram./Stegun ('72), p.299, f.7.1.26
!      2: Hastings ('55), p.185, sh.61 or Abram./Stegun ('72), p.299, f.7.1.27
!      3: Hastings ('55), p.187, sh.63 or Abram./Stegun ('72), p.299, f.7.1.28
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfAStegun( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, four, sixteen, limiterf
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: s             ! help variable
    real   (rk)                 :: t             ! help variable
    integer(ik), save           :: selec         ! help variable
!
!   parameter Hastings (1955), page 169, sheet 45 or
!   parameter Abramowitz/Stegun (1972), page 299, formula 7.1.26
    real   (rk), parameter      ::                                   &
            p0 =   0.327591100E+00_rk,   p1 =   1.061405429E+00_rk,  &
            p2 =  -1.453152027E+00_rk,   p3 =   1.421413741E+00_rk,  &
            p4 =  -0.284496736E+00_rk,   p5 =   0.254829592E+00_rk
!
!   parameter Hastings (1955), page 185, sheet 61 or
!   Abramowitz/Stegun (1972), page 299, formula 7.1.27
    real   (rk), parameter      ::                                   &
            q1 =   0.278393000E+00_rk,   q2 =   0.230389000E+00_rk,  &
            q3 =   0.000972000E+00_rk,   q4 =   0.078108000E+00_rk
!
!   parameter Hastings (1955), page 187, sheet 63 or
!   Abramowitz/Stegun (1972), page 299, formula 7.1.28
    real   (rk), parameter      ::                                   &
            r1 =  0.0705230784E+00_rk,   r2 =  0.0422820123E+00_rk,  &
            r3 =  0.0092705272E+00_rk,   r4 =  0.0001520143E+00_rk,  &
            r5 =  0.0002765672E+00_rk,   r6 =  0.0000430638E+00_rk
!
!
    if( x <= -HUGE(x) .and. x >= -HUGE(x) ) then
        write( *,                                                     '(  &
         & t2,"choose selec <1,2 or 3>: ",                             /, &
         & t2,"1: Abramowitz/Stegun (1972), page 299, formula 7.1.26", /, &
         & t2,"2: Abramowitz/Stegun (1972), page 299, formula 7.1.27", /, &
         & t2,"3: Abramowitz/Stegun (1972), page 299, formula 7.1.28"  )' )
        read(*,*) selec                          ! selection is ...
    end if
!
    z = ABS( x )                                 ! z = |x|
!
!   z=|x| == zero  =>  erf( [z=|x|] ≤ zero ) = zero
         if( z <= zero ) then                    ! x ≤ 0.0
             f = zero                            ! f = 0.0
!
!   0.0 < z=|x| ≤ limiterf
    else if( z <= limiterf ) then
!
!        Abramowitz/Stegun (1972), page 299, formula 7.1.26
         if( selec == 1_ik ) then
             t = one / (one + p0*z)
             f = ((((p1*t +p2)*t + p3)*t + p4)*t + p5)*t
             f = one  -  EXP( -(z*z) ) * f
         end if
!
!        Abramowitz/Stegun (1972), page 299, formula 7.1.27
         if( selec == 2_ik ) then
             t = one + q1*   z    + q2*  (z*z)   &
                     + q3*(z*z*z) + q4*(z*z*z*z)
             f = one - one/( t**four )
         end if
!
!        Abramowitz/Stegun (1972), page 299, formula 7.1.28
         if( selec == 3_ik ) then
             s = z * z                           ! s = z²
             t = one + r1*   z    + r2*   s      &
                     + r3* (s*z)  + r4* (s*s)    &
                     + r5*(s*s*z) + r6*(s*s*s)
             f = one - one/( t**sixteen )
         end if                                  ! end if selec
!
         if( x < zero ) f = -f                   ! erf(-x) = -erf(x)
!
!   limiterf < z=|x| ≤ +∞
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfAStegun
!
!-----------------------------------------------------------------------------
!
!   erfMenzel
!     Error Function f = erf(x) with straight formula referenced Menzel (1975)
!   Reference:
!     R. Menzel, "Approximate closed form solution to the error function",
!      American Journal of Physics, 1975, Volume 43, pages 366-367 .
!     R. Menzel, "Erratum: Approximate closed form solution to the error
!      function", American Journal of Physics, 1975, Volume 43, page 923 .
!   Remark:
!     Realisation of formula on page 367 with erratum on page 923 .
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfMenzel( x )   result( f )
!
    use kinds, only : rk
    use const, only : zero, one, four, pi, limiterf
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
!
!
    z = ABS( x )                                 ! z = |x|
!
!   z=|x| == zero  =>  erf( [z=|x|] ≤ zero ) = zero
         if( z <= zero ) then                    ! x ≤ 0.0
             f = zero                            ! f = 0.0
!
!   0.0 < z=|x| ≤ limiterf
    else if( z <= limiterf ) then
!
!            Menzel (1975), page 367 and erratum page 923
             f = SQRT( one - EXP(-four * (z * z) / pi) )
!
!            conversion of erf(-x) = -erf(x) for negative x
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
!   limiterf < z=|x| ≤ +∞
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfMenzel
!
!-----------------------------------------------------------------------------
!
!   erfTANH
!     Error Function f = erf(x) through Tangens hyperbolicus referenced Vedder
!     (1987), Vazquez-Leal et al. (2012) and Sandoval-Hernandez et al. (2019)
!   Reference:
!     J.D. Vedder, "Simple approximations for the error function and its
!      inverse", American Journal of Physics, 1987, Volume 55, Issue 8,
!      Pages 762-763 .
!     H. Vazquez-Leal et al., "High Accurate Simple Approximation of Normal
!      Distribution Integral", Mathematical Problems in Engineering (MPE),
!      2012, Volume 2012, Article ID 124029, 22 pages.
!     M.A. Sandoval-Hernandez et al. "New handy and accurate approximation for
!      the Gaussian integrals with applications to science and engineering",
!      Open Mathematics, 2019, Volume 17, Issue 1, Pages 1774-1793.
!   Remark:
!     Elimination of "pure function" due to write/read statement.
!     Selection ("selec") according input
!      1: Vedder              (1987), p. 762, formula (5)
!      2: Vazquez-Leal et al. (2012), p.   5, formula (3.1)
!      3: Vazquez-Leal et al. (2012), p.  20, formula (7.1)
!      4: Sandoval-He. et al. (2019), p.1781, formula (25)
!     Alternative TANH(t) := (EXP(2*t) - 1) / (EXP(2*t) + 1)
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfTANH( x )   result( f )
!
    use kinds, only : ik, rk  
    use const, only : zero, one, two, sqrtpi, limiterf
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    integer(ik), save           :: selec         ! help variable
!
!
    if( x <= -HUGE(x) .and. x >= -HUGE(x) ) then
        write( *,                                                    '(  &
         & t2,"choose selec <1,2,3 or 4>: ",                          /, &
         & t2,"1: Vedder              (1987), p. 762, formula (5)  ", /, &
         & t2,"2: Vazquez-Leal et al. (2012), p.   5, formula (3.1)", /, &
         & t2,"3: Vazquez-Leal et al. (2012), p.  20, formula (7.1)", /, &
         & t2,"4: Sandoval-He. et al. (2019), p.1781, formula (25) "  )' )
        read(*,*) selec                          ! selection is ...
    end if
!
    z = ABS( x )                                 ! z = |x|
!
!   z=|x| == zero  =>  erf( [z=|x|] ≤ zero ) = zero
         if( z <= zero ) then                    ! x ≤ 0.0
             f = zero                            ! f = 0.0
!
!   0.0 < z=|x| ≤ limiterf
    else if( z <= limiterf ) then
!
!        Vedder (1987), page 762, formula (5), 0 ≤ x
         if( selec == 1_ik ) then
             f = TANH(   167.0E+00_rk/148.0E+00_rk *    z    &
                   &   +  11.0E+00_rk/109.0E+00_rk * (z*z*z) )
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
         end if
!
!        Vazquez-Leal et al. (2012), page 5, formula (3.1), -∞ ≤ x +∞
         if( selec == 2_ik ) then
             f = TANH( ( 39.00E+00_rk * x  ) / (two*sqrtpi) -   &
                       (111.00E+00_rk / two)                *   &
                       ATAN( (  35.00E+00_rk * x      )     /   &
                             ( 111.00E+00_rk * sqrtpi )       ) )
         end if
!
!        Vazquez-Leal et al. (2012), page 20, formula (7.1), -∞ ≤ x +∞
         if( selec == 3_ik ) then
             f = TANH( ( 77.00E+00_rk / 75.00E+00_rk ) * x  +   &
                       (116.00E+00_rk / 25.00E+00_rk )      *   &
                 TANH( (147.00E+00_rk / 73.00E+00_rk ) * x  -   &
                       ( 76.00E+00_rk /  7.00E+00_rk )      *   &
                 TANH( ( 51.00E+00_rk /278.00E+00_rk ) * x    )))
         end if
!
!        Sandoval-H. et al (2019), p. 1781, formula (25), -∞ ≤ x +∞
         if( selec == 4_ik ) then
             f = two*x * (one  +  0.089430E+00_rk * (x*x))
             f = TANH( f / sqrtpi )
         end if
!
!   limiterf < z=|x| ≤ +∞
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfTANH
!
!-----------------------------------------------------------------------------
!
!   erfWinitzki
!     Error Function f = erf(x) with simple formula referenced Winitzki (2008)
!     and improvements referenced Soranzo/Epure (2012 and 2014)
!   Reference:
!     Serge Winitzki, "A handy approximation for the error function and its
!      inverse". 6. February 2008. Downloaded from:
!      http://homepages.physik.uni-muenchen.de/winitzki/erf-approx.pdf,
!      meanwhile: https://web.archive.org/web/20100601094000/https://homepages
!      .physik.uni-muenchen.de/~Winitzki/erf-approx.pdf,
!      last visited 3. August 2023 .
!      here: page 1, formula (3) and (4)
!     Alessandro Soranzo and Emanuela Epure, "Practical Explicitly Invertible
!      Approximation to 4 Decimals of Normal Cumulative Distribution Function
!      Modifying Winitzki's Approximation of erf", arXiv:1211.6403v1 [math.ST],
!      27. November 2012, 4 Pages, https://doi.org/10.48550/arXiv.1211.6403
!      here: page 2, formula (2) converted
!     Alessandro Soranzo and Emanuela Epure, "Very Simply Explicitly
!      Invertible Approximations of Normal Cumulative and Normal Quantile
!      Function", Applied Mathematical Sciences, 2014, Volume 8, Issue 87,
!      Pages 4323-4341, https://doi.org/10.12988/ams.2014.45338
!      here: page 4330, 2nd and 3rd formula
!   Remark:
!     Elimination of "pure function" due to write/read statement.
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfWinitzki( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, two, three, four, eight,  &
                      pi, limiterf
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x or x*x
    real   (rk)                 :: c, d          ! variable
    integer(ik), save           :: selec         ! help variable
!
!   mathematical & machine-dependent constants/parameters
    real   (rk), parameter      ::                                  &
            a = eight / (three * pi) * (pi - three) / (four - pi),  &
            b = four/pi
!
!
    if( x <= -HUGE(x) .and. x >= -HUGE(x) ) then
        write( *,                                                   '(  &
         & t2,"choose selec <1,2,3 or 4>: ",                         /, &
         & t2,"1: Winitzki          (2008), p.   1, f. (3) and (4)", /, &
         & t2,"2: Soranzo and Epure (2012), p.   2, formula (2)",    /, &
         & t2,"3: Soranzo and Epure (2014), p.4330, 2nd formula",    /, &
         & t2,"4: Soranzo and Epure (2014), p.4330, 3rd formula"     )' )
        read(*,*) selec                          ! selection is ...
    end if
!
    z = ABS( x )                                 ! z = |x|
!
!   z=|x| == zero  =>  erf( [z=|x|] ≤ zero ) = zero
         if( z <= zero ) then                    ! x ≤ 0.0
             f = zero                            ! f = 0.0
!
!   0.0 < z=|x| ≤ limiterf
    else if( z <= limiterf ) then
!
!        Winitzki (2008), page 1, formula (3) and (4)
         if( selec == 1_ik ) then
             z = z * z                           ! z = x * x
             f = SQRT( one - EXP(-z * (b + a*z)/(one + a*z)) )
         end if
!
!        Soranzo and Epure (2012), page 2, formula (2) converted
         if( selec == 2_ik ) then
             c = z * z;   d = c * c
             f =   (    - 1.2735457E+00_rk*c - 0.1487936E+00_rk*d) &
                 / (one + 0.1480931E+00_rk*c + 0.0005160E+00_rk*d)
             f = SQRT( one - EXP(f) )
         end if
!
!        Soranzo and Epure (2014), page 4330, second formula
         if( selec == 3_ik ) then
             z = z * SQRT( two)                  ! back transformation
             c = z * z
             f = (17.000E+00_rk + c) / (26.694E+00_rk + two*c)
             f = SQRT( one - EXP(-c * f) )
         end if
!
!        Soranzo and Epure (2014), page 4330, third formula
         if( selec == 4_ik ) then
             z = z * SQRT( two)                  ! back transformation
             c = z * z;   d = c * c
             f =   (    - 1.2735457E+00_rk*c - 0.0743968E+00_rk*d) &
                 / (two + 0.1480931E+00_rk*c + 0.0002580E+00_rk*d)
             f = SQRT( one - EXP(f) )
         end if
!
         if( x < zero ) f = -f                   ! erf(-x) = -erf(x)
!
!   limiterf < z=|x| ≤ +∞
    else
         f = one                                 ! f = 1.0
         if( x < zero ) f = -f                   ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfWinitzki
!
!-----------------------------------------------------------------------------
!
!   erfPade
!     Error Function f = erf(x) with padé [5,6] approximation referenced
!     Bercu (2020)
!   Reference:
!     Gabriel Bercu, "New Refinements for the Error Function with Applications
!      in Diffusion Theory", Symmetry, 2020, Volume 12, Issue 12,
!      Article 2017, 13 pages, http://doi.org/10.3390/sym12122017
!     Roy M. Howard, "Arbitrarily Accurate Analytical Approximations for the
!      Error Function", Mathematical and Computational Applications, 2022,
!      Volume 27, Issue 1, Article 14, 44 pages,
!      https://doi.org/10.3390/mca27010014
!   Remark:
!     Elimination of "pure function" due to write/read statement.
!     Selection ("selec") according input
!      1: Bercu  (2020), page 3, Padé [5,6] approximation
!      2: Howard (2022), page 3, formula (4) and (5), Padé [4,2]
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfPade( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, four,  &
                      pi, twodivsqrtpi, limiterf
!
    implicit none
!
!   interface
     real  (rk), intent(in )    :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: z2, z3, z4,   &
                                   z5, z6        ! z^2,z^3,...,z^6
    integer(ik), save           :: selec         ! help variable
!
!   parameter for padé [5,6] approximation from Bercu (2020)
    integer(ik), parameter      :: n             =  3_ik
    real   (rk), parameter      :: a(1_ik:n     )=             (/ &
            10242540.0E+0_rk,  1027320.0E+0_rk,  302379.0E+0_rk  /)
    real   (rk), parameter      :: b(1_ik:n+1_ik)=             (/ &
            10242540.0E+0_rk,  4441500.0E+0_rk,  758625.0E+0_rk,  &
               52595.0E+0_rk                                     /)
!
!   parameter for padé [4,2] approximation from Howard (2022)
    real   (rk), parameter      ::                                           &
     n1 =       279.0E+0_rk/100.0E+5_rk, n2 =   -303923.0E+0_rk/100.0E+5_rk, &
     n3 =     34783.0E+0_rk/ 50.0E+5_rk, n4 =     40793.0E+0_rk/100.0E+5_rk, &
     d1 = -21941279.0E+0_rk/100.0E+5_rk, d2 =   3329407.0E+0_rk/ 25.0E+5_rk
!
!
    if( x <= -HUGE(x) .and. x >= -HUGE(x) ) then
        write( *,                                                      '(  &
         & t2,"choose selec <1 or 2>: ",                                /, &
         & t2,"1: Bercu  (2020), p. 3, Pade [5,6] approximation",       /, &
         & t2,"2: Howard (2022), p. 3, formula (4) and (5), Pade [4,2]" )' )
        read(*,*) selec                          ! selection is ...
    end if
!
    z = ABS( x )                                 ! z = |x|
!
!   z=|x| == zero  =>  erf( [z=|x|] ≤ zero ) = zero
         if( z <= zero ) then                    ! x ≤ 0.0
             f = zero                            ! f = 0.0
!
!   0.0 < z=|x| ≤ limiterf
    else if( z <= limiterf ) then
!
!        Bercu (2020), Padé [5,6] approximation on page 3
         if( selec == 1_ik ) then
             z2 =z*z;  z3=z2*z;  z4=z2*z2;  z5=z3*z2;  z6=z3*z3
             f  =   ( a(1)*z + a(2)*z3 + a(3)*z5)          &
                  / ( b(1)   + b(2)*z2 + b(3)*z4 + b(4)*z6 )
             f  = twodivsqrtpi * f               ! 2/√pi*Padé[5,6]
         end if
!
!        Howard (2020), Padé [4,2] approx. on p.3, formula (4) and (5)
         if( selec == 2_ik ) then
             z  = z / ( z + one )                ! x1
             z2 = z*z;  z3 = z2*z;  z4 = z2*z2   ! x1^2, x1^3, x1^4
             f = ( one +  (n1*z + n2*z2 + n3*z3 + n4*z4 ) / &
                          (one + d1*z + d2*z2 )             )
             f = SQRT( one - EXP( -(x*x) * (four / pi) * f ) )
         end if
!
         if( x < zero ) f = -f                   ! erf(-x) = -erf(x)
!
!   limiterf < z=|x| ≤ +∞
    else
!
         f = one                                 ! f = 1.0
         if( x < zero ) f = -f                   ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfPade
!
!-----------------------------------------------------------------------------
!