!
!
!   Remark:
!   1. Alles o.k.
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   Error Function (erf): Power Series / Series Expansions
!
!   real(rk) function erfSE           ( x, it )   result( f )  SE 1: Abram./Stegun 7.1.5. (1972),  Mclaurin Series
!                                                              SE 2: Abram./Stegun 7.1.6. (1972),  Power Series Expansion
!                                                              SE 3: Decker               (1975),  Power Series Expansion
!                                                              SE 4: Zhang/Jin            (1996),  Power Series Expansion
!   real(rk) function erfSEasymp      ( x, it )   result( f )  Zhang/Jin                  (1996),  asymptotic Series EXpansion
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   erfSE
!     Error Function f = erf(x) with Maclaurin and others, referenced by
!     various sources, Hoering (2023)
!   Reference:
!     Maclaurin Series Expansion (one from many sources)
!      selec == 1
!       Milton Abramowitz, Irene A. Stegun, "Handbook of Mathematical
!        Functions with Formulas, Graphs, and Mathematical Tables", Ninth
!        Dover printing, 1972, here: page 297, formula 7.1.5.
!     Power Series Expansion, various expansions:
!      selec == 2
!       Milton Abramowitz, Irene A. Stegun, "Handbook of Mathematical
!        Functions with Formulas, Graphs, and Mathematical Tables", Ninth
!        Dover printing, 1972,
!        here: page 297, formula 7.1.6.
!      selec == 3
!       Daniel L. Decker, "Computer evaluation of the complementary error
!        function", American Journal of Physics, 1975, Volume 43, Issue 9,
!        Pages 833–834, https://doi.org/10.1119/1.9699,
!        here: page 833, formula (1)
!       Forman S. Acton, "Numerical Methods That Work", 1990, 2nd printing,
!        updated and revised, XX, 549 pages, Washington, DC: Mathematical
!        Association of America, PPN: 114363080, E-ISBN: 978-1-4704-5727-3,
!        here: page 16, formula (1.12)
!      selec == 4
!       Shanjie Zhang and Jianming Jin, "Computation of Special Functions",
!        1996, XXVI, 717 pages, John Wiley & Sons Inc., New York, 
!        ISBN: 0-471-11963-6,
!        here: page 621, formula 16.1.4
!   Remark:
!     Parameter "limiterf" is defined in "Constants" for erf,
!      where erf( |x| ≥ limiterf ) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfSE( x, it )   result( f )
!
    use kinds, only : ik, rki, rk
    use const, only : zero, half, one, two,        &
                      onedivsqrtpi, twodivsqrtpi,  &
                      eps, limiterf
!
    implicit none
!
!   interface
     real   (rk), intent(in )   :: x             ! x from erf(x)
     integer(ik), intent(out)   :: it            ! iterations
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: zz            ! zz = z² = |x|²
    integer(ik), save           :: selec         ! series selection
!
!   local variable
    real   (rk)                 :: n             ! variable sum
    real   (rk)                 :: p1, p2        ! variables numerator
    real   (rk)                 :: q1, q2        ! variables denominator
    real   (rk)                 :: s             ! incremental sum
!
!   mathematical & machine-dependent constants/parameters
    integer(ik)                 :: itmax         ! max. number of iterations
    integer(ik), parameter      ::                     & ! vector for itmax
       it1(1_ik:3_ik) = (/  31_ik, 100_ik, 200_ik /),  & ! sp, dp, ep/qp
       it2(1_ik:3_ik) = (/  27_ik, 100_ik, 200_ik /),  & ! sp, dp, ep/qp
       it3(1_ik:3_ik) = (/  25_ik,  95_ik, 200_ik /),  & ! sp, dp, ep/qp
       it4(1_ik:3_ik) = (/ 200_ik, 200_ik, 200_ik /)     ! sp, dp, ep/qp
!
!
    if( x <= -HUGE(x) .and. x >= -HUGE(x) ) then
        write( *,                                             '(  &
         & t2,"choose selec <1,2,3 or 4>:",                    /, &
         & t2,"1: Maclaurin:    A/Stegun (1972), itmax = ",i3, /, &
         & t2,"2: Power Series: A/Stegun (1972), itmax = ",i3, /, &
         & t2,"3: Power Series: Decker   (1975), itmax = ",i3, /, &
         & t2,"4: Power Series: Zhang/Jin(1996), itmax = ",i3  )' ) &
         &           it1(rki), it2(rki), it3(rki), it4(rki)
        read(*,*) selec
    end if
!
    it    = 0_ik                                 ! no iterations
    z     = ABS( x )                             ! z = |x|
!
!   z=|x| == zero  =>  erf( [z=|x|] ≤ zero ) = zero
         if( z <= zero ) then                    ! x ≤ 0.0
             f = zero                            ! f = 0.0
!
!   0 < z=|x| ≤ limiterf
    else if( z < limiterf ) then
!
             if( selec == 1_ik ) then
!            Maclaurin Series Expansion, referenced based e.g.
!            Abramowitz/Stegun (1972), page 297, formula 7.1.5.
!
!                      2     ∞  (-1)^n * z^(2n+1)
!            erf(z) = --- *  ∑  -----------------
!                     √pi   n=0    n! * (2n+1)
!
!                      2      z^1   z^3   z^5   z^7
!            erf(z) = --- * ( --- - --- + --- - --- + ... )
!                     √pi     1*1   1*3   2*5   6*7
!
             itmax = it1( rki )                  ! max. iterations
             zz    = z*z                         ! zz = z² = |x|²
             n     = zero                        ! (n=0)
             p1    = z                           ! z^[2*(n=0) + 1]
             q1    = one                         ! (n=0)!
             q2    = one                         ! (2*(n=0)+1)
             f     = z                           ! sum for (n=0)
             do while ( it < itmax ); it=it+1_ik ! iterations (n≥1)
                p1 = -p1 * zz                    ! alt. -/+ *(p1*zz)
                n  =  n  + one                   ! n + 1
                q1 =  q1 * n                     ! n!, n=1,2,3,...
                q2 =  q2 + two                   ! [2*(n=1,2,3,...) + 1]
                s  =  p1 / (q1 * q2)             ! increment s
                f  =  f  + s                     ! sum for (n≥1)
                if( ABS(s) <= ABS(f)*eps ) exit  ! precision achieved
             end do
             f  = twodivsqrtpi * f               ! finalize f, x ≥ 0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
!alternative
!a
!a           The first 11 items of the Maclaurin Series Expansion, with
!a           n = 0, 1, 2, ..., 11 for the denominator = [n! * (2n+1)]
!a           f  = z                      - z** 3/        3.0E+0_rk &
!a              + z** 5/      10.0E+0_rk - z** 7/       42.0E+0_rk &
!a              + z** 9/     216.0E+0_rk - z**11/     1320.0E+0_rk &
!a              + z**13/    9360.0E+0_rk - z**15/    75600.0E+0_rk &
!a              + z**17/  685440.0E+0_rk - z**19/  6894720.0E+0_rk &
!a              + z**21/76204800.0E+0_rk - z**23/918086400.0E+0_rk
!a           finalize the formula for positive x
!a           f  = twodivsqrtpi * f               ! finalize f, x ≥ 0
!a           if( x < zero ) f = -f               ! erf(-x) = -erf(x)
             end if                              ! end if selec == 1
!
!
             if( selec == 2_ik ) then
!            Power Series Expansion, referenced
!            Abramowitz/Stegun (1972), page 297, formula 7.1.6.
!
!                     2*exp[-z²]   ∞  2^n * z^(2n+1)
!            erf(z) = ---------- * ∑  --------------
!                        √pi      n=0   (2n+1)!!
!            or
!                     2*exp[-z²]     1z^1   2z^3   4z^5
!            erf(z) = ---------- * [ ---- + ---- + ----- + ...]
!                        √pi           1    1*3    1*3*5
!
             itmax = it2( rki )                  ! max. iterations
             zz    = z*z                         ! zz = z² = |x|²
             p1    = one                         ! 2^(n=0)
             p2    = z                           ! z^[2*(n=0) + 1]
             q1    = one                         ! 1 * ...
             q2    = q1                          ! 1!! = 1
             f     = z                           ! sum for (n=0)
             do while ( it < itmax ); it=it+1_ik ! iterations (n≥1)
                p1 =  p1 * two                   ! 2^n = 1*2,2*2,4*2,
                p2 =  p2 * zz                    ! z^(2n+1)
                q1 =  q1 + two                   ! (2n+1) = 3,5,7,...
                q2 =  q2 * q1                    ! (2n+1)!!
                s  = (p1 * p2) / q2              ! increment s
                f  =  f  + s                     ! sum for (n≥0)
                if( ABS(s) <= ABS(f)*eps ) exit  ! precision achieved
             end do
             f = twodivsqrtpi * EXP(-zz) * f     ! finalize f, x ≥ 0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
             end if                              ! end if selec == 2
!
!
             if( selec == 3_ik ) then
!            Power Series Expansion, referenced
!            Decker (1975), page 833, formula (1) or
!            Acton (1990), page 16, formula (1.12)
!
!                     2*exp[-z²]*x   ∞  (u=2z²)^n
!            erf(z) = ------------ * ∑  ---------
!                        √pi        n=0  (2n+1)!!
!            or
!                     2*exp[-z²]*x     u^0   u^1     u^2
!            erf(z) = ------------ * [ --- + --- + ------ + ...]
!                        √pi            1    1*3    1*3*5
!
             itmax = it3( rki )                  ! max. iterations
             zz    = z*z                         ! zz = z² = |x|²
             p1    = two * zz                    ! 2z²
             p2    = one                         ! (2z²)^(n=0)
             q1    = one                         ! 1 * ...
             q2    = q1                          ! 1!! = 1
             f     = one                         ! sum for (n=0)
             do while ( it < itmax ); it=it+1_ik ! iterations (n≥1)
                p2 = p2 * p1                     ! (2z²)^(n=1,2,3,...)
                q1 = q1 + two                    ! (2n+1) = 3,5,7,...
                q2 = q2 * q1                     ! (2n+1)!!
                s  = p2 / q2                     ! increment s
                f  = f  + s                      ! sum for (n≥1)
                if( ABS(s) <= ABS(f)*eps ) exit  ! precision achieved
             end do
             f = twodivsqrtpi * x * EXP(-zz) * f ! finalize f, -/+ x
             end if                              ! end if selec == 3
!
!
             if( selec == 4_ik ) then
!            Power Series Expansion, referenced 
!            Zhang/Jin (1996), page 621, formula (16.1.4)
!
!                      2                   ∞  2^n * z^(2n+1)
!            erf(z) = --- * exp[-z²] *     ∑  --------------
!                     √pi                 n=0   (2n+1)!!
!            or
!                     2*z                  ∞   n     z²
!            erf(z) = --- * exp[-z²] *[1 + ∑   ∏  ---------]
!                     √pi                 n=1 k=1 (k + 1/2)
!            or
!                     2*z                      z^2   z^2    z^2    z^2
!            erf(z) = --- * exp[-z²] *[1 + ((( --- * ---) * ---) * ---) * ..]
!                     √pi                      1.5   2.5    3.5    4.5
!
             itmax = it4( rki )                  ! max. iterations
             zz    = z*z                         ! zz = z² = |x|²
             q1    = half                        ! [k=0 + (q1=0.5)]
             s     = one                         ! product increment
             f     = one                         ! f = 1 + (sum=0)
             do while ( it < itmax ); it=it+1_ik ! iterations (n≥1)
                q1  = q1 + one                   ! [(k>0) + 0.5]
                s   = s  * zz / q1               ! s * z²/(k+0.5)
                f   = f  + s                     ! sum for (n≥1)
                if( ABS(s) <= ABS(f)*eps ) exit  ! precision achieved
             end do
             f = twodivsqrtpi * x * EXP(-zz) * f ! finalize f, -/+x
             end if                              ! end if selec == 5
!
!   limiterf ≤ |x| ≤ +∞
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfSE
!
!-----------------------------------------------------------------------------
!
!   erfSEasymp
!     Error Function f = erf(x) with asymptotic Series Expansion, referenced
!     Zhang/Jin (1996)
!   Reference:
!     Shanjie Zhang and Jianming Jin, "Computation of Special Functions",
!     1996, XXVI, 717 pages, John Wiley & Sons Inc., New York,
!     ISBN: 0-471-11963-6,
!     here: page 621, formula (16.1.5)
!   Remark:
!     Parameter "eMax", defined in "Constants", avoids that the formula
!      "EXP(eMax*eMax)" overflows, where erf(z=|x| ≥ eMax) := +1.00
!     Cut off at [ABS(x) < 0.25] with f = 0.0 to avoid underflow.
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfSEasymp( x, it )   result( f )
!
    use kinds, only : ik, rki, rk
    use const, only : zero, half, one,  &
                      sqrtpi,           &
                      eps, eMax
!
    implicit none
!
!   interface
     real   (rk), intent(in )   :: x             ! x from erf(x)
     integer(ik), intent(out)   :: it            ! iterations
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: zz            ! z*z
    real   (rk)                 :: p1            ! variable numerator
    real   (rk)                 :: s             ! incremental sum
!
!   mathematical & machine-dependent constants/parameters
    integer(ik)                 :: itmax         ! max. number of iterations
    integer(ik), parameter      ::               &   ! vector for itmax
       it1(1_ik:3_ik) = (/  18_ik, 100_ik, 200_ik /) ! sp, dp, ep/qp
!
!
    it    = 0_ik                                 ! no iterations
    z     = ABS( x )                             ! z = |x|
!
!   z=|x| == zero  =>  normally "erf( [z=|x|] ≤ zero ) = zero",
!   but to avoid an underflow:  "erf( [z=|x|] < 0.25 ) = zero"
         if( z < 0.25E+00_rk ) then              ! x ≈ 0.0
             f = zero                            ! f = 0.0
!
!   0 < z=|x| ≤ normally limiterf, but here "< eMax" 
!   to show the asymptotic values of the Series Expansion
    else if( z < eMax ) then
!
!            asymptotic Series Expansion, referenced e.g.
!            Zhang/Jin (1996), page 621, formula (16.1.5)
!
!                         exp[-z²]         ∞  (-1)^n * (2n-1)!!
!            erf(z) = 1 - -------- *       ∑  -----------------
!                           √pi           n=0      (2z²)^n
!
!            or as formula (16.1.5)b
!                         exp[-z²]         ∞   n  -(2k-1)
!            erf(z) = 1 - -------- * [ 1 + ∑   ∏  -------- ]
!                         √pi * z         n=1 k=1    2z²
!
!            or reformulated
!                         exp[-z²]         ∞   n  -(k-1/2)
!            erf(z) = 1 - -------- * [ 1 + ∑   ∏  -------- ]
!                         √pi * z         n=1 k=1     z²
!
!            or
!                         exp[-z²]           -0.5   -1.5    -2.5    -3.5
!            erf(z) = 1 - -------- * [ 1 +(((---- * ----) * ----) * ----)* ..]
!                         √pi * z             z²     z²      z²      z²
!
             itmax = it1( rki )                  ! max. iterations
             zz    =  z*z                        ! zz = z² = |x|²
             p1    = -half                       ! [(k=0) - 1/2]
             s     =  one                        ! increment = 1*
             f     =  one                        ! sum for (n=0)
             do while( it < itmax ); it=it+1_ik  ! iterations (n≥1)
                 p1 =  p1 + one                  ! [k-1/2] = 0.5, 1.5,...
                 s  = -s  * p1 / zz              ! -/+ s * (k - 1/2)/z²
                 f  =  f  + s                    ! sum for (n≥1)
                 if( ABS(s) <= ABS(f)*eps ) exit ! precision achieved
             end do
             f = one - f / (EXP(zz) * sqrtpi*z)  ! finalize f, x ≥ 0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
!   normally limiterf < z=|x| ≤ +∞, but here: eMax ≤ z ≤ +∞
!   to show the asymptoic values of the Ser. Expansion
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfSEasymp
!
!-----------------------------------------------------------------------------
!