!
!   Remark:
!   1.
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   Error Function (erf): Continued Fraction (CF)
!
!   real(rk) function erfCF           ( x, it )   result( f )  Hoering              (2023)
!                                                              CF 1: Wall           (1948),  itmax=200
!                                                              CF 2: Stegun/Zucker  (1970),  itmax=200
!                                                                    Wolfram Res.   (2023)
!                                                              CF 3: Al-Fhaid       (2006),  itmax=200
!                                                              CF 4: Keisan         (2023),  itmax=200
!                                                                    Wolfram Res.   (2023)
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   erfCF
!     Error Function f = erf(x) with Continued Fraction referenced by various
!     sources, Hoering (2023)
!   Reference:
!     selec == 1
!      Hubert Stanley Wall, "Analytic Theory of Continued Fractions", 1948,
!       449 pages, D. van Nostrand Company, Inc., New York,
!       ISBN: 978-0-486-83044-5, PPN: 1024406237,
!       here: page 357, formula 92.12 with erf(z) = 2/√pi*I and z > 0
!
!     selec == 2
!      Irene A. Stegun and Ruth Zucker, "Automatic Computing Methods for
!       Special Functions", Journal of Research of the National Bureau of
!       Standards - B. Mathematical Sciences, 1970, Volume 74B, Issue 3,
!       Pages 211-224, http://dx.doi.org/10.6028/jres.074B.019, 
!       here: page 214, 5th formula
!      Wolfram Research, 2023, formula 06.25.10.0001.01  last visited
!       Aug 3, 2023 on http://functions.wolfram.com/06.25.10.0001.01
!       here: formula 06.25.10.0001.01
!
!     selec == 3
!      Abdulrahman S. Al-Fhaid, "Continued Fraction Evaluation of the Error
!       Function", Journal of King Abdulaziz University - Science Journal,
!       2006, Volume 18, Issue 1, page 85-92,
!       https://doi.org/10.4197/Sci.18-1.8,
!       here: page 89, formula (17) for 0 ≤ x and r = √2*|x|
!
!     selec == 4
!      Keisan Online Calculator, provided by Casio Computer Co., Ltd.,
!       ("Casio"), https://keisan.casio.com/menu/system/000000000760,
!       last visited 30. December 2023, now only available in Japanese
!       language
!      Wolfram Research Inc., "formula 06.27.10.0003.01", last visited
!       Aug 3, 2023 on http://functions.wolfram.com/06.25.10.0003.01
!       here: formula 06.27.10.0003.01
!
!     Modified Lentz method for Continued Fractions:
!      I.J. Thompson and A.R. Barnett, "Coulomb and Bessel functions of
!       complex arguments and order", Journal of Computational Physics,
!       1986, Volume 64, Issue 2, Pages 490–509,
!       https://doi.org/10.1016/0021-9991(86)90046-X,
!       here: page 507 and 508, (III) Lentz's method
!       Local Tiny (page 508, "small"): "loctiny" set to 1.00E-35, where
!        case 1: if( b(start) == zero ) f     = loctiny
!        case 2: if( C        == zero ) C     = loctiny
!        case 3: if( b+a*D    == zero ) b+a*D = loctiny
!   Remark:
!     Parameter "eMax", defined in "Constants", avoids that the formula
!      "EXP(eMax*eMax)" overflows, where erf(z=|x| ≥ eMax) := +1.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfCF( x, it )   result( f )
!
    use kinds, only : ik, rk, lk
    use const, only : zero, half, one, two,             &
                      pi, sqrtpi, twodivsqrtpi, eps, eMax
!
    implicit none
!
!   interface
     real   (rk), intent(in )   :: x             ! x for erf(x)
     integer(ik), intent(out)   :: it            ! iterations
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: z2            ! 2 * |x|
    real   (rk)                 :: zz            ! zz = z² = |x|²
    real   (rk)                 :: r             ! √2 * z
    real   (rk)                 :: r2            ! r2 = r² = (√2*z)²
    real   (rk)                 :: b, a, C, D, h ! Lentz variables
    real   (rk)                 :: k, n          ! help variable
    logical(lk)                 :: v             ! logical help variable
    integer(ik), save           :: selec         ! selection
    real   (rk)                 :: limit         ! regions limit
!
!   mathematical & machine-dependent constants/parameters
    integer(ik), parameter      :: itmax         = 200_ik      ! max. iterat.
!c  real   (rk), parameter      :: loctiny       = 1.00E-35_rk ! Lentz para.
!
    if( x <= -HUGE(x) .and. x >= -HUGE(x) ) then
        write( *,                                                    '(  &
         & t2,"choose selec <1,2,3 or 4>:",                           /, &
         & t2,"1: Wall          (1948) p.357 f. 92.12,  itmax = 200", /, &
         & t2,"2: Stegun/Zucker (1970) p.214 5th form., itmax = 200", /, &
         & t2,"3: Al-Fhaid      (2006) p. 89 f. (17),   itmax = 200", /, &
         & t2,"4: Keisan        (2023) online,          itmax = 200"  )' )
        read(*,*) selec
    end if
!
!
    it = 0_ik                                    ! no iterations
    z  = ABS( x )                                ! z = |x|
!
!   z=|x| == zero  =>  normally "erf( [z=|x|] ≤ zero ) = zero",
!   but to avoid division by zero: "erf( [z=|x|] ≤ limit) = zero"
          if( selec == 1_ik .or. selec == 2_ik ) then
              limit = 0.020E+00_rk               ! high regions CF
          else
              limit = TINY(x)                    ! low regions CF
          end if
          if( z <= limit ) then
              f = zero                           !   f = 0.0
!
!   0.00 < z=|x| ≤ normally limiterf, but here "< eMax" to show
!   the values of the Continued Fraction
    else if( z < eMax ) then
!
             if( selec == 1_ik ) then
!            Continued Fraction, Wall (1948), page 357, formula 92.12 with
!            erf(z) = 2/√pi * I, with z > 0 and modified Lentz method for
!            Continued Fraction
!
!                z                                       1/2 * exp[-z²]
!            I = ∫ e**(-t²) dt = 1/2*√pi - -----------------------------
!                0                                               a1 = 1
!                                          b0=z + ----------------------
!                                                                a2 = 2
!                                                 b1=2z + --------------
!                                                                a3 = 3
!                                                         b2=z + -------
!                                                                  ...
!            and formulated as erf(z) = 2/√pi * I :
!
!                         exp(-z²)                         a0 = 1
!            erf(z) = 1 - -------- * ------------------------------
!                           √pi                            a1 = 1
!                                    b0=z + -----------------------
!                                                          a2 = 2
!                                           b1=2z + ---------------
!                                                          a3 = 3
!                                                   b2=z + --------
!                                                            ...
!            or
!                         exp(-z²)     a0 a1  a2       an
!            erf(z) = 1 - -------- * ( -- --- -- ... ------- ... )
!                           √pi        z+ 2z+ z+     [z,2z]+
!
!                     with a(0) = 1, a(n) = n, for n ≥ 1
!
             z2 = z * two                        ! 2*z = 2*|x|
             b  = z;           v = .false.       ! b0
!c no need   if( b<=zero .and. b>=zero ) then    ! b0 == 0
!c no need       f = loctiny                     ! b0 == 0
!c no need   else
                 f = b                           ! b0 /= 0
!c no need   end if
             a  = zero                           ! prepare a1
             C  = f                              ! C <- f
             D  = zero                           ! D <- 0
             do while ( it < itmax ); it=it+1_ik ! iterations
                a = a + one                      ! a1, ..., an
                if( v ) then                     ! b1, ..., bn
                    b = z;     v = .false.       ! b =   z
                else
                    b = z2;    v = .true.        ! b = 2*z
                end if
!               calculation of C and D, incl. h = C*D
                C =         b + a/C              ! bn >> + <<
                D = one / ( b + a*D )            ! bn >> + <<
!c no need      C = b + a/C                      ! bn >> + <<
!c no need          if( C <= zero .and. C >= zero ) C = loctiny
!c no need      D = b + a*D                      ! bn >> + <<
!c no need          if( D <= zero .and. D >= zero ) then
!c no need              D = one / loctiny
!c no need          else
!c no need              D = one / D
!c no need          end if
                h = C * D                        ! h step
                f = f * h                        ! new function value
                if( ABS( h - one ) <= eps ) exit ! precision check
             end do
             f = one - (EXP(-(z*z))/sqrtpi) / f  ! finalize for pos. x
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
             end if                              ! selec == 1
!
!
             if( selec == 2_ik ) then
!            Continued Fraction, Stegun/Zucker (1970), p. 214, 5th formula or
!            Wolfram Research (2023), formula 06.25.10.0001.01 for 0 ≤ x and
!            modified Lentz method for Continued Fraction
!
!                          exp(-z²)                         a0 = 1.0
!            erf(z) = 1 - ---------- * -----------------------------
!                           √pi                             a1 = 0.5
!                                      b0=z + ----------------------
!                                                           a2 = 1.0
!                                             b1=z + ---------------
!                                                           a3 = 1.5
!                                                    b2=z + --------
!                                                             ...
!            or equivalent adjusted:
!
!                         2*exp(-z²)                           a0 = 1.0
!            erf(z) = 1 - ---------- * ---------------------------------
!                                                              a1 = 2.0
!                                      b0=2z + -------------------------
!                                                              a2 = 4.0
!                                              b1=2z + -----------------
!                                                              a3 = 6.0
!                                                      b2=2z + ---------
!                                                                ...
!            or
!                         2*exp(-z²)     a0   a1   a2      an
!            erf(z) = 1 - ---------- * ( ---  ---  --- ... --- ... )
!                           √pi          2z+  2z+  2z+     2z+
!
!                     with a(0) = 1, a(n) = 2*(n-1), for n ≥ 1
!
             b = z                               ! b0
             f = b                               ! b0 /= 0
             a = zero                            ! prepare a1
             C = f                               ! C <- f
             D = zero                            ! D <- 0
             do while ( it < itmax ); it=it+1_ik ! iterations
                a = a + half                     ! a1, ..., an
                C =        b + a/C               ! bn >> + <<
                D = one / (b + a*D)              ! bn >> + <<
                h = C * D                        ! h step
                f = f * h                        ! new function value
                if( ABS( h - one ) <= eps ) exit ! precision check
             end do
             f = one - (EXP(-(z*z))/sqrtpi) / f  ! finalize for pos. x
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
             end if                              ! selec == 2
!
!
             if( selec == 3_ik ) then
!            Continued Fraction, Al-Fhaid (2006), page 89, formula (17) for
!            0 ≤ x, parameter: r = SQRT(2)*z and modified Lentz method for
!            Continued Fraction
!
!                     √(2/pi)                         a0 = [r = √2*z]
!            erf(z) = ------- * --------------------------------------
!                     exp(z²)                               a1 = 1*r²
!                               b0=1 - -------------------------------
!                                                           a2 = 2*r²
!                                      b1=3 + ------------------------
!                                                           a3 = 3*r²
!                                             b2=5 - -----------------
!                                                           a4 = 4*r²
!                                                    b3=7 + ----------
!                                                              ...
!            or
!                     √(2/pi)      +a0  -a1  +a2     (-1)^n * (n*r²)
!            erf(z) = ------- * (  ---  ---  --- ... --------------- ...  )
!                     exp(z²)      1+   3+   5+        [b(n-1)+2]+
!
!                     with a(0) = r,  a(n) = (-1)^n * (n*r²),  for n ≥ 1
!                          b(0) = 1,  b(n) = b(n-1) + 2,       for n ≥ 1
!
             r  = SQRT( two ) * z                ! √2 * z
             r2 = r * r                          ! r2 = r² = (√2*z)²
             b  = one                            ! b0
             f  = b                              ! b0 /= 0
             k  = one                            ! altering sign an
             n  = zero                           ! prepare an
             C  = f                              ! C <- f
             D  = zero                           ! D <- 0
             do while ( it < itmax ); it=it+1_ik ! iterations
                k = -k                           ! (-1)**n
                n = n + one                      ! real it
                a = k * n*r2                     ! a1, ..., an
                b = b + two                      ! b1, ..., bn
                C =        b + a/C               ! bn >> + <<
                D = one / (b + a*D)              ! bn >> + <<
                h = C * D                        ! h step
                f = f * h                        ! new function value
                if( ABS( h - one ) <= eps ) exit ! precision check
             end do
             f = (SQRT(two/pi)*EXP(-(z*z))*r)/ f ! finalize for pos. x
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
             end if                              ! selec == 3
!
!
             if( selec == 4_ik ) then
!            Continued Fraction, Keisan Online Calculator (2023) for 0 ≤ x and
!            parameter:  a(n=0) = 2*z/√pi * exp[-(z*z)]      b(n=0) = 1
!                        a(n>0) = (-1)**n * 2*n * x²         b(n>0) = 2*n + 1
!            or Wolfram Research (2023), formula 06.27.10.0003.01 for 0 ≤ x
!            and modified Lentz method for Continued Fraction
!
!                     2z*exp(-z²)                               a0 = +1
!            erf(z) = ----------- * --------------------------------------
!                         √pi                                   a1 = -2z²
!                                   b0=1 + -------------------------------
!                                                               a2 = +4z²
!                                          b1=3 + ------------------------
!                                                               a3 = -6z²
!                                                 b2=5 + -----------------
!                                                               a4 = +8z²
!                                                        b3=7 + ----------
!                                                                  ...
!            or
!                     2z*exp(-z²)     +a0  -a1  +a2     (-1)^n * (2n*z²)
!            erf(z) = ----------- * ( ---  ---  --- ... ---------------- ... )
!                         √pi         1+   3+   5+         [b(n-1)+2]+
!
!                     with a(0) = r,  a(n) = (-1)^n * (2n*z²),  for n ≥ 1
!                          b(0) = 1,  b(n) = b(n-1) + 2,        for n ≥ 1
!
             zz = z * z                          ! zz = z²
             b  = one                            ! b0
             f  = b                              ! b0 /= 0
             n  = zero                           ! prepare an
             k  = one                            ! prepare (-1)**n
             C  = f                              ! C <- f
             D  = zero                           ! D <- 0
             do while ( it < itmax ); it=it+1_ik ! iterations
                k = -k                           ! (-1)**n
                n = n + two                      ! n=2,4,6,...,(2*n)
                a = k * n*zz                     ! a1, ..., an
                b = b + two                      ! b1, ..., bn
                C =        b + a/C               ! bn >> + <<
                D = one / (b + a*D)              ! bn >> + <<
                h = C * D                        ! h step
                f = f * h                        ! new function value
                if( ABS( h - one ) <= eps ) exit ! precision check
             end do
             f = (twodivsqrtpi * EXP(-zz) * z)/f ! finalize for pos. x
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
             end if                              ! selec == 4
!
!
!   normally limiterf < z=|x| ≤ +∞, but here:
!   eMax ≤ z ≤ +∞ to show the values of the Continued Fraction
    else
             f = one                             ! f = 1.0
             if( x < zero ) f = -f               ! erf(-x) = -erf(x)
!
    end  if                                      ! end if cases
!
    return
    end function erfCF
!
!-------------------------------------------------------------------------------
!
