!
!   Remark:
!   1. ./.
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   Complementary Error Function (erfc): High Precision
!
!   real(rk) function erfcIntrinsic   ( x )       result( f )  GFortran/Silverfrost (2023),  Intrinsic ERFC
!   real(rk) function erfcCody        ( x )       result( f )  Cody                 (1969),  Rat. Chebyshev Approximation
!   real(rk) function erfcSchonfelder ( x )       result( f )  Schonfelder          (1978),  Chebyshev Approximation
!   real(rk) function erfcShepherd    ( x )       result( f )  Shepherd/Laframboise (1981),  Chebyshev Approximation
!   real(rk) function erfcSlatec      ( x )       result( f )  Fullerton            (1993),  Chebyshev Series
!   real(rk) function erfcSun         ( x )       result( f )  Sun Microsystems     (1993),  Various Polynomials
!   real(rk) function erfcOoura8a     ( x )       result( f )  Ooura                (1996),  Polynomial
!   real(rk) function erfcOoura8b     ( x )       result( f )  Ooura                (1996),  Polynomial
!   real(rk) function erfcOoura16     ( x )       result( f )  Ooura                (1996),  Polynomial - quad precision
!   real(rk) function erfcDia         ( x )       result( f )  Dia                  (2023),  Mill Ratio
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   erfcIntrinsic
!     Complementary Error Function f = erfc(x) with intrinsic function
!     referenced GFortran or Silverfrost FTN95
!   Reference:
!     GFortran (GNU Fortran) version 6.3.0-1, part of GCC (GNU
!      Compiler Collection). “class: elemental function ERF”,
!      2024, http://gcc.gnu.org
!     Silverfrost FORTRAN 95, “class: elemental function ERF”,
!      2024, https://www.silverfrost.com/ftn95-help/intrin/
!      erf.aspx
!   Remark:
!     For Silverfrost FTN95 this intrinsic function is not implemented
!      for extended precision and therefore set to "zero".
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcIntrinsic( x )   result( f )
!
    use kinds, only : compiler, ik, rki, rk
    use const, only : zero
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    integer(ik)                 :: h1            ! = var. "compiler"
    integer(ik)                 :: h2            ! = var. "kind-i"
!
!
    h1 = compiler                                ! var. compiler
!
!   Compiler = 1: GFortran part
         if( h1 == 1_ik ) then                   ! GFortran
             f = ERFC( x )                       ! intrinsic defined
!
!   Compiler = 2: Silverfrost FTN95 part
    else if( h1 == 2_ik ) then                   ! Silverfrost ftn 95
             h2 = rki                            ! variabilize rki
             if( h2 <= 2_ik ) then               ! kind = 1 or 2
                 f = ERFC( x )                   ! intrinsic
             else
!                Silverfrost is not able to handle kind=3 => f = 0.0
                 f = zero                        ! f = zero
             end if
    end  if
!
    return
    end function erfcIntrinsic
!
!-----------------------------------------------------------------------------
!
!   erfcCody
!     Complementary Error Function f = erfc(x) with Rational Chebyshev
!     Approximations referenced Cody (1969)
!   Reference:
!     W.J. Cody, "Rational Chebyshev Approximations for the Error Function",
!      Mathematics of Computation, 1969, Volume 23, Issue 107, Pages 631-637,
!      https://doi.org/10.1090/S0025-5718-1969-0247736-4
!     W.J. Cody, "SUBROUTINE CALERF(ARG,RESULT,JINT)", 1990, Latest
!      modification 19. March 1990, a fortran subroutine for error and scaled
!      and unscaled complementary error functions at
!      http://www.netlib.org/specfun/erf
!   Remark:
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "limiterfc" is defined in "Constants" for erfc,
!      where erfc( x ≤ limiterfc )   := +2.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcCody( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, two, four, sixteen,  &
                      onedivsqrtpi, twodivsqrtpi,     &
                      eMax, limiterfc
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfc(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: u             ! help variable
    real   (rk)                 :: xnum          ! sum numerator
    real   (rk)                 :: xden          ! sum denominator
    integer(ik)                 :: i             ! do...
!
!   Cody mathematical & machine-dependent constants/parameters
    real   (rk), parameter      :: xsmall   = EPSILON(one)/two
    real   (rk), parameter      :: thresh   = 0.46875E+00_rk
!                                  thresh  := erf(x) == erfc(x)
!
!   coefficients erf(x), p. 633 for xsmall < |x| ≤ (thresh = 0.46875)
    real   (rk), parameter      :: p1(0_ik:4_ik) =                (/ &
     3.209377589138469472562E+03_rk, 3.774852376853020208137E+02_rk, &
     1.138641541510501556495E+02_rk, 3.161123743870565596947E+00_rk, &
     1.857777061846031526730E-01_rk                                 /)
    real   (rk), parameter      :: q1(0_ik:4_ik) =                (/ &
     2.844236833439170622273E+03_rk, 1.282616526077372275645E+03_rk, &
     2.440246379344441733056E+02_rk, 2.360129095234412093499E+01_rk, &
     1.000000000000000000000E+00_rk                                 /)
!
!   coefficients erfc(x), p. 635 for (thresh = 0.46875) < |x| ≤ (four = 4.0)
    real   (rk), parameter      :: p2(0_ik:8_ik) =                (/ &
     1.23033935479799725272E+03_rk,  2.05107837782607146533E+03_rk,  &
     1.71204761263407058314E+03_rk,  8.81952221241769090411E+02_rk,  &
     2.98635138197400131132E+02_rk,  6.61191906371416294775E+01_rk,  &
     8.88314979438837594118E+00_rk,  5.64188496988670089180E-01_rk,  &
     2.15311535474403846343E-08_rk                                  /)
    real   (rk), parameter      :: q2(0_ik:8_ik) =                (/ &
     1.23033935480374942043E+03_rk,  3.43936767414372163696E+03_rk,  &
     4.36261909014324715820E+03_rk,  3.29079923573345962678E+03_rk,  &
     1.62138957456669018874E+03_rk,  5.37181101862009857509E+02_rk,  &
     1.17693950891312499305E+02_rk,  1.57449261107098347253E+01_rk,  &
     1.00000000000000000000E+00_rk                                  /)
!
!   coefficients erfc(x), p. 636 for (four = 4.0) < |x| < (six = 6.0)
    real   (rk), parameter      :: p3(0_ik:5_ik) =                (/ &
     6.58749161529837803157E-04_rk,  1.60837851487422766278E-02_rk,  &
     1.25781726111229246204E-01_rk,  3.60344899949804439429E-01_rk,  &
     3.05326634961232344035E-01_rk,  1.63153871373020978498E-02_rk  /)
    real   (rk), parameter      :: q3(0_ik:5_ik) =                (/ &
     2.33520497626869185443E-03_rk,  6.05183413124413191178E-02_rk,  &
     5.27905102951428412248E-01_rk,  1.87295284992346042097E+00_rk,  &
     2.56852019228982242072E+00_rk,  1.00000000000000000000E+00_rk  /)
!
!
!   -∞ ≤ x ≤ limiterfc
         if( x <= limiterfc ) then               ! cut off limiterfc
             f = two                             ! f = 2.00
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   0.00E+00 ≤ z=|x| ≤ ( xsmall = EPSILON(x)/two )
         if( z <= xsmall ) then
!            f = one - x * p1(0_ik) / q1(0_ik)   ! original Cody
             f = one - twodivsqrtpi * x          ! erfc(x) = 1.00 - erf(x)
!
!   (xsmall = EPSILON(x)/two) < z=|x| ≤ (thresh = 0.46875E+00)
    else if( z <= thresh ) then
             u    = z * z                        ! 1st formula p. 631
             xnum = p1(4_ik) * u
             xden = q1(4_ik) * u
             do i = 3_ik, 1_ik, -1_ik
                xnum = (xnum + p1(i)) * u
                xden = (xden + q1(i)) * u
             end do
             f = x * (xnum + p1(0_ik)) / (xden + q1(0_ik))
             f = one - f                         ! erfc(x) = 1.00 - erf(x)
!
!   (thresh=0.46875E+00) < z=|x| ≤ (four=4.00E+00)
    else if( z <= four ) then
             xnum = p2(8_ik) * z                 ! 2nd formula p. 631
             xden = q2(8_ik) * z
             do i = 7_ik, 1_ik, -1_ik
                xnum = (xnum + p2(i)) * z
                xden = (xden + q2(i)) * z
             end do
             f = (xnum + p2(0_ik)) / (xden + q2(0_ik))
             u = INT( z * sixteen ) / sixteen ! erfc specific
             f = EXP( -u*u ) * EXP( -(z-u)*(z+u) ) * f
             if( x < zero ) f = two - f          ! erfc(-x)= 2.00 - erfc(x)
!
!   4.00E+00 < z=|x| ≤ eMax
    else if( z <= eMax ) then
             u    = one / (z * z)                ! 3rd formula p. 631
             xnum = p3(5_ik) * u
             xden = q3(5_ik) * u
             do i = 4_ik, 1_ik, -1_ik
                xnum = (xnum + p3(i)) * u
                xden = (xden + q3(i)) * u
             end do
             f = u * (xnum + p3(0_ik)) / (xden + q3(0_ik))
             u = INT( z * sixteen ) / sixteen
             f = EXP(-u*u) * EXP(-(z-u)*(z+u)) *  &
                 ( (onedivsqrtpi-f) / z )
             if( x < zero ) f = two - f          ! erfc(-x)= 2.00 - erfc(x)
!
!   eMax < z=|x| ≤ +∞
    else                                         ! cut off z > eMax
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcCody
!
!-----------------------------------------------------------------------------
!
!   erfcSchonfelder
!     Complementary Error Function f = erfc(x) with Chebyshev Expansions
!     referenced Schonfelder (1978)
!   Reference:
!     J.L. Schonfelder, "Chebyshev Expansions for the Error and Related
!     Functions", Mathematics of Computation, 1978, Volume 32, Issue 144,
!     Pages 1232-1240, https://doi.org/10.1090/S0025-5718-1978-0494846-8
!     here: page 1235, table 3
!   Remark:
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "sqrteps" is defined in "Constants" for erfc,
!      where erfc( |x| ≤ sqrteps )   := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfc" is defined in "Constants" for erfc,
!      where erfc( x ≤ limiterfc )   := +2.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcSchonfelder( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, two, twodivsqrtpi, &
                      eMax, sqrteps, limiterfc
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfc(x)
!   end interface
    real   (rk)                 :: f             ! function value
!
!   local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
    real   (rk)                 :: t,  u         ! var. Chebyshev
    real   (rk)                 :: sv, d, dd     ! var. Clenshaw
    integer(ik)                 :: j             ! do ...
!
!   parameter for y = exp(x²)*erfc(x) and t = (x-k) / (x+k),
!   with k = 3.75 on page 1235, table 3: valid for 0 ≤ x < +∞
    real   (rk), parameter      :: k             =   3.750E+00_rk
    integer(ik), parameter      :: m             =  44_ik
    real   (rk), parameter      :: c(1_ik:m)     =                                 (/ &
    +6.10143081923200417926465815756E-01_rk, -4.34841272712577471828182820888E-01_rk, &
    +1.76351193643605501125840298123E-01_rk,  -6.0710795609249414860051215825E-02_rk, &
     +1.7712068995694114486147141191E-02_rk,   -4.321119385567293818599864968E-03_rk, &
       +8.54216676887098678819832055E-04_rk,    -1.27155090609162742628893940E-04_rk, &
        +1.1248167243671189468847072E-05_rk,       +3.13063885421820972630152E-07_rk, &
          -2.70988068537762022009086E-07_rk,        +3.0737622701407688440959E-08_rk, &
            +2.515620384817622937314E-09_rk,         -1.028929921320319127590E-09_rk, &
              +2.9944052119949939363E-11_rk,           +2.6051789687266936290E-11_rk, &
               -2.634839924171969386E-12_rk,             -6.43404509890636443E-13_rk, &
                +1.12457401801663447E-13_rk,              +1.7281533389986098E-14_rk, &
                  -4.264101694942375E-15_rk,                -5.45371977880191E-16_rk, &
                   +1.58697607761671E-16_rk,                 +2.0899837844334E-17_rk, &
                     -5.900526869409E-18_rk,                   -9.41893387554E-19_rk, &
                      +2.14977356470E-19_rk,                    +4.6660985008E-20_rk, &
                        -7.243011862E-21_rk,                     -2.387966824E-21_rk, &
                         +1.91177535E-22_rk,                      +1.20482568E-22_rk, &
                            -6.72377E-25_rk,                        -5.747997E-24_rk, &
                            -4.28493E-25_rk,                         +2.44856E-25_rk, &
                             +4.3793E-26_rk,                           -8.151E-27_rk, &
                              -3.089E-27_rk,                             +9.3E-29_rk, &
                               +1.74E-28_rk,                             +1.6E-29_rk, &
                                -8.0E-30_rk,                             -2.0E-30_rk /)
!
!
!   -∞ ≤ x ≤ limiterfc
         if( x <= limiterfc ) then               ! cut off limiterfc
             f = two                             ! f = 2.00
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ eMax
    else if( z <= eMax ) then
!            Chebyhev Approximation, page 1235, table 3
             t  = (z - k) / (z + k)              ! Chebyshev t
             u  = two * t
             d  = zero
             dd = zero
             do j = m, 2_ik, -1_ik               ! Clenshaw's recurr.
                sv = d; d = u*d - dd + c(j); dd = sv
             end do
             f = t*d - dd + c(1_ik)/two          ! finalize Clenshaw
             f = EXP( -z*z ) * f                 ! finalize erfc(z)
             if( x < zero ) f = two - f          ! erfc(-x)= 2.00 - erfc(x)
!
!   eMax < z=|x| ≤ +∞
    else                                         ! cut off z > eMax
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcSchonfelder
!
!-----------------------------------------------------------------------------
!
!   erfcShepherd
!     Complementary Error Function f = erfc(x) with Chebyshev
!     Approximation referenced Shepherd/Laframboise (1981)
!   Reference:
!     M.M. Shepherd and J.G. Laframboise, "Chebyshev Approximation of
!     (1 + 2x)exp(x²)erfc x in 0 ≤ x < ∞", Mathematics of Computation,
!     1981, Volume 36, Issue 153, Pages 249-253,
!     https://doi.org/10.1090/S0025-5718-1981-0595058-X,
!     here: page 250-251
!   Remark:
!     Chebyhev Approximation for f = (1 + 2x) * exp(x²) * erfc(x),
!     including back transformation to erfc(x) and the extension
!     for negative input of x
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "sqrteps" is defined in "Constants" for erfc,
!      where erfc( |x| ≤ sqrteps )   := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfc" is defined in "Constants" for erfc,
!      where erfc( x ≤ limiterfc )   := +2.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcShepherd( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, two, twodivsqrtpi, &
                      eMax, sqrteps, limiterfc
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfc(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z,  t, u      ! var. Chebyshev
    real   (rk)                 :: sv, d, dd     ! var. Clenshaw
    integer(ik)                 :: j             ! do ...
!
!   parameter page 252, table 1 for (1+2x)*exp(x²)*erfc(x):
    real   (rk), parameter      :: k             =   3.750E+00_rk
    integer(ik), parameter      :: m             =  31_ik
    real   (rk), parameter      :: c(1_ik:m)     =                (/ &
    0.1177578934567401754080E+01_rk, -0.45900545806464773310E-02_rk, &
     -0.84249133366517915584E-01_rk,  0.59209939998191890498E-01_rk, &
     -0.26658668435305752277E-01_rk,   0.9074997670705265094E-02_rk, &
      -0.2413163540417608191E-02_rk,    0.490775836525808632E-03_rk, &
        -0.69169733025012064E-04_rk,      0.4139027986073010E-05_rk, &
           0.774038306619849E-06_rk,      -0.218864010492344E-06_rk, &
            0.10764999465671E-07_rk,         0.4521959811218E-08_rk, &
             -0.775440020883E-09_rk,          -0.63180883409E-10_rk, &
               0.28687950109E-10_rk,             0.194558685E-12_rk, &
                -0.965469675E-12_rk,              0.32525481E-13_rk, &
                  0.33478119E-13_rk,              -0.1864563E-14_rk, &
                  -0.1250795E-14_rk,                 0.74182E-16_rk, &
                     0.50681E-16_rk,                 -0.2237E-17_rk, &
                     -0.2187E-17_rk,                    0.27E-19_rk, &
                        0.97E-19_rk,                     0.3E-20_rk, &
                        -0.4E-20_rk                                 /)
!
!
!   -∞ ≤ x ≤ limiterfc
         if( x <= limiterfc ) then               ! cut off limiterfc
             f = two                             ! f = 2.00
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ eMax
    else if( z <= eMax ) then
!            Chebyshev Approximation, page 250-251
             t  = (z - k) / (z + k)              ! Chebyshev t
             u  = two * t
             d  = zero
             dd = zero
             do j = m, 2_ik, -1_ik               ! Clenshaw's recurr.
                sv = d; d = u*d - dd + c(j); dd = sv
             end do
             f = t*d - dd + c(1_ik)              ! finalize Chebyshev
             f = f / ((one + two*z) * exp(z*z))  ! finalize erfc(z)
             if( x < zero ) f = two - f          ! erfc(-x)=2-erfc(x)
!
!   eMax < z=|x| ≤ +∞
    else                                         ! cut off z > eMax
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcShepherd
!
!-----------------------------------------------------------------------------
!
!   erfcSlatec
!     Complementary Error Function f = erfc(x) with Chebyshev Series
!     referenced Slatec (1993)
!   Reference:
!     Chebyshev series "c1, c2 and c3" and other coefficients. SLATEC Common 
!      Mathematical Library, Version 4.1, July 1993: A comprehensive software
!      library containing over 1400 general purpose mathematical and statist.
!      routines written in Fortran 77. Energy Science and Technology Software
!      Center, U.S. Department of Energy. The Library is in the public domain
!      and distributed by the Energy Science and Technology Software Center.
!      Energy Science and Technology Software Center, P.O. Box 1020 Oak Ridge,
!      TN 37831, Telephone 615-576-2606, 
!      E-mail: estsc%a1.adonis.mrouter@zeus.osti.gov . Quoted from
!      http://www.netlib.org/slatec/guide, Section 4. "Obtaining the Library"
!      http://www.netlib.org/slatec/ , Referenced function: "DERF" by
!      the author Fullerton
!     Evaluation of n-term Chebyshev series
!      R. Broucke, "Ten subroutines for the manipulation of Chebyshev series,
!      Algorithm 446", Communications of the ACM, Association for Computer
!      Machinery, 1973, Volume 16, Issue 4, Page 254-256,
!      https://doi.org/10.1145/362003.362037
!      here: recurrence formula (4) on page 254 f
!   Remark:
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "sqrteps" is defined in "Constants" for erf,
!      where erfc( |x| ≤ sqrteps )   := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfc" is defined in "Constants" for erfc,
!      where erfc( x ≤ limiterfc )   := +2.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcSlatec( x )   result( f )
!
    use kinds, only : compiler, ik, rki, rk
    use const, only : zero, half, one, two, three, five, eight, &
                      twodivsqrtpi, eMax, sqrteps, limiterfc
!
    implicit none
!
!   interface
     real  (kind=rk), intent(in )    :: x        ! x from erfc(x)
!   end interface
    real   (kind=rk)                 :: f        ! function result
!
!   local variable
    real   (kind=rk)                 :: z        ! absolute x
    real   (kind=rk)                 :: zz       ! (z=|x|)²
    real   (kind=rk)                 :: t, u     ! Cheby coefficients
    real   (kind=rk)                 :: b0,b1,b2 ! help var. Clenshaw
    integer(kind=ik)                 :: i        ! do ...
!
!   Chebyshev series coefficients (former "erfcs") for the 
!   interval 0.00 to  1.00, with weighted error 1.28..E-32
    integer(ik), parameter      :: d1            = 21_ik
    real   (rk), parameter      :: c1(1_ik:d1)   =                                       (/ &
    -0.49046121234691808039984544033376E-01_rk, -0.14226120510371364237824741899631E+00_rk, &
    +0.10035582187599795575754676712933E-01_rk, -0.57687646997674847650827025509167E-03_rk, &
    +0.27419931252196061034422160791471E-04_rk, -0.11043175507344507604135381295905E-05_rk, &
    +0.38488755420345036949961311498174E-07_rk, -0.11808582533875466969631751801581E-08_rk, &
    +0.32334215826050909646402930953354E-10_rk, -0.79910159470045487581607374708595E-12_rk, &
    +0.17990725113961455611967245486634E-13_rk, -0.37186354878186926382316828209493E-15_rk, &
    +0.71035990037142529711689908394666E-17_rk, -0.12612455119155225832495424853333E-18_rk, &
    +0.20916406941769294369170500266666E-20_rk, -0.32539731029314072982364160000000E-22_rk, &
    +0.47668672097976748332373333333333E-24_rk, -0.65980120782851343155199999999999E-26_rk, &
    +0.86550114699637626197333333333333E-28_rk, -0.10788925177498064213333333333333E-29_rk, &
    +0.12811883993017002666666666666666E-31_rk                                             /)

!
!   Chebyshev series coefficients (former "erc2cs") for the 
!   interval 2.50E-01 to 1.00E+00, with weighted error 2.67..E-32
    integer(ik), parameter      :: d2            = 49_ik
    real   (rk), parameter      :: c2(1_ik:d2)   =                                       (/ &
     -0.6960134660230950112739150826197E-01_rk,  -0.4110133936262089348982212084666E-01_rk, &
     +0.3914495866689626881561143705244E-02_rk,  -0.4906395650548979161280935450774E-03_rk, &
     +0.7157479001377036380760894141825E-04_rk,  -0.1153071634131232833808232847912E-04_rk, &
     +0.1994670590201997635052314867709E-05_rk,  -0.3642666471599222873936118430711E-06_rk, &
     +0.6944372610005012589931277214633E-07_rk,  -0.1371220902104366019534605141210E-07_rk, &
     +0.2788389661007137131963860348087E-08_rk,  -0.5814164724331161551864791050316E-09_rk, &
     +0.1238920491752753181180168817950E-09_rk,  -0.2690639145306743432390424937889E-10_rk, &
     +0.5942614350847910982444709683840E-11_rk,  -0.1332386735758119579287754420570E-11_rk, &
     +0.3028046806177132017173697243304E-12_rk,  -0.6966648814941032588795867588954E-13_rk, &
     +0.1620854541053922969812893227628E-13_rk,  -0.3809934465250491999876913057729E-14_rk, &
     +0.9040487815978831149368971012975E-15_rk,  -0.2164006195089607347809812047003E-15_rk, &
     +0.5222102233995854984607980244172E-16_rk,  -0.1269729602364555336372415527780E-16_rk, &
     +0.3109145504276197583836227412951E-17_rk,  -0.7663762920320385524009566714811E-18_rk, &
     +0.1900819251362745202536929733290E-18_rk,  -0.4742207279069039545225655999965E-19_rk, &
     +0.1189649200076528382880683078451E-19_rk,  -0.3000035590325780256845271313066E-20_rk, &
     +0.7602993453043246173019385277098E-21_rk,  -0.1935909447606872881569811049130E-21_rk, &
     +0.4951399124773337881000042386773E-22_rk,  -0.1271807481336371879608621989888E-22_rk, &
     +0.3280049600469513043315841652053E-23_rk,  -0.8492320176822896568924792422399E-24_rk, &
     +0.2206917892807560223519879987199E-24_rk,  -0.5755617245696528498312819507199E-25_rk, &
     +0.1506191533639234250354144051199E-25_rk,  -0.3954502959018796953104285695999E-26_rk, &
     +0.1041529704151500979984645051733E-26_rk,  -0.2751487795278765079450178901333E-27_rk, &
     +0.7290058205497557408997703680000E-28_rk,  -0.1936939645915947804077501098666E-28_rk, &
     +0.5160357112051487298370054826666E-29_rk,  -0.1378419322193094099389644800000E-29_rk, &
     +0.3691326793107069042251093333333E-30_rk,  -0.9909389590624365420653226666666E-31_rk, &
     +0.2666491705195388413323946666666E-31_rk                                             /)
!
!   Chebyshev series coefficients (former "erfccs") for the 
!   interval 0.00 to 2.50E-01, with weighted error 1.53..E-31
    integer(ik), parameter      :: d3            = 59_ik
    real   (rk), parameter      :: c3(1_ik:d3)   =                                       (/ &
      +0.715179310202924774503697709496E-01_rk,   -0.265324343376067157558893386681E-01_rk, &
      +0.171115397792085588332699194606E-02_rk,   -0.163751663458517884163746404749E-03_rk, &
      +0.198712935005520364995974806758E-04_rk,   -0.284371241276655508750175183152E-05_rk, &
      +0.460616130896313036969379968464E-06_rk,   -0.822775302587920842057766536366E-07_rk, &
      +0.159214187277090112989358340826E-07_rk,   -0.329507136225284321486631665072E-08_rk, &
      +0.722343976040055546581261153890E-09_rk,   -0.166485581339872959344695966886E-09_rk, &
      +0.401039258823766482077671768814E-10_rk,   -0.100481621442573113272170176283E-10_rk, &
      +0.260827591330033380859341009439E-11_rk,   -0.699111056040402486557697812476E-12_rk, &
      +0.192949233326170708624205749803E-12_rk,   -0.547013118875433106490125085271E-13_rk, &
      +0.158966330976269744839084032762E-13_rk,   -0.472689398019755483920369584290E-14_rk, &
      +0.143587337678498478672873997840E-14_rk,   -0.444951056181735839417250062829E-15_rk, &
      +0.140481088476823343737305537466E-15_rk,   -0.451381838776421089625963281623E-16_rk, &
      +0.147452154104513307787018713262E-16_rk,   -0.489262140694577615436841552532E-17_rk, &
      +0.164761214141064673895301522827E-17_rk,   -0.562681717632940809299928521323E-18_rk, &
      +0.194744338223207851429197867821E-18_rk,   -0.682630564294842072956664144723E-19_rk, &
      +0.242198888729864924018301125438E-19_rk,   -0.869341413350307042563800861857E-20_rk, &
      +0.315518034622808557122363401262E-20_rk,   -0.115737232404960874261239486742E-20_rk, &
      +0.428894716160565394623737097442E-21_rk,   -0.160503074205761685005737770964E-21_rk, &
      +0.606329875745380264495069923027E-22_rk,   -0.231140425169795849098840801367E-22_rk, &
      +0.888877854066188552554702955697E-23_rk,   -0.344726057665137652230718495566E-23_rk, &
      +0.134786546020696506827582774181E-23_rk,   -0.531179407112502173645873201807E-24_rk, &
      +0.210934105861978316828954734537E-24_rk,   -0.843836558792378911598133256738E-25_rk, &
      +0.339998252494520890627359576337E-25_rk,   -0.137945238807324209002238377110E-25_rk, &
      +0.563449031183325261513392634811E-26_rk,   -0.231649043447706544823427752700E-26_rk, &
      +0.958446284460181015263158381226E-27_rk,   -0.399072288033010972624224850193E-27_rk, &
      +0.167212922594447736017228709669E-27_rk,   -0.704599152276601385638803782587E-28_rk, &
      +0.297976840286420635412357989444E-28_rk,   -0.126252246646061929722422632994E-28_rk, &
      +0.539543870454248793985299653154E-29_rk,   -0.238099288253145918675346190062E-29_rk, &
      +0.109905283010276157359726683750E-29_rk,   -0.486771374164496572732518677435E-30_rk, &
      +0.152587726411035756763200828211E-30_rk                                             /)
!
!   coefficients n1, n2 and n3 to achieve the required precision according kind specification
    integer(ik),parameter :: pn1(3,2) = reshape( (/13,13,21, 13,13,15/), shape(pn1) )
    integer(ik),parameter ::  n1      = pn1( rki, compiler )
    integer(ik),parameter :: pn2(3,2) = reshape( (/25,25,49, 25,25,30/), shape(pn2) )
    integer(ik),parameter ::  n2      = pn2( rki, compiler )
    integer(ik),parameter :: pn3(3,2) = reshape( (/26,26,59, 26,26,33/), shape(pn3) )
    integer(ik),parameter ::  n3      = pn3( rki, compiler )
!
!
!   -∞ ≤ x ≤ limiterfc
         if( x <= limiterfc ) then               ! cut off f ≈ 2.0
             f = two                             ! f = 2.00
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| < ( sqrteps = SQRT(EPSILON(x)) )
         if( z < sqrteps ) then
! o          f = one - two * x / sqrtpi          ! original (slow)
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps ≤ z=|x| ≤ (one = 1.00E+00)
    else if( z <= one ) then
             t  = two * x*x - one; u  = two * t  ! n1-term Chebyshev series
             b0 = zero;            b1 = zero
             do i = n1,  1_ik, -1_ik
                b2 = b1; b1 = b0; b0 = u*b1 - b2 + c1(i)
             end do
             f = half * (b0 - b2)                ! finalize Clenshaw
             f = one - x * (one + f)             ! finalize calculation
!
!   (one = 1.00E+00) < z=|x| ≤ (two = 2.00E+00)
    else if( z <= two ) then
             zz = z * z                          ! (z=|x|)²
             t  = (eight/zz - five)/three        ! n2-term Chebyshev series
             u  = two * t
             b0 = zero;            b1 = zero
             do i = n2, 1_ik, -1_ik
                b2 = b1; b1 = b0; b0 = u*b1 - b2 + c2(i)
             end do
             f = half * (b0 - b2)                ! finalize Clenshaw
             f = EXP(-zz) / z * (half + f )      ! finalize calculation
             if( x < zero ) f = two - f          ! erfc(-x) = 2 - erfc(x)
!
!   2.00E+00 < |x| ≤ eMax
    else if( z <= eMax ) then
             zz = z * z                          ! (z=|x|)²
             t  = eight/zz - one;  u  = two * t  ! n3-term Chebyshev series
             b0 = zero;            b1 = zero
             do i = n3, 1_ik, -1_ik
                b2 = b1; b1 = b0; b0 = u*b1 - b2 + c3(i)
             end do
             f = half * (b0 - b2)                ! finalize Clenshaw
             f = EXP(-zz) / z * (half + f )      ! finalize calculation
             if( x < zero ) f = two - f          ! erfc(-x) = 2 - erfc(x)
!
!   eMax < z=|x| ≤ +∞
    else                                         ! cut off z > eMax
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculations
    end  if                                      ! end if cases
!
    return
    end function erfcSlatec
!
!-----------------------------------------------------------------------------
!
!   erfcSun
!     Complementary Error Function f = erfc(x) with several Polynomials
!     referenced Sun Microsystems (1995)
!   Reference:
!     Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
!     Developed at SunSoft, a Sun Microsystems, Inc. business. Permission to
!     use, copy, modify, and distribute this software is freely granted,
!     provided that this notice is preserved.
!     Original "C" routines at http://netlib.org/fdlibm/s_erf.c
!     @(#)s_erf.c 1.3 95/01/18
!   Remark:
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "limiterfc" is defined in "Constants" for erfc,
!      where erfc( x ≤ limiterfc )   := +2.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcSun( x )   result( f )
!
    use kinds, only : rk
    use const, only : zero, half, one, two, eMax, limiterfc
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfc(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: u, y          ! help variable
    real   (rk)                 :: r, s, p, q    ! help variable
!
!   mathematical & machine-dependent constants/parameters
!a  real   (rk), parameter      :: loctiny       = TINY(one)
!
!   coefficients for  approximation to erf in [2**(-56), 0.84375=9.0/16.0]
    real   (rk), parameter      ::                                           &
    pp0 =  1.28379167095512558561E-1_rk,pp1 = -3.25042107247001499370E-1_rk, &
    pp2 = -2.84817495755985104766E-2_rk,pp3 = -5.77027029648944159157E-3_rk, &
    pp4 = -2.37630166566501626084E-5_rk,                                     &
    qq1 =  3.97917223959155352819E-1_rk,qq2 =  6.50222499887672944485E-2_rk, &
    qq3 =  5.08130628187576562776E-3_rk,qq4 =  1.32494738004321644526E-4_rk, &
    qq5 = -3.96022827877536812320E-6_rk

!   coefficients for approximation to erf in [0.84375=9.0/16.0, 1.25=5.0/4.0]
    real   (rk), parameter      ::       erx =  8.45062911510467529297E-1_rk
!                               =  0.84506291151 rounded to single (24 bits)
    real   (rk), parameter      ::                                           &
    pa0 = -2.36211856075265944077E-3_rk,pa1 =  4.14856118683748331666E-1_rk, &
    pa2 = -3.72207876035701323847E-1_rk,pa3 =  3.18346619901161753674E-1_rk, &
    pa4 = -1.10894694282396677476E-1_rk,pa5 =  3.54783043256182359371E-2_rk, &
    pa6 = -2.16637559486879084300E-3_rk,                                     &
    qa1 =  1.06420880400844228286E-1_rk,qa2 =  5.40397917702171048937E-1_rk, &
    qa3 =  7.18286544141962662868E-2_rk,qa4 =  1.26171219808761642112E-1_rk, &
    qa5 =  1.36370839120290507362E-2_rk,qa6 =  1.19844998467991074170E-2_rk
!
!   coefficients for approximation to erfc in [1.25=5.0/4.0, 1.00/0.35]
    real   (rk), parameter      ::                                           &
    ra0 = -9.86494403484714822705E-3_rk,ra1 = -6.93858572707181764372E-1_rk, &
    ra2 = -1.05586262253232909814E+1_rk,ra3 = -6.23753324503260060396E+1_rk, &
    ra4 = -1.62396669462573470355E+2_rk,ra5 = -1.84605092906711035994E+2_rk, &
    ra6 = -8.12874355063065934246E+1_rk,ra7 = -9.81432934416914548592E+0_rk, &
    sa1 =  1.96512716674392571292E+1_rk,sa2 =  1.37657754143519042600E+2_rk, &
    sa3 =  4.34565877475229228821E+2_rk,sa4 =  6.45387271733267880336E+2_rk, &
    sa5 =  4.29008140027567833386E+2_rk,sa6 =  1.08635005541779435134E+2_rk, &
    sa7 =  6.57024977031928170135E+0_rk,sa8 = -6.04244152148580987438E-2_rk
!
!   coefficients for approximation to erfc in [1.00/0.35, eMax]
    real   (rk), parameter      ::                                           &
    rb0 = -9.86494292470009928597E-3_rk,rb1 = -7.99283237680523006574E-1_rk, &
    rb2 = -1.77579549177547519889E+1_rk,rb3 = -1.60636384855821916062E+2_rk, &
    rb4 = -6.37566443368389627722E+2_rk,rb5 = -1.02509513161107724954E+3_rk, &
    rb6 = -4.83519191608651397019E+2_rk,                                     &
    sb1 =  3.03380607434824582924E+1_rk,sb2 =  3.25792512996573918826E+2_rk, &
    sb3 =  1.53672958608443695994E+3_rk,sb4 =  3.19985821950859553908E+3_rk, &
    sb5 =  2.55305040643316442583E+3_rk,sb6 =  4.74528541206955367215E+2_rk, &
    sb7 = -2.24409524465858183362E+1_rk
!
!
!   -∞ ≤ x ≤ limiterfc
         if( x <= limiterfc ) then               ! cut off limiterfc
             f = two                             ! f = 2.00
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero=0.00E+00) ≤ z=|x| < (2**(-56)= 1.3877...E-17)
         if( z < 1.3877787807814456755E-0017_rk ) then
             f = one - x                         ! f = 1.00 - x
!
!   (2**(-56)= 1.3877...E-17) ≤ z=|x| < (0.25E+00 = 1.0/4.0)
    else if( z < 0.25E+00_rk ) then
             u = x*x
             r = pp0 + u*(pp1 + u*(pp2 + u*(pp3 +      &
                       u* pp4                        )))
             s = one + u*(qq1 + u*(qq2 + u*(qq3 +      &
                       u*(qq4 + u* qq5)              )))
             y = r/s
             f = one - (x + x*y)                 ! f = 1.00 - erf(x)
!
!   (0.25E+00 = 1.0/4.0) ≤ z=|x| < (0.84375E+00 = 27.0/32.0)
    else if( z < 0.84375E+00_rk ) then
             u = x*x
             r = pp0 + u*(pp1 + u*(pp2 + u*(pp3 +      &
                       u* pp4                        )))
             s = one + u*(qq1 + u*(qq2 + u*(qq3 +      &
                       u*(qq4 + u* qq5)              )))
             y = r/s
             r = x*y
             r = r + (x - half)
             f = half - r
! summary    f = half - ( x*y + (x - half) )     ! f = erfc(x)
!
!   (0.84375E+00 = 27.0/32.0) ≤ z=|x| < (1.25E+00 = 5.0/4.0)
    else if( z < 1.25E+00_rk ) then
             u = ABS(x) - one
             p = pa0 + u*(pa1 + u*(pa2 + u*(pa3 +      &
                       u*(pa4 + u*(pa5 + u* pa6    )))))
             q = one + u*(qa1 + u*(qa2 + u*(qa3 +      &
                       u*(qa4 + u*(qa5 + u* qa6)    ))))
             if( x > zero ) then
                 u = one - erx;   f = u   - p/q
             else
                 u = erx + p/q;   f = one + u
             end if
!
!   (1.25E+00 = 5.0/4.0) ≤ z=|x| < (1.00/0.35 = 2.8571...E+00)
    else if( z < 2.857142857E+00_rk ) then
             u = one / (z * z)
             r = ra0 + u*(ra1 + u*(ra2 + u*(ra3 +      &
                       u*(ra4 + u*(ra5 + u*(ra6 +      &
                       u* ra7                     ))))))
             s = one + u*(sa1 + u*(sa2 + u*(sa3 +      &
                       u*(sa4 + u*(sa5 + u*(sa6 +      &
                       u*(sa7 + u* sa8)           ))))))
             u = REAL( z )
             r = EXP( -u*u - 0.5625E+00_rk) * EXP( (u-z)*(u+z) + r/s )
!                            0.5625E+00 = 9.0/16.0
             if( x > zero ) then
                 f =       r/z
             else
                 f = two - r/z
             end if
!
!   (1.00/0.35) ≤ z=|x| ≤ eMax
    else if( z <= eMax ) then
             u = one / (z * z)
             r = rb0 + u*(rb1 + u*(rb2 + u*(rb3 +      &
                       u*(rb4 + u*(rb5 + u* rb6    )))))
             s = one + u*(sb1 + u*(sb2 + u*(sb3 +      &
                       u*(sb4 + u*(sb5 + u*(sb6 +      &
                       u* sb7                     ))))))
             u = REAL( z )
             r = EXP( -u*u - 0.5625E+00_rk ) * EXP( (u-z) * (u+z) + r/s)
!                            0.5625E+00 = 9.0/16.0
             if( x > zero ) then
                 f =       r/z
             else
                 f = two - r/z
             end if
!
!   eMax < z=|x| ≤ +∞
!   [no need := x is ALWAYS positive, therefore f = 2.0 - TINY(x) is omitted]
    else                                         ! cut off z > eMax
!no need     if( x > zero ) then
!a               f =       loctiny * loctiny     ! original with underflow
                 f =       zero                  ! f ≈ 0.0
!no need     else
!no need         f = two - loctiny               ! f ≈ 2.0
!no need     end if
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcSun
!
!-----------------------------------------------------------------------------
!
!   erfcOoura8a
!     Complementary Error Function f = erfc(x) with Rational Function
!     referenced Ooura (1996)
!   Reference:
!     Masatake Mori, "A Method for Evaluation of the Error Function of Real
!      and Complex Variable with High Relative Accuracy", Publications of the
!      Research Institute for Mathematical Sciences (PRIMS), Kyoto University
!      1983, Volume 19, Issue 3, Page 1081–1094,
!      https://doi.org/10.2977/PRIMS/1195182021
!     Copyright(C) 1996 Takuya Ooura, email:ooura@mmm.t.u-tokyo.ac.jp.
!      You may use, copy, modify this code for any purpose and without fee.
!      You may distribute this ORIGINAL package. 
!      Internet: http://www.kurims.kyoto-u.ac.jp/~ooura/gamerf.html
!      here: referenced file derfc.f, function derfc
!   Remark:
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "sqrteps" is defined in "Constants" for erfc,
!      where erfc( |x| ≤ sqrteps )   := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfc" is defined in "Constants" for erfc,
!      where erfc( x ≤ limiterfc )   := +2.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfcOoura8a( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, half, one, two, twodivsqrtpi, &
                      eMax, sqrteps, limiterfc
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfc(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: t, u          ! variable
!a  integer(ik)                 :: i             ! do ...
!
!   parameters for the Rational Function
    real   (rk), parameter      :: pa  =  3.97886080735226000E+00_rk
    real   (rk), parameter      :: p(0_ik:22_ik) =                 (/ &
            2.75374741597376782E-01_rk,   4.90165080585318424E-01_rk, &
            7.74368199119538609E-01_rk,   1.07925515155856677E+00_rk, &
            1.31314653831023098E+00_rk,   1.37040217682338167E+00_rk, &
            1.18902982909273333E+00_rk,   8.05276408752910567E-01_rk, &
            3.57524274449531043E-01_rk,   1.66207924969367356E-02_rk, &
           -1.19463959964325415E-01_rk,  -8.38864557023001992E-02_rk, &
            2.49367200053503304E-03_rk,   3.90976845588484035E-02_rk, &
            1.61315329733252248E-02_rk,  -1.33823644533460069E-02_rk, &
           -1.27223813782122755E-02_rk,   3.83335126264887303E-03_rk, &
            7.73672528313526668E-03_rk,  -8.70779635317295828E-04_rk, &
           -3.96385097360513500E-03_rk,   1.19314022838340944E-04_rk, &
            1.27109764952614092E-03_rk                                /)
!
!
!   -∞ ≤ x ≤ limiterfc
         if( x <= limiterfc ) then               ! cut off limiterfc
             f = two                             ! f = 2.00
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ eMax
    else if( z <= eMax ) then
             t = pa / ( pa + z )
             u = t - half
             f = EXP( -z*z ) * t *                         &
               & ( p(0_ik) + u*( p( 1_ik) + u*(p( 2_ik) +  &
               &             u*( p( 3_ik) + u*(p( 4_ik) +  &
               &             u*( p( 5_ik) + u*(p( 6_ik) +  &
               &             u*( p( 7_ik) + u*(p( 8_ik) +  &
               &             u*( p( 9_ik) + u*(p(10_ik) +  &
               &             u*( p(11_ik) + u*(p(12_ik) +  &
               &             u*( p(13_ik) + u*(p(14_ik) +  &
               &             u*( p(15_ik) + u*(p(16_ik) +  &
               &             u*( p(17_ik) + u*(p(18_ik) +  &
               &             u*( p(19_ik) + u*(p(20_ik) +  &
               &             u*( p(21_ik) + u* p(22_ik)    &
               &              )))))))))))    ))))))))))    )
!
!a           alternative calculation
!a           t = pa / ( pa + z )
!a           u = t - half
!a           f = zero
!a           do i = 22_ik, 1_ik, -1_ik
!a              f = u * (p(i) + f)
!a           end do
!a           f = EXP( -z*z ) * t * ( p(0_ik) + f )
!
             if( x < zero ) f = two - f          ! erfc(-x) = 2 - erfc(x)
!
!   eMax < z=|x| ≤ +∞
    else                                         ! cut off z > eMax
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcOoura8a
!
!-----------------------------------------------------------------------------
!
!   erfcOoura8b
!     Complementary Error Function f = erfc(x) with Rational Function
!     referenced Ooura (1996)
!   Reference:
!     Masatake Mori, "A Method for Evaluation of the Error Function of Real
!      and Complex Variable with High Relative Accuracy", Publications of the
!      Research Institute for Mathematical Sciences (PRIMS), Kyoto University
!      1983, Volume 19, Issue 3, Page 1081–1094,
!      https://doi.org/10.2977/PRIMS/1195182021
!     Copyright(C) 1996 Takuya Ooura, email:ooura@mmm.t.u-tokyo.ac.jp.
!      You may use, copy, modify this code for any purpose and without fee.
!      You may distribute this ORIGINAL package. 
!      Internet: http://www.kurims.kyoto-u.ac.jp/~ooura/gamerf.html
!      here: referenced file gamerf2a.f, function derfc
!   Remark:
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "sqrteps" is defined in "Constants" for erfc,
!      where erfc( |x| ≤ sqrteps )   := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfc" is defined in "Constants" for erfc,
!      where erfc( x ≤ limiterfc )   := +2.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcOoura8b( x )   result( f )
!
    use kinds, only : rk
    use const, only : zero, one, two, twodivsqrtpi, &
                      eMax, sqrteps, limiterfc
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfc(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: y             ! x²
!
!   parameters for the Rational Function
    real   (rk), parameter      ::                                      &
     pv = 1.26974899965115684E+01_rk, ph = 6.10399733098688199E+00_rk,  &
     p0 = 2.96316885199227378E-01_rk, p1 = 1.81581125134637070E-01_rk,  &
     p2 = 6.81866451424939493E-02_rk, p3 = 1.56907543161966709E-02_rk,  &
     p4 = 2.21290116681517573E-03_rk, p5 = 1.91395813098742864E-04_rk,  &
     p6 = 9.71013284010551623E-06_rk, p7 = 1.66642447174307753E-07_rk,  &
     q0 = 6.12158644495538758E-02_rk, q1 = 5.50942780056002085E-01_rk,  &
     q2 = 1.53039662058770397E+00_rk, q3 = 2.99957952311300634E+00_rk,  &
     q4 = 4.95867777128246701E+00_rk, q5 = 7.41471251099335407E+00_rk,  &
     q6 = 1.04765104356545238E+01_rk, q7 = 1.48455557345597957E+01_rk
!
!
!   -∞ ≤ x ≤ limiterfc
         if( x <= limiterfc ) then               ! cut off limiterfc
             f = two                             ! f = 2.00
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ eMax
    else if( z <= eMax ) then
             y = x * x                           ! y = x²
             f = EXP( -y ) * x *                    &
               & ( p7 / (y + q7) + p6 / (y + q6) +  &
               &   p5 / (y + q5) + p4 / (y + q4) +  &
               &   p3 / (y + q3) + p2 / (y + q2) +  &
               &   p1 / (y + q1) + p0 / (y + q0)    )
             if( x <  ph ) f = f + two / ( EXP( pv*x ) + one )
!
!   eMax < z=|x| ≤ +∞
    else                                         ! cut off z > eMax
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcOoura8b
!
!-----------------------------------------------------------------------------
!
!   erfcOoura16
!     Complementary Error Function f = erfc(x) with Polynomials referenced
!     Ooura (1996)
!   Reference:
!     Masatake Mori, "A Method for Evaluation of the Error Function of Real
!      and Complex Variable with High Relative Accuracy", Publications of the
!      Research Institute for Mathematical Sciences (PRIMS), Kyoto University
!      1983, Volume 19, Issue 3, Page 1081–1094,
!      https://doi.org/10.2977/PRIMS/1195182021
!     Copyright(C) 1996 Takuya Ooura, email:ooura@mmm.t.u-tokyo.ac.jp.
!      You may use, copy, modify this code for any purpose and without fee.
!      You may distribute this ORIGINAL package. 
!      Internet: http://www.kurims.kyoto-u.ac.jp/~ooura/gamerf.html
!      here: referenced file gamerf2a.f, function qerfc
!   Remark:
!     This version is especially developed for quadruple precision.
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "sqrteps" is defined in "Constants" for erfc,
!      where erfc( |x| ≤ sqrteps )   := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfc" is defined in "Constants" for erfc,
!      where erfc( x ≤ limiterfc )   := +2.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcOoura16( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, two, twodivsqrtpi, &
                      eMax, sqrteps, limiterfc
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfc(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: y             ! x²
!a  integer(ik)                 :: i             ! do ...
!
!   Parameter for the Polynomials
    real   (rk), parameter      ::                           &
       pv = 1.8296570980424689847157930974106706835E+01_rk,  &
       ph = 8.9588287394342176848213494031807385567E+00_rk
    real   (rk), parameter      :: p(0_ik:16_ik) =                                               (/ &
    2.1226887418241545314975570224238841543E-01_rk, 1.6766968820663231170102487414107148110E-01_rk, &
    1.0461429607758480243524362040994242137E-01_rk, 5.1557963860512142911764627378588661742E-02_rk, &
    2.0070986488528139460346647533434778000E-02_rk, 6.1717726506718148117513762897928828534E-03_rk, &
    1.4990611906920858646769185063310410160E-03_rk, 2.8760540416705806615617926157307107830E-04_rk, &
    4.3585593590380741491013549969419946961E-05_rk, 5.2174364856655433775383935118049845471E-06_rk, &
    4.9333351722974670085736982894474122277E-07_rk, 3.6846914376723888190666722894010079935E-08_rk, &
    2.1729515092764086499231043367920037215E-09_rk, 9.9870022842895735663712411206346261651E-11_rk, &
    3.1775163189596489863458236395414830880E-12_rk, 4.5657943993597540327708145643160878201E-14_rk, &
    1.1940964427370412648558173558044106203E-16_rk                                                 /)
    real   (rk), parameter      :: q(0_ik:16_ik) =                                               (/ &
    2.9482230394292049252878077330764031337E-02_rk, 2.6534007354862844327590269604581049764E-01_rk, &
    7.3705575985730123132195272141160572532E-01_rk, 1.4446292893203104133929687855854497896E+00_rk, &
    2.3880606619376559912235584857800710490E+00_rk, 3.5673498777093386979273977202889759348E+00_rk, &
    4.9824969366355296879760903991854492762E+00_rk, 6.6335018387405633238409855625402006223E+00_rk, &
    8.5203645862651289478197632097553870199E+00_rk, 1.0643085317662274170216548777166393329E+01_rk, &
    1.3001669850030489723387515813223808078E+01_rk, 1.5596282517377690399267249728222735970E+01_rk, &
    1.8429903207271748464995406180854691072E+01_rk, 2.1533907893494593530979123915138686107E+01_rk, &
    2.5076752889217226137869837117288885077E+01_rk, 2.9515380437412601845256918753602002410E+01_rk, &
    3.5792848810704122499184545805923520658E+01_rk                                                 /)
!
!
!   -∞ ≤ x ≤ limiterfc
         if( x <= limiterfc ) then               ! cut off limiterfc
             f = two                             ! f = 2.00
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ eMax
    else if( z <= eMax ) then
             y = x * x                           ! y = x²
             f = EXP( -y ) * x *                               ( &
               & p(16_ik)/(y+q(16_ik)) + p(15_ik)/(y+q(15_ik)) + &
               & p(14_ik)/(y+q(14_ik)) + p(13_ik)/(y+q(13_ik)) + &
               & p(12_ik)/(y+q(12_ik)) + p(11_ik)/(y+q(11_ik)) + &
               & p(10_ik)/(y+q(10_ik)) + p( 9_ik)/(y+q( 9_ik)) + &
               & p( 8_ik)/(y+q( 8_ik)) + p( 7_ik)/(y+q( 7_ik)) + &
               & p( 6_ik)/(y+q( 6_ik)) + p( 5_ik)/(y+q( 5_ik)) + &
               & p( 4_ik)/(y+q( 4_ik)) + p( 3_ik)/(y+q( 3_ik)) + &
               & p( 2_ik)/(y+q( 2_ik)) + p( 1_ik)/(y+q( 1_ik)) + &
               & p( 0_ik)/(y+q( 0_ik))                         )
             if( x < ph ) f = f + two / ( EXP( pv*x ) + one )
!altern.     y = z * z                           ! y = z²
!a           f = zero
!a           do i = 0_ik, 16_ik, 1_ik
!a              f = f  +  p(i) / (y + q(i))
!a           end do
!a           f = EXP( -y ) * x * f
!a           if( z < ph ) f = f + two / ( EXP( pv*x ) + one )
!
!   eMax < z=|x| ≤ +∞
    else                                         ! cut off z > eMax
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcOoura16
!
!-----------------------------------------------------------------------------
!
!   erfcDia
!     Complementary Error Function f = erfc(x) with Mill's Ratio referenced
!     Dia (2023)
!   Reference:
!     Yaya D. Dia, "Approximate Incomplete Integrals, Application to
!     Complementary Error Function", 2023, June 21, paper, 15 pages, 
!     Available at SSRN: https://ssrn.com/abstract=4487559 or
!     https://doi.org/10.2139/ssrn.4487559,
!     here: page 7, formula (61); page 9, formula(69) and page 10, table 1
!   Remarks:
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "sqrteps" is defined in "Constants" for erfc,
!      where erfc( |x| ≤ sqrteps )   := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfc" is defined in "Constants" for erfc,
!      where erfc( x ≤ limiterfc )   := +2.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcDia( x )   result ( f )
!
    use kinds, only : rk
    use const, only : zero, one, two,                     &
                      sqrtpi, onedivsqrtpi, twodivsqrtpi, &
                      eMax, sqrteps, limiterfc
!a  use const, only : twodivpi
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfc(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z,  zz        ! z=|x|, z*z
!a  real   (rk)                 :: z2, zz2       ! original z and zz
    real   (rk)                 :: M             ! Mill's ratio (M)
!
!   calculation of M (Mills ratio), page 9, formula (69) with:
!   a) parameters for M on page 10 in table 1, beginning with "o..."
!   b) parameters eliminating "√2 * x" in the interface of M with modified
!      parameters, beginning with "m..."
    real   (rk), parameter      ::                                        &
      ob00 =  2.92678600515804815402E+00_rk,  mb00 = ob00 / SQRT( two ),  &
      ob11 =  8.97280659046817350354E+00_rk,  mb11 = ob11 /       two,    &
      ob12 = 10.27157061171363078863E+00_rk,  mb12 = ob12 /       two,    &
      ob13 = 12.72323261907760928036E+00_rk,  mb13 = ob13 /       two,    &
      ob14 = 16.88639562007936907786E+00_rk,  mb14 = ob14 /       two,    &
      ob15 = 24.12333774572479110372E+00_rk,  mb15 = ob15 /       two,    &
      ob21 =  5.81582518933527390512E+00_rk,  mb21 = ob21 / SQRT( two ),  &
      ob22 =  5.70347935898051436684E+00_rk,  mb22 = ob22 / SQRT( two ),  &
      ob23 =  5.51862483025707963145E+00_rk,  mb23 = ob23 / SQRT( two ),  &
      ob24 =  5.26184239579604207321E+00_rk,  mb24 = ob24 / SQRT( two ),  &
      ob25 =  4.92081346632882032881E+00_rk,  mb25 = ob25 / SQRT( two ),  &
      oc11 = 11.61511226260603247078E+00_rk,  mc11 = oc11 /       two,    &
      oc12 = 18.25323235347346524796E+00_rk,  mc12 = oc12 /       two,    &
      oc13 = 18.38871225773938486923E+00_rk,  mc13 = oc13 /       two,    &
      oc14 = 18.61193318971775795045E+00_rk,  mc14 = oc14 /       two,    &
      oc15 = 24.14804072812762821134E+00_rk,  mc15 = oc15 /       two,    &
      oc21 =  3.83362947800146179416E+00_rk,  mc21 = oc21 / SQRT( two ),  &
      oc22 =  7.30756258553673541139E+00_rk,  mc22 = oc22 / SQRT( two ),  &
      oc23 =  8.42742300458043240405E+00_rk,  mc23 = oc23 / SQRT( two ),  &
      oc24 =  5.66479518878470764762E+00_rk,  mc24 = oc24 / SQRT( two ),  &
      oc25 =  4.91396098895240075156E+00_rk,  mc25 = oc25 / SQRT( two )
!
!
!   -∞ ≤ x ≤ limiterfc
         if( x <= limiterfc ) then               ! cut off limiterfc
             f = two                             ! f = 2.00
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ eMax
    else if( z <= eMax ) then
!            modified calculation M (Mills ratio), page 9, formula(69),
!            using z in the interface
             zz = z * z                          ! zz = z² = |x|²
             M  = (      onedivsqrtpi      / (      z + mb00      )   ) *  &
                  ( (zz  + mc21*z  + mc11) / (zz  + mb21*z  + mb11)   ) *  &
                  ( (zz  + mc22*z  + mc12) / (zz  + mb22*z  + mb12)   ) *  &
                  ( (zz  + mc23*z  + mc13) / (zz  + mb23*z  + mb13)   ) *  &
                  ( (zz  + mc24*z  + mc14) / (zz  + mb24*z  + mb14)   ) *  &
                  ( (zz  + mc25*z  + mc15) / (zz  + mb25*z  + mb15)   )
!            finalize erfc(x), page 7, formula (61) with modified parameters
             f = EXP( -zz )  *  M                ! f = erfc( x)
!
!alternative original M (Mills ratio), page 9, formula (69)
!a           z2  = SQRT( two ) * z               ! interface to M
!a           zz2 = z2 * z2                       ! square interface
!a           M   = (         one            / (           z2 + ob00)   ) *  &
!a                 ( (zz2 + oc21*z2 + oc11) / (zz2 + ob21*z2 + ob11)   ) *  &
!a                 ( (zz2 + oc22*z2 + oc12) / (zz2 + ob22*z2 + ob12)   ) *  &
!a                 ( (zz2 + oc23*z2 + oc13) / (zz2 + ob23*z2 + ob13)   ) *  &
!a                 ( (zz2 + oc24*z2 + oc14) / (zz2 + ob24*z2 + ob14)   ) *  &
!a                 ( (zz2 + oc25*z2 + oc15) / (zz2 + ob25*z2 + ob15)   )
!a           finalize erfc(x), page 7, formula (61)
!a           f = SQRT( twodivpi )  *  EXP( -(z*z) )  *  M
!
             if( x < zero ) f = two - f          ! erfc(-x) = 2 - erfc(x)
!
!   eMax < z=|x| ≤ +∞
    else                                         ! cut off z > eMax
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcDia
!
!-----------------------------------------------------------------------------
!