!
!   Remark:
!   1.
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   Complementary Error Function (erfc): Power Series / Series Expansions
!
!   real(rk) function erfcSE          ( x, it )   result( f )  Zhang/Jin            (1996),  Power      Series Expansion
!   real(rk) function erfcSEasymp     ( x, it )   result( f )  SE 1: Decker         (1975),  asymptotic Series Expansion
!                                                              SE 2: Ferreira et al.(2023),  asymptotic Series Expansion (Pochhammer Symbol)
!                                                              SE 3: Zhang/Jin      (1996),  asymptotic Series Expansion
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   erfcSE
!     Complementary Error Function f = erfc(x) with Power Series
!     Expansion referenced Zhang/Jin (1996)
!   Reference:
!     Shanjie Zhang and Jianming Jin, "Computation of Special Functions",
!     1996, XXVI, 717 pages, John Wiley & Sons Inc., New York,
!     ISBN: 0-471-11963-6;
!     here: page 621, formula (16.1.4)
!   Remark:
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfcSE( x, it )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, half, one, two, eps,  &
                      twodivsqrtpi, eMax, limiterfc
!
    implicit none
!
!   interface
     real   (rk), intent(in )   :: x             ! x from erfc(x)
     integer(ik), intent(out)   :: it            ! iterations
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: zz            ! zz = z² = |x|²
!
!   local variable
    real   (rk)                 :: q1            ! variables denominator
    real   (rk)                 :: s             ! incremental sum
!
!   mathematical & machine-dependent constants/parameters
    integer(kind=ik), parameter      :: itmax    = 200_ik
!
!
    it    = 0_ik                                 ! no iterations
!
!   -∞ ≤ x < (normally limiterfc), but here "-eMax" to show 
!   the symptotic values of the Series Expansion
!   -∞ ≤ x < -eMax
         if( x < -eMax ) then                    ! cut off f ≈ 2.0
             f = two - TINY(x)                   ! 2.00 - TINY
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   z=|x| == zero  =>  erfc( [z=|x|] ≤ zero ) = one
         if( z <= zero ) then                    ! x = 0.00
             f = one                             ! f = 1.00
!
!   0.00E+00 < z=|x| ≤ eMax
    else if( z <= eMax ) then
!            Power Series Expansion, Zhang/Jin (1996), p. 621, form. (16.1.4)
!
!                           2                   ∞  2^n * z^(2n+1)
!            erfc(z) = 1 - --- * exp[-z²] *     ∑  --------------
!                          √pi                 n=0   (2n+1)!!
!            or
!                          2*z                  ∞   n     z²
!            erfc(z) = 1 - --- * exp[-z²] *[1 + ∑   ∏  ---------]
!                          √pi                 n=1 k=1 (k + 1/2)
!            or
!                          2*z                     z^2   z^2  z^2  z^2
!            erfc(z) = 1 - --- * exp[-z²] *[1 + (((--- * ---)*---)*---) * ..]
!                          √pi                     1.5   2.5  3.5  4.5
!
             zz = z*z                            ! zz = z² = |x|²
             q1 = half                           ! [k=0 + (q1=0.5)]
             s  = one                            ! product increment
             f  = one                            ! f = 1 + (sum=0)
             do while ( it < itmax ); it=it+1_ik ! iterations
                q1  = q1 + one                   ! [(k>0) + 0.5]
                s   = s  * zz / q1               ! s * z²/(k+0.5)
                f   = f  + s                     ! sum for (n≥0)
                if( ABS(s) <= ABS(f)*eps ) exit
             end do
             f = one - twodivsqrtpi*x*EXP(-zz)*f ! finalize f, -/+x
!
!   eMax < z=|x| ≤ +∞
    else                                         ! cut off z > eMax
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcSE
!
!-----------------------------------------------------------------------------
!
!   erfcSEasymp
!     Complementary Error Function f = erfc(x) with asymptotic Series
!     Expansion, referenced by various sources, Hoering (2023)
!   Reference:
!     selec == 1
!      D.L. Decker, "Computer evaluation of the complementary error function",
!       American Journal of Physics, 1975, Volume 43, Issue 9, Pages 833–834,
!       https://doi.org/10.1119/1.9699,
!       here: page 833, formula (2)
!      J.M. Blair, C. A. Edwards, J. H. Johnson, "Rational Chebyshev
!       Approximations for the Inverse of the Error Function", Mathematics of
!       Computation, October 1976, Volume 30, Issue 136, pp 827-830,
!       https://doi.org/10.2307/2005402,
!       here:  page 828, formula (1)
!      M.M. Shepherd and J.G. Laframboise, "Chebyshev Approximation of 
!       (1 + 2x)exp(x²)erfc x in 0 ≤ x < ∞", Mathematics of Computation, 1981,
!       Volume 36, Issue 153, Pages 249-253, 
!       https://doi.org/10.1090/S0025-5718-1981-0595058-X,
!       here: page 251, formula (8), z ≤ 2.83
!      S. Winitzki, "Introduction to asymptotic series", 30 June 2006,
!       4 pages, https://sites.google.com/site/winitzki/sergei-winitzkis-files
!       here: page 4, formula (17)
!
!     selec == 2
!      Chelo Ferreira, José L. López and Ester Pérez Sinusía, "Uniform
!       Convergent Expansions of the Error Function in Terms of Elementary
!       Functions", Mediterranean Journal of Mathematics (MedJM), 2023,
!       Volume 20, Article Number 117, 10 pages,
!       https://doi.org/10.1007/s00009-023-02297-2,
!       here: page 2, formula (4) and (5)
!
!     selec == 3
!      Shanjie Zhang and Jianming Jin, "Computation of Special
!       Functions", 1996, XXVI, 717 pages, John Wiley & Sons Inc.,
!       New York, ISBN: 0-471-11963-6,
!       here: page 621, formula (16.1.5)
!   Remark:
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Cut off at [ABS(x) < 0.25] with f = 0.0 to avoid underflow
!      and upper limits at "-eMax" and "+eMax" to show the asymptotic
!      expansion.
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfcSEasymp( x, it )   result( f )
!
    use kinds, only : ik, rki, rk
    use const, only : zero, half, one,  two, eps,  &
                      sqrtpi, eMax
!
    implicit none
!
!   interface
     real   (rk), intent(in )   :: x             ! x from erfc(x)
     integer(ik), intent(out)   :: it            ! iterations
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: zz            ! zz = z² = |x|²
    real   (rk)                 :: p1, p2        ! variables numerator
    real   (rk)                 :: q1, q2        ! variables denominator
    real   (rk)                 :: s             ! incremental sum
    integer(ik), save           :: selec         ! selection
!
!   mathematical & machine-dependent constants/parameters
    integer(ik)                 :: itmax         ! max. number of iterations
    integer(ik), parameter      ::                     & ! vector for itmax
       it1(1_ik:3_ik) = (/  18_ik, 100_ik, 200_ik /),  & ! sp, dp, ep/qp
       it2(1_ik:3_ik) = (/  18_ik, 100_ik, 200_ik /),  & ! sp, dp, ep/qp
       it3(1_ik:3_ik) = (/  18_ik, 100_ik, 200_ik /)     ! sp, dp, ep/qp
!
!
    if( x <= -HUGE(x) .and. x >= -HUGE(x) ) then
        write( *,                                                 '(  &
         & t2,"choose selec <1,2 or 3>:",                          /, &
         & t2,"1: Asymp. SE: Decker          (1975), itmax = ",i3, /, &
         & t2,"2: Asymp. SE: Ferreira et al. (2023), itmax = ",i3, /, &
         & t2,"3: Asymp. SE: Zhang/Jin       (1996), itmax = ",i3  )' ) &
         &                                   it1(rki), it2(rki), it3(rki)
        read(*,*) selec
    end if
!
    it = 0_ik                                    ! no iterations
!
!   -∞ ≤ x < (normally limiterfc), but here "-eMax" to show 
!   the symptotic values of the Series Expansion
!   -∞ ≤ x < -eMax
         if( x < -eMax ) then                    ! ! cut off x < -eMax
             f = two                             ! f = 2.00
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   z=|x| == zero  =>  normally "erfc( [z=|x|] ≤ zero ) = one",
!   but to avoid an underflow:  "erfc( [z=|x|] < 0.25 ) = one"
         if( z < 0.25E+00_rk ) then              ! x ≈ 0.00
             f = one                             ! f = 1.00
!
!   ≈0.00E+00 < z=|x| ≤ eMax
    else if( z <= eMax ) then
!
             if( selec == 1_ik ) then
!            asymptotic Series Expansion, referenced e.g.
!            Decker (1975), page 833, formula (2) or  Blair et al. (1976),
!            page 828, formula (1) or Shepherd/Laframboise (1981), p. 251,
!            f. (8), z ≤ 2.83 or Winitzki (2006), p. 4, f. (17)
!
!                      exp[-z²]         ∞ (-1)^n * (2n-1)!!
!            erfc(z) = -------- * [ 1 + ∑ ----------------- ]
!                      √pi * z         n=1     (2z²)^n
!            or
!                      exp[-z²]         ∞ (-1)^n * 1*3*5...*(2n-1)
!            erfc(z) = -------- * [ 1 + ∑ ------------------------ ]
!                      √pi * z         n=1     (2z²)^n
!
!            with the 1+sum in detail and odd double factorial(!!):
!
!              ∞            1        1*3      1*3*5    1*3*5*7
!            1+∑ = [ 1 - ------- + ------- - ------- + ------- - ... ]
!             n=1        (2z²)^1   (2z²)^2   (2z²)^3   (2z²)^4
!            
!              ∞            1         3        15        105
!            1+∑ = [ 1 - ------- + ------- - ------- + ------- - ... ]
!             n=1         2z^2       4z^4     8z^6      16z^8
!
             itmax = it1( rki )                  ! max. iterations
             zz    =  z*z                        ! zz = z² = |x|²
             p1    = -one                        ! (2*(n=0) - 1)
             p2    =  one                        ! 0!! = 1
             q1    =  two * zz                   ! 2z² = 2|x|²
             q2    =  one                        ! (2z²)^(n=0)
             f     =  one                        ! sum for (n=0)
             do while ( it < itmax ); it=it+1_ik ! iterations (n≥1)
                p1 =  p1 + two                   ! (2n-1)=1,3,5,7,...
                p2 = -p2 * p1                    ! -/+ (2n-1)!!
                q2 =  q2 * q1                    ! (2z²)^(n=1,2,3,...)
                s  =  p2 / q2                    ! increment s[n≥1]
                f  =  f  + s                     ! sum for (n≥1)
                if( ABS(s) <= ABS(f)*eps ) exit  ! precision achieved
             end do
             f = f / (EXP(zz) * sqrtpi * z)      ! finalize f, x ≥ 0
             if( x < zero ) f = two - f          ! erfc(-x) = 2 - erfc(x)
             end if                              ! end if selec == 1
!
!
             if( selec == 2_ik ) then
!            asymptotic Series Expansion with Pochhammer symbol, referenced
!            Ferreira et al. (2023), page 2, formula (4) and (5)
!
!                                               (1)
!                                               (-) <- Pochhammer Symbol
!                      exp[-z²]    ∞   (-1)^n * (2)n
!            erfc(z) = -------- *  ∑   -----------------
!                        √pi      n=0      z^(2n+1)
!
!                      exp[-z²]    ∞            (2n-1)!!      1
!            erfc(z) = -------- *  ∑   (-1)^n * -------- * --------
!                        √pi      n=0             2^n      z^(2n+1)
!
!                      exp[-z²]     1      1     1*3   1*3*5
!            erfc(z) = -------- * [--- - ---- + ---- - ----- + ...]
!                        √pi       z^1   2z^3   4z^5    8z^7
!
             itmax = it2( rki )                  ! max. iterations
             zz    =  z*z                        ! zz = z² = |x|²
             p1    = -one                        ! [2(n=0)-1]
             p2    =  one                        ! p1!!=(-1)!!=1
             q1    =  one                        ! 2^(n=0)
             q2    =  z                          ! z^[2(n=0)+1]
             f     =  one / q2                   ! sum for (n=0)
             do while ( it < itmax ); it=it+1_ik ! iterations (n≥1)
                p1 =  p1 + two                   ! [2(n≥1)-1]
                p2 = -p2 * p1                    ! -/+ (2(n≥1)-1)!!
                q1 =  q1 * two                   ! 2^(n≥1)
                q2 =  q2 * zz                    ! z^(2(n≥1)+1)
                s  =  p2 / (q1 * q2)             ! increment s[n≥1]
                f  =  f  + s                     ! sum for (n≥1)
                if( ABS(s) <= ABS(f)*eps ) exit  ! precision achieved
             end do
             f = f / (EXP(zz) * sqrtpi)          ! finalize f, x ≥ 0
             if( x < zero ) f = two - f          ! erfc(-x) = 2 - erfc(x)
             end if                              ! end if selec == 2
!
!
             if( selec == 3_ik ) then
!            asymptotic Series Expansion, referenced e.g.
!            Zhang/Jin (1996), page 621, formula (16.1.5)
!
!                      exp[-z²]         ∞  (-1)^n * (2n-1)!!
!            erfc(z) = -------- *       ∑  -----------------
!                        √pi           n=0      (2z²)^n
!
!            or as formula (16.1.5)b
!                      exp[-z²]         ∞   n  -(2k-1)
!            erfc(z) = -------- * [ 1 + ∑   ∏  -------- ]
!                      √pi * z         n=1 k=1    2z²
!
!            or reformulated
!                      exp[-z²]         ∞   n  -(k-1/2)
!            erfc(z) = -------- * [ 1 + ∑   ∏  -------- ]
!                      √pi * z         n=1 k=1     z²
!
!            or
!                      exp[-z²]           -0.5   -1.5    -2.5    -3.5
!            erfc(z) = -------- * [ 1 +(((---- * ----) * ----) * ----) * ...]
!                      √pi * z             z²     z²      z²      z²
!
             itmax = it3( rki )                  ! max. iterations
             zz    =  z*z                        ! zz = z² = |x|²
             p1    = -half                       ! -[(k=0)-1/2]
             s     =  one                        ! increment = 1*
             f     =  one                        ! sum for (n=0)
             do while( it < itmax ); it=it+1_ik  ! iterations (n≥1)
                 p1 =  p1 + one                  ! [k-1/2]=0.5,1.5,...
                 s  = -s  * p1 / zz              ! -/+ s * (k-1/2)/z²
                 f  =  f  + s                    ! sum for (n≥1)
                 if( ABS(s) <= ABS(f)*eps ) exit ! precision achieved
             end do
             f = f / (EXP(zz) * sqrtpi * z)      ! finalize f, x ≥ 0
             if( x < zero ) f = two - f          ! erfc(-x) = 2 - erfc(x)
             end if                              ! end if selec == 3
!
!
!   eMax < z=|x| ≤ +∞
    else                                         ! cut off z > eMax
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcSEasymp
!
!-----------------------------------------------------------------------------
!