!
!   Remark:
!   1. ./.
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   Complementary Error Function (erfc): Continued Fraction (CF)
!
!   real(rk) function erfcCF          ( x, it )   result( f )  Hoering              (2023)
!                                                              CF 1: Strecok        (1968),  page 145, formula (3) and (4)
!                                                              CF 2: Stegun/Zucker  (1970),  page 214, 4th formula
!                                                              CF 3: Fettis         (1974),  page 585 and
!                                                                    Fettis         (1975),  page 673
!                                                              CF 4: Keisan         (2023),  webpage
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   erfcCF
!     Complementary Error Function f = erfc(x) with Continued Fraction
!     referenced by various sources, Hoering (2023)
!   Reference:
!     selec == 1
!      Anthony J. Strecok, "On the Calculation of the Inverse of
!       the Error Function", Mathematics of Computation, Jan 1968,
!       Volume 22, Issue 101, pp 144-158,
!       https://doi.org/10.1090/S0025-5718-1968-0223070-2,
!       here: page 145, formula (3) and (4)
!
!     selec == 2
!      Irene A. Stegun and Ruth Zucker, "Automatic Computing Methods
!       for Special Functions", Journal of Research of the National
!       Bureau of Standards - B. Mathematical Sciences, 1970,
!       Volume 74B, Issue 3, Pages 211-224,
!       http://dx.doi.org/10.6028/jres.074B.019,
!       here: page 214, 4th formula
!
!     selec == 3
!      Henry E. Fettis, "A Stable Algorithm for Computing the Inverse
!       Error Function in the "Tail-End" Region", Mathematics of
!       Computation, 1974, Volume 28, Issue 126, Page 585-587,
!       https://doi.org/10.1090/S0025-5718-1974-0341812-5,
!       here: page 585, 4th formula,
!      Henry E. Fettis, "Corrigendum: “A stable algorithm for
!       computing the inverse error function in the "tail-end" Region
!       (Math. Comp. 28 (1974), 585-587)", Mathematics of Computation,
!       1975, Volume 29, Issue 130, Page 673,
!       https://doi.org/10.1090/S0025-5718-1975-0359272-8,
!       here: page 673
!
!     selec == 4
!      Keisan Online Calculator, provided by Casio Computer Co., Ltd.,
!      ("Casio"), https://keisan.casio.com/menu/system/000000000760,
!      last visited 30. December 2023, now only available in Japanese
!      language
!
!     Modified Lentz method for Continued Fraction:
!      I.J. Thompson and A.R. Barnett, "Coulomb and Bessel functions
!       of complex arguments and order", Journal of Computational
!       Physics, 1986, Volume 64, Issue 2, Page 490–509,
!       https://doi.org/10.1016/0021-9991(86)90046-X,
!       here: page 507 and 508, (III) Lentz's method
!       Local Tiny (p. 508, "small"): "loctiny" set to 1.00E-050_dp,
!       where
!       case 1: if( b(start) == zero ) f     = loctiny
!       case 2: if( C        == zero ) C     = loctiny
!       case 3: if( b+a*D    == zero ) b+a*D = loctiny
!   Remark:
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "limiterfc" is defined in "Constants" for erfc,
!      where erfc( x ≤ limiterfc )   := +2.00
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfcCF( x, it )   result( f )
!
    use kinds, only : ik, rk, lk
    use const, only : zero, half, one, two, four, &
                      onedivsqrtpi, twodivsqrtpi, &
                      eps, eMax, limiterfc
!
    implicit none
!
!   interface
     real   (rk), intent(in )   :: x             ! x from erfc(x)
     integer(ik), intent(out)   :: it            ! iterations
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: zz            ! zz = z² = |x|²
    real   (rk)                 :: z2            ! 2 * |x|
    real   (rk)                 :: t             ! 1 / (z=|x|)
    real   (rk)                 :: t2            ! t² / 2
    real   (rk)                 :: b, a, C, D, h ! Lentz variables
    real   (rk)                 :: k             ! help variable
    logical(lk)                 :: v             ! logical variable
    integer(ik), save           :: selec         ! series selection
    real   (rk)                 :: limit         ! regions limit
!
!   mathematical & machine-dependent constants/parameters
    integer(ik), parameter      :: itmax    = 200_ik
!c  real   (rk), parameter      :: loctiny  = 1.00E-35_rk ! Lentz parameter
!
!
    if( x <= -HUGE(x) .and. x >= -HUGE(x) ) then
        write( *,                                      '(  &
         & t2,"choose selec <1,2,3 or 4>:",             /, &
         & t2,"1: Strecok         (1968), itmax = 200", /, &
         & t2,"2: Stegun/Zucker   (1970), itmax = 200", /, &
         & t2,"3: Fettis          (1974), itmax = 200", /, &
         & t2,"4: Keisan (online) (2023), itmax = 200"  )' )
        read(*,*) selec
    end if
!
    it = 0_ik                                    ! no iterations
!
!   -∞ ≤ x ≤ limiterfc
         if( x <= limiterfc ) then               ! cut off limiterfc
             f = two                             ! f = 2.00
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   z=|x| == zero  =>  normally "erfc( [z=|x|] ≤ zero ) = one",
!   but to avoid an overflow:   "erfc( [z=|x|] ≤ limit) = one"
          if( selec == 1_ik .or. selec == 2_ik .or. selec == 3_ik ) then
              limit = 1.00E-16_rk                ! low regions CF
          else
              limit = TINY(x)                    ! low regions CF
          end if
          if( z <= limit ) then
              f = one                            ! f = 1.0
!
!   0.00E+00 < z=|x| ≤ eMax
    else if( z <= eMax ) then
!
             if( selec == 1_ik ) then
!            modified CF, Strecok (1968), page 145, formula (3) & (4) and
!            modified Lentz method for Continued Fraction
!
!                       2                                       a0 = 1
!            erfc(z) = --- * exp(-z²) * -------------------------------
!                      √pi                                      a1 = 1
!                                       b0=2z + -----------------------
!                                                               a2 = 2
!                                               b1=z + ----------------
!                                                               a3 = 3
!                                                      b2=2z + --------
!                                                                 ...
!            or
!                       2                 a0  a1 a2        an
!            erfc(z) = --- * exp(-z²) * ( --- -- --- ... ------- ... )
!                      √pi                2z+ z+ 2z+     [2z,z]+
!
!                      with a(0) = 1, a(n) = n, for n ≥ 1
!
             zz = z * z                          ! zz  = z * z
             z2 = z * two                        ! 2*z = 2*|x|
             b  = z2;          v = .false.       ! b0
!c no need   if( b<=zero .and. b>=zero ) then    ! b0 == 0
!c no need       f = loctiny                     ! b0 == 0
!c no need   else
                 f = b                           ! b0 /= 0
!c no need   end if
             a  = zero                           ! prepare a1
             D  = zero                           ! D <- 0
             C  = f                              ! C <- f
             do while ( it < itmax ); it=it+1_ik ! iterations
                a = a + one                      ! a1, ..., an
                if( v ) then                     ! b1, ..., bn
                    b = z2;     v = .false.      ! b = 2*z
                else
                    b = z;      v = .true.       ! b =   z
                end if
!               calculation of C and D, incl. h = C*D
                C = b + a/C                      ! bn >> + <<
!c no need          if( C <= zero .and. C >= zero ) C = loctiny
                D = b + a*D                      ! bn >> + <<
!c no need          if( D <= zero .and. D >= zero ) then
!c no need              D = one / loctiny
!c no need          else
                        D = one / D
!c no need          end if
                h = C * D                        ! h step
                f = f * h                        ! new function value
                if( ABS( h - one ) <= eps ) exit ! precision check
             end do
             f = (twodivsqrtpi * EXP(-zz)) / f   ! finalize for pos. x
             if( x < zero ) f = two - f          ! erfc(-x) = 2 - erfc(x)
             end if                              ! selec == 1
!
!
             if( selec == 2_ik ) then
!            Continued Fraction, Stegun/Zucker (1970), page 214, 4th formula
!            and modified Lentz method for Continued Fraction
!
!                       1                                    a0 = 1.0
!            erfc(z) = --- * exp(-z²)* -------------------------------
!                      √pi                                   a1 = 0.5
!                                      b0=z + ------------------------
!                                                            a2 = 1.0
!                                             b1=z + -----------------
!                                                            a3 = 1.5
!                                                    b2=z + ----------
!                                                              ...
!            or
!                       1                 a0  a1  a2       an
!            erfc(z) = --- * exp(-z²) * ( --  --  --  ...  --  ... )
!                      √pi                z+  z+  z+       z+
!
!                      with a(0) = 1, a(n) = n/2, for n ≥ 1
!
!
             zz = z * z                          ! zz = z * z
             b  = z                              ! b0, b1, ..., bn
             f  = b                              ! b0 /= 0
             a  = zero                           ! prepare a1
             D  = zero                           ! D <- 0
             C  = f                              ! C <- f
             do while(it < itmax); it = it+1_ik  ! iterations
                a = a + half                     ! a1, ..., an
                C =        b + a/C               ! bn >> + <<
                D = one / (b + a*D)              ! bn >> + <<
                h = C * D                        ! h step
                f = f * h                        ! new function value
                if( ABS(h - one) <  eps ) exit   ! precision check
             end do
             f = ( onedivsqrtpi * EXP(-zz) ) / f ! finalize for pos. x
             if( x < zero ) f = two - f          ! erfc(-x) = 2 - erfc(x)
             end if                              ! selec == 2
!
!
             if( selec == 3_ik ) then
!            Continued Fraction, Fettis (1974), page 585, 4th formula,
!            Fettis (1975), page 673 and modified Lentz method for CF
!
!                       1                                  a0 =  (1/z)
!            erfc(z) = ---*exp(-z²) * -----------------------------------
!                      √pi                                 a1 = 1*(t²/2)
!                                     b0=1 + ----------------------------
!                                                          a2 = 2*(t²/2)
!                                            b1=1 + ---------------------
!                                                          a3 = 3*(t²/2)
!                                                   b2=1 + --------------
!                                                              ...
!            or
!                       1                a0  a1  a2       an
!            erfc(z) = ---*exp(-z²) * (  --  --  --  ...  --  ...  )
!                      √pi               1+  1+  1+       1+
!
!                      with a(0) = t, a(n) = n*(t²/2), for n ≥ 1
!
             zz = z * z                          ! zz = z²
             t  = one / z                        ! t  = 1 / z
             t2 = (t * t) / two                  ! t2 = t² / 2
             b  = one                            ! b0, ..., bn
             f  = b                              ! b0 /= 0
             a  = zero                           ! prepare a1
             D  = zero                           ! D <- 0
             C  = f                              ! C <- f
             do while ( it < itmax ); it=it+1_ik ! iterations
                a = a + t2                       ! a1, ..., an
                C =        b + a/C               ! bn >> + <<
                D = one / (b + a*D)              ! bn >> + <<
                h = C * D                        ! h step
                f = f * h                        ! new function value
                if( ABS( h - one ) <= eps ) exit ! precision check
             end do
             f = ( onedivsqrtpi*EXP(-zz)*t ) / f ! finalize for pos. x
             if( x < zero ) f = two - f          ! erfc(-x) = 2 - erfc(x)
             end if                              ! selec == 3
!
!
             if( selec == 4_ik ) then
!            Continued Fraction, Keisan Online Calculator (2023) for 0 ≤ x
!            and modified Lentz method for Continued Fraction
!
!                       2*z*exp(-z²)                             a0 = 1.0
!            erfc(z) = ------------- *----------------------------------------
!                          √pi                                   a1 = -2*(2-1)
!                                     b0=2z²+1 + -----------------------------
!                                                                a2 = -4*(4-1)
!                                                2z²+5 + ---------------------
!                                                                a3 = -6*(6-1)
!                                                        2z²+9 + -------------
!                                                                    ...
!            or
!                      2*z*exp(-z²)      a0   a1   a2        an
!            erfc(z) = ------------ * (  ---  ---  ---  ...  ---  ...  )
!                         √pi            b0+  b1+  b2+       bn+
!
!             with  a(0) = 1.0;       a(n) = -2*n*(2*n - 1),    for n ≥ 1
!                   b(0) = 2z² + 1;   b(n) = 2z² + 4*n +1,      for n ≥ 1
!
             zz = z * z                          ! zz = z * z
             b  = two*zz + one                   ! b0
             f  = b                              ! b0 /= 0
             k  = zero                           ! prepare a1
             C  = f                              ! C <- f
             D  = zero                           ! D <- 0
             do while ( it < itmax ); it=it+1_ik ! iterations
                k = k + two                      ! prepare a
                a = k * (k - one )               ! a1, ..., an
                b = b + four                     ! b1, ..., bn
                C =        b - a/C               ! bn >> - <<
                D = one / (b - a*D)              ! bn >> - <<
                h = C * D                        ! h step
                f = f * h                        ! new function value
                if( ABS( h - one ) <= eps ) exit ! precision check
             end do
!            finalize the integral from CF for positive x
             f = (twodivsqrtpi * EXP(-zz) * z)/f ! finalize for pos. x
             if( x < zero ) f = two - f          ! erfc(-x) = 2 - erfc(x)
             end if                              ! selec == 4
!
!
!   eMax < z=|x| ≤ +∞
    else                                         ! cut off z > eMax
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcCF
!
!---------------------------------------------------------------------
!