!
!   Remark:
!   1. ./.
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   Scaled Complementary Error Function (erfcx): High Precision
!
!   real(rk) function erfcxIntrinsic  ( x )       result( f )  GFortran/Silverfrost (2023),  Intrinsic ERFC_SCALED
!   real(rk) function erfcxCody       ( x )       result( f )  Cody                 (1969),  Rat. Chebyshev Approximation
!   real(rk) function erfcxSchonfelder( x )       result( f )  Schonfelder          (1978),  Chebyshev Expansions
!   real(rk) function erfcxShepherd   ( x )       result( f )  Shepherd/Laframboise (1981),  Chebyshev Approximation
!   real(rk) function erfcxSlatec     ( x )       result( f )  Fullerton            (1993),  Chebyshev Series
!   real(rk) function erfcxOoura8a    ( x )       result( f )  Ooura                (1996),  Polynomial
!   real(rk) function erfcxOoura8b    ( x )       result( f )  Ooura                (1996),  Polynomial
!   real(rk) function erfcxOoura16    ( x )       result( f )  Ooura                (1996),  Polynomial - quad precision
!   real(rk) function erfcxJohnson    ( x )       result( f )  Johnson/Wuttke       (2012),  Various and Chebyshev Look-Up Approximation
!   real(rk) function erfcxJohnsonLUp ( x )       result( f )  Johnson/Wuttke       (2012),  Look-Up function for erfcxJohnson
!   real(rk) function erfcxZaghloul   ( x )       result( f )  Zaghloul             (2024),  Various Algorithms/Calculations
!   real(rk) function erfcxZaghloulopt( x )       result( f )  Zaghloul             (2024),  Various Algorithms/Calculations optimized
!   real(rk) function erfcxTH         ( x )       result( f )  Hoering              (2024),  Composed erfcx
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   erfcxIntrinsic
!     Scaled Complementary Error Function f = erfcx(x) with intrinsic function
!     referenced GFortran or Silverfrost FTN95
!   Reference:
!     GFortran (GNU Fortran) version 6.3.0-1, part of GCC (GNU Compiler
!      Collection). “class: elemental function ERF”, 2024, http://gcc.gnu.org
!     Silverfrost FORTRAN 95, “class: elemental function ERF”, 2024, 
!      https://www.silverfrost.com/ftn95-help/intrin/erf.aspx
!   Remark:
!     For Silverfrost FTN95 this intrinsic function is not implemented for
!      extended precision and therefore set to "zero".
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxIntrinsic( x )   result( f )
!
    use kinds, only : compiler, ik, rki, rk
    use const, only : zero
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erf(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    integer(ik)                 :: h1            ! = variable "compiler"
    integer(ik)                 :: h2            ! = variable "kind-i"
!
!
    h1 = compiler                                ! variabilize compiler
!
!   Compiler = 1: GFortran part
         if( h1 == 1_ik ) then                   ! GFortran
             f = ERFC_SCALED( x )                ! always intrinsic defined
!
!   Compiler = 2: Silverfrost FTN95 part
    else if( h1 == 2_ik ) then                   ! Silverfrost ftn 95
             h2 = rki                            ! variabilize rki
             if( h2 == 2_ik ) then               ! kind = 2
                 f = ERFC_SCALED( x )            ! intrinsic
             else
!                Silverfrost is not able to handle kind= 1 or 3 => f = 0.0
                 f = zero                        ! f = zero
             end if
    end  if
!
    return
    end function erfcxIntrinsic
!
!-----------------------------------------------------------------------------
!
!   erfcxCody
!     Scaled Complementary Error Function erfcx(x) with Rational
!     Chebyshev Approximations referenced Cody (1969)
!   Reference:
!     W.J. Cody, "Rational Chebyshev Approximations for the Error
!       Function", Mathematics of Computation, July 1969, Volume 23,
!       Issue 107, pages 631-637
!     Original Fortran Version by
!     W.J. Cody, Mathematics and Computer Science Division, Argonne
!       National Laboratory, Argonne, IL 60439, last modification
!       19.03.1990, http://www.netlib.org/specfun/erf
!   Remark:
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )   := +∞  = HUGE(x)
!     Parameter "limiterfcxH" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ limiterfcxH )   := (1/√pi)/x = 1/(√pi*x)
!      where erfcx( z > limiterfcxH )   := 0.0
!     Parameter "xhuge", original dp from Cody is redefined,
!      where xhuge                      := SQRT( one / EPSILON(x) )
!     Cody's original conversion for erfcx(-x) = 2*exp(x²)-erfc(|x|) is:
!      if( x < zero ) then
!          if( x < xlimit ) then                ! x < -26.628
!              f = HUGE( x )                    ! 1.7976..D+308
!          else                                 ! -26.628 < x < 0
!              z = INT(x*sixteen, kind=ik) / sixteen ! rounding
!              y = EXP(z*z) * EXP((x-z)*(x+z))  ! exp rounding
!              f = (y + y) - f                  ! 2*exp(x²) - erfcx(|x|)
!          end if
!      end if
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxCody( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, two, four,        &
                      onedivsqrtpi, twodivsqrtpi,  &
                      limiterfcxL, limiterfcxH
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfcx(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
    real   (rk)                 :: u             ! help variable
    real   (rk)                 :: xnum          ! sum numerator
    real   (rk)                 :: xden          ! sum denominator
    integer(ik)                 :: i             ! variable do...
!
!   mathematical & machine-dependent constants/parameters
    real   (rk), parameter      :: xsmall        = EPSILON(one)/two
    real   (rk), parameter      :: thresh        = 0.46875E+0000_rk
!                                  thresh       := erf(x) == erfc(x)
    real   (rk), parameter      :: xhuge         = SQRT(one/EPSILON(x))
!
!   coefficients erf(x), p. 633 for xsmall < |x| ≤ (thresh = 0.46875)
    real   (rk), parameter      :: p1(0_ik:4_ik) =                (/ &
     3.209377589138469472562E+03_rk, 3.774852376853020208137E+02_rk, &
     1.138641541510501556495E+02_rk, 3.161123743870565596947E+00_rk, &
     1.857777061846031526730E-01_rk                                 /)
    real   (rk), parameter      :: q1(0_ik:4_ik) =                (/ &
     2.844236833439170622273E+03_rk, 1.282616526077372275645E+03_rk, &
     2.440246379344441733056E+02_rk, 2.360129095234412093499E+01_rk, &
     1.000000000000000000000E+00_rk                                 /)
!
!   coefficients erfc(x), p. 635 for (thresh = 0.46875) < |x| ≤ (four = 4.0)
    real   (rk), parameter      :: p2(0_ik:8_ik) =                (/ &
     1.23033935479799725272E+03_rk,  2.05107837782607146533E+03_rk,  &
     1.71204761263407058314E+03_rk,  8.81952221241769090411E+02_rk,  &
     2.98635138197400131132E+02_rk,  6.61191906371416294775E+01_rk,  &
     8.88314979438837594118E+00_rk,  5.64188496988670089180E-01_rk,  &
     2.15311535474403846343E-08_rk                                  /)
    real   (rk), parameter      :: q2(0_ik:8_ik) =                (/ &
     1.23033935480374942043E+03_rk,  3.43936767414372163696E+03_rk,  &
     4.36261909014324715820E+03_rk,  3.29079923573345962678E+03_rk,  &
     1.62138957456669018874E+03_rk,  5.37181101862009857509E+02_rk,  &
     1.17693950891312499305E+02_rk,  1.57449261107098347253E+01_rk,  &
     1.00000000000000000000E+00_rk                                  /)
!
!   coefficients erfc(x), p. 636 for (four = 4.0) < |x| < (six = 6.0)
    real   (rk), parameter      :: p3(0_ik:5_ik) =                (/ &
     6.58749161529837803157E-04_rk,  1.60837851487422766278E-02_rk,  &
     1.25781726111229246204E-01_rk,  3.60344899949804439429E-01_rk,  &
     3.05326634961232344035E-01_rk,  1.63153871373020978498E-02_rk  /)
    real   (rk), parameter      :: q3(0_ik:5_ik) =                (/ &
     2.33520497626869185443E-03_rk,  6.05183413124413191178E-02_rk,  &
     5.27905102951428412248E-01_rk,  1.87295284992346042097E+00_rk,  &
     2.56852019228982242072E+00_rk,  1.00000000000000000000E+00_rk  /)
!
!
!   -∞ ≤ x ≤ limiterfcxL
         if( x <= limiterfcxL ) then             ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z=|x| : 0, ..., +∞
!
!   (zero=0.00) ≤ z=|x| ≤ ( xsmall = EPSILON(one)/two )
         if( z <= xsmall ) then
!detail      u = zero                            ! EXP(u²=0) = 1.0
!detail      f = x * p1(0_ik) / q1(0_ik)         ! erf(x)  = x * 2./sqrtpi
!detail      f = EXP( u*u ) * (one - f)          ! [exp(u²=0)=1.]*(1.-erf(x))
!detail      f = one - x * p1(0_ik) / q1(0_ik)   ! or only  : 1. - x*2./sqrtpi
             f = one - x * twodivsqrtpi          ! erfcx(x) = 1. - x*2./sqrtpi
!
!   xsmall < z=|x| ≤ (thresh=0.46875E+00)
    else if( z <= thresh ) then
             u    = z * z                        ! 1st formula p. 631
             xnum = p1(4_ik) * u
             xden = q1(4_ik) * u
             do i = 3_ik, 1_ik, -1_ik
                xnum = (xnum + p1(i)) * u
                xden = (xden + q1(i)) * u
             end do
             f = x * (xnum + p1(0_ik)) / (xden + q1(0_ik))
             f = EXP( u ) * (one - f)            ! exp(x²)*(1-erf(x))
!
!   (thresh=0.46875E+00) < z=|x| ≤ (four=4.000E+00)
    else if( z <= four ) then
             xnum = p2(8_ik) * z                 ! 2nd formula p. 631
             xden = q2(8_ik) * z
             do i = 7_ik, 1_ik, -1_ik
                xnum = (xnum + p2(i)) * z
                xden = (xden + q2(i)) * z
             end do
             f = (xnum + p2(0_ik)) / (xden + q2(0_ik))
             if( x < zero ) f = two*EXP(z*z) -f ! erfcx(-x)=2*exp(z²)-erfcx(z)
!
!   4.00E+00 < z=|x| < xhuge
    else if( z < xhuge ) then
             u = one / (z * z)                   ! 3rd formula p. 631
             xnum = p3(5_ik) * u
             xden = q3(5_ik) * u
             do i = 4_ik, 1_ik, -1_ik
                xnum = (xnum + p3(i)) * u
                xden = (xden + q3(i)) * u
             end do
             f = u * (xnum + p3(0_ik)) / (xden + q3(0_ik))
             f = (onedivsqrtpi -  f) / z
             if( x < zero ) f = two*EXP(z*z) -f ! erfcx(-x)=2*exp(z²)-erfcx(z)
!
!   xhuge ≤ z ≤ limiterfcxH
    else if( z <= limiterfcxH ) then
             f = onedivsqrtpi / z                ! f = (1/√pi)*(1/z)
!
!   limiterfcxH < z=|x| ≤ +∞
    else                                         ! cut off "High"
             f = zero                            ! f = 0.0
!
    end if                                       ! end if calculation
    end if                                       ! end if cases
!
    return
    end function erfcxCody
!
!-----------------------------------------------------------------------------
!
!   erfcxSchonfelder
!     Scaled Complementary Error Function erfcx(x) with Chebyshev Expansions
!     referenced Schonfelder (1978)
!   Reference:
!     J.L. Schonfelder, "Chebyshev Expansions for the Error and Related
!     Functions", Mathematics of Computation, 1978, Volume 32, Issue 144,
!     Pages 1232-1240, https://doi.org/10.1090/S0025-5718-1978-0494846-8
!   Remarks:
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )   := +∞ = HUGE(x)
!     Parameter "sqrteps" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ sqrteps )       := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfcxBig" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ limiterfcxBig ) := 1/(√pi*(x + 0.5/x))
!     Parameter "limiterfcxH" is defined in "Constants" for erfcx,
!      where erfcx( z < limiterfcxH )   := (1/√pi)/x = 1/(√pi*x)
!      where erfcx( z ≥ limiterfcxH )   := 0.0
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxSchonfelder( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, two,                         &
                      onedivsqrtpi, twodivsqrtpi,             &
                      eMax, sqrteps,                          &
                      limiterfcxL, limiterfcxBig, limiterfcxH
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfcx(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
    real   (rk)                 :: t,  u         ! var. Chebyshev
    real   (rk)                 :: sv, d, dd     ! var. Clenshaw
    integer(ik)                 :: j             ! do ...
!
!   parameter for y = exp(x²)*erfc(x) and t = (x-k) / (x+k),
!   with k = 3.75 on page 1235, table 3: valid for 0 ≤ x < +∞
    real   (rk), parameter      :: k             =   3.750E+00_rk
    integer(ik), parameter      :: m             =  44_ik
    real   (rk), parameter      :: c(1_ik:m)     =                                 (/ &
    +6.10143081923200417926465815756E-01_rk, -4.34841272712577471828182820888E-01_rk, &
    +1.76351193643605501125840298123E-01_rk,  -6.0710795609249414860051215825E-02_rk, &
     +1.7712068995694114486147141191E-02_rk,   -4.321119385567293818599864968E-03_rk, &
       +8.54216676887098678819832055E-04_rk,    -1.27155090609162742628893940E-04_rk, &
        +1.1248167243671189468847072E-05_rk,       +3.13063885421820972630152E-07_rk, &
          -2.70988068537762022009086E-07_rk,        +3.0737622701407688440959E-08_rk, &
            +2.515620384817622937314E-09_rk,         -1.028929921320319127590E-09_rk, &
              +2.9944052119949939363E-11_rk,           +2.6051789687266936290E-11_rk, &
               -2.634839924171969386E-12_rk,             -6.43404509890636443E-13_rk, &
                +1.12457401801663447E-13_rk,              +1.7281533389986098E-14_rk, &
                  -4.264101694942375E-15_rk,                -5.45371977880191E-16_rk, &
                   +1.58697607761671E-16_rk,                 +2.0899837844334E-17_rk, &
                     -5.900526869409E-18_rk,                   -9.41893387554E-19_rk, &
                      +2.14977356470E-19_rk,                    +4.6660985008E-20_rk, &
                        -7.243011862E-21_rk,                     -2.387966824E-21_rk, &
                         +1.91177535E-22_rk,                      +1.20482568E-22_rk, &
                            -6.72377E-25_rk,                        -5.747997E-24_rk, &
                            -4.28493E-25_rk,                         +2.44856E-25_rk, &
                             +4.3793E-26_rk,                           -8.151E-27_rk, &
                              -3.089E-27_rk,                             +9.3E-29_rk, &
                               +1.74E-28_rk,                             +1.6E-29_rk, &
                                -8.0E-30_rk,                             -2.0E-30_rk /)
!
!   -∞ ≤ x ≤ limiterfcxL
    if( x <= limiterfcxL ) then                  ! cut off "Low"
        f = +HUGE( x )                           ! f = +∞
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ limiterfcxBig
    else if( z <= limiterfcxBig ) then
!            Chebyhev Approximation, page 1235, table 3
             t  = (z - k) / (z + k)              ! Chebyshev t
             u  = two * t
             d  = zero
             dd = zero
             do j = m, 2_ik, -1_ik               ! Clenshaw's recurrence
                sv = d; d = u*d - dd + c(j); dd = sv
             end do
             f = t*d - dd + c(1_ik)/two          ! finalize Clenshaw=erfcx(z)
             if( x < zero ) f = two*EXP(z*z) -f ! erfcx(-x)=2*exp(z²)-erfcx(z)
!
!   limiterfcxBig < z ≤ limiterfcxH
    else if( z <= limiterfcxH ) then
             f = onedivsqrtpi / z                ! f = (1/√pi)*(1/z)
!
!   limiterfcxH < z=|x| ≤ +∞
    else                                         ! cut off "High"
             f = zero                            ! f = 0.0
!
    end if                                       ! end if calculation
    end if                                       ! end if cases
!
    return
    end function erfcxSchonfelder
!
!-----------------------------------------------------------------------------
!
!   erfcxShepherd
!     Scaled Complementary Error Function erfcx(x) with Chebyshev 
!     Approximation referenced Shepherd/Laframboise (1981)
!   Reference:
!     M.M. Shepherd and J.G. Laframboise, "Chebyshev Approximation of
!     (1 + 2x)exp(x²)erfc x in 0 ≤ x < ∞", Mathematics of Computation, 1981,
!     Volume 36, Issue 153, Page 249-253,
!     https://doi.org/10.1090/S0025-5718-1981-0595058-X
!   Remarks:
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )   := +∞  = HUGE(x)
!     Parameter "sqrteps" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ sqrteps )       := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfcxBig" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ limiterfcxBig ) := 1/(√pi*(x + 0.5/x))
!     Parameter "limiterfcxH" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ limiterfcxH )   := (1/√pi)/x = 1/(√pi*x)
!      where erfcx( z > limiterfcxH )   := 0.0
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxShepherd( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, two,                         &
                      onedivsqrtpi, twodivsqrtpi,             &
                      eMax, sqrteps,                          &
                      limiterfcxL, limiterfcxBig, limiterfcxH
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfcx(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   Local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
    real   (rk)                 :: t,  u         ! var. Chebyshev
    real   (rk)                 :: sv, d, dd     ! var. Clenshaw
    integer(ik)                 :: j             ! do ...
!
!   coefficients for (1+2x)*exp(x²)*erfc(x), page 252, table 1:
    real   (rk), parameter      :: k             =   3.750E+00_rk
    integer(ik), parameter      :: m             =  31_ik
    real   (rk), parameter, dimension(1_ik:m) :: c =         (/ &
    0.1177578934567401754080E+01_rk, -0.45900545806464773310E-02_rk, &
     -0.84249133366517915584E-01_rk,  0.59209939998191890498E-01_rk, &
     -0.26658668435305752277E-01_rk,   0.9074997670705265094E-02_rk, &
      -0.2413163540417608191E-02_rk,    0.490775836525808632E-03_rk, &
        -0.69169733025012064E-04_rk,      0.4139027986073010E-05_rk, &
           0.774038306619849E-06_rk,      -0.218864010492344E-06_rk, &
            0.10764999465671E-07_rk,         0.4521959811218E-08_rk, &
             -0.775440020883E-09_rk,          -0.63180883409E-10_rk, &
               0.28687950109E-10_rk,             0.194558685E-12_rk, &
                -0.965469675E-12_rk,              0.32525481E-13_rk, &
                  0.33478119E-13_rk,              -0.1864563E-14_rk, &
                  -0.1250795E-14_rk,                 0.74182E-16_rk, &
                     0.50681E-16_rk,                 -0.2237E-17_rk, &
                     -0.2187E-17_rk,                    0.27E-19_rk, &
                        0.97E-19_rk,                     0.3E-20_rk, &
                        -0.4E-20_rk                                 /)
!
!
!   -∞ ≤ x ≤ limiterfcxL
         if( x <= limiterfcxL ) then             ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ limiterfcxBig
    else if( z <= limiterfcxBig ) then
!            Chebyshew Approximation, page 250-251
             t = (z - k) / (z + k); u  = two * t ! Chebyshev t
             d = zero;              dd = zero
             do j = m, 2_ik, -1_ik               ! Clenshaw's recurr.
                sv = d;  d = u*d - dd + c(j);  dd = sv
             end do
             f = t*d - dd + c(1_ik)              ! finalize Chebyshev!
             f = f / ( one + two*z )             ! finalize erfcx(z)
             if( x < zero ) f = two*EXP(z*z) -f ! erfcx(-x)=2*exp(z²)-erfcx(z)
!
!   limiterfcxBig < z ≤ limiterfcxH
    else if( z <= limiterfcxH ) then
             f = onedivsqrtpi / z                ! f = (1/√pi)*(1/z)
!
!   limiterfcxH < z=|x| ≤ +∞
    else                                         ! cut off "High"
             f = zero                            ! f = 0.0
!
    end if                                       ! end if calculation
    end if                                       ! end if cases
!
    return
    end function erfcxShepherd
!
!-----------------------------------------------------------------------------
!
!   erfcxSlatec
!     Scaled Complementary Error Function f = erfcx(x) with Chebyshev series
!     referenced Slatec (1993)
!   Reference:
!     Chebyshev series "c1, c2 and c3" and other coefficients SLATEC Common 
!      Mathematical Library, Version 4.1, July 1993: A comprehensive software
!      library containing over 1400 general purpose mathematical and 
!      statistical routines written in Fortran 77. Energy Science and
!      Technology Software Center, U.S. Department of Energy. The Library is
!      in the public domain and distributed by the Energy Science and
!      Technology Software Center. Energy Science and Technology Software
!      Center, P.O. Box 1020 Oak Ridge, TN 37831, Telephone 615-576-2606,
!      E-mail: estsc%a1.adonis.mrouter@zeus.osti.gov . Quoted from
!      http://www.netlib.org/slatec/guide, Section 4. "Obtaining the Library"
!      http://www.netlib.org/slatec/ , Referenced function: "DERFC"
!     Evaluation of n-term Chebyshev series
!      R. Broucke, "Ten subroutines for the manipulation of Chebyshev series,
!      Algorithm 446", Communications of the ACM, Association for Computer
!      Machinery, 1973, Volume 16, Issue 4, Page 254-256,
!      https://doi.org/10.1145/362003.362037
!      here: recurrence formula (4) on page 254 f
!   Remark:
!     For the alternative calculation of "n1, n2 and n3" see erfSlatec.f95 .
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )   := +∞  = HUGE(x)
!     Parameter "sqrteps" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ sqrteps )       := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfcxBig" is defined in "Constants" for erfcx,
!      where y = erfcx( z ≤ limiterfcxBig ) := 1/(√pi*(x + 0.5/x)),
!      but here SLATEC calculation. Remark: Big-definition from Cody.
!     Parameter "limiterfcxH" is defined in "Constants" for erfcx,
!      where erfcx( z < limiterfcxH )   := (1/√pi)/x = 1/(√pi*x)
!      where erfcx( z ≥ limiterfcxH )   := 0.0
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxSlatec( x )   result( f )
!
    use kinds, only : compiler, ik, rki, rk
    use const, only : zero, half, one, two, three, five, eight, &
                      sqrtpi, twodivsqrtpi, sqrteps,            &
                      limiterfcxL, limiterfcxBig, limiterfcxH
!
    implicit none
!
!   interface
     real  (rk), intent(in )    :: x             ! x from erfc(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
    real   (rk)                 :: zz            ! zz = z² = |x|²
    real   (rk)                 :: t,  u         ! Cheby coefficients
    real   (rk)                 :: b0, b1, b2    ! help var. Clenshaw
    integer(ik)                 :: i             ! do ...
!
!   Chebyshev series coefficients (former "erfcs") for the 
!   interval 0.00 to  1.00, with weighted error 1.28..E-32
    integer(ik), parameter      :: d1            = 21_ik
    real   (rk), parameter      :: c1(1_ik:d1)   =                                       (/ &
    -0.49046121234691808039984544033376E-01_rk, -0.14226120510371364237824741899631E+00_rk, &
    +0.10035582187599795575754676712933E-01_rk, -0.57687646997674847650827025509167E-03_rk, &
    +0.27419931252196061034422160791471E-04_rk, -0.11043175507344507604135381295905E-05_rk, &
    +0.38488755420345036949961311498174E-07_rk, -0.11808582533875466969631751801581E-08_rk, &
    +0.32334215826050909646402930953354E-10_rk, -0.79910159470045487581607374708595E-12_rk, &
    +0.17990725113961455611967245486634E-13_rk, -0.37186354878186926382316828209493E-15_rk, &
    +0.71035990037142529711689908394666E-17_rk, -0.12612455119155225832495424853333E-18_rk, &
    +0.20916406941769294369170500266666E-20_rk, -0.32539731029314072982364160000000E-22_rk, &
    +0.47668672097976748332373333333333E-24_rk, -0.65980120782851343155199999999999E-26_rk, &
    +0.86550114699637626197333333333333E-28_rk, -0.10788925177498064213333333333333E-29_rk, &
    +0.12811883993017002666666666666666E-31_rk                                             /)
!
!   Chebyshev series coefficients (former "erc2cs") for the 
!   interval 2.50E-01 to 1.00E+00, with weighted error 2.67..E-32
    integer(ik), parameter      :: d2            = 49_ik
    real   (rk), parameter      :: c2(1_ik:d2)   =                                       (/ &
     -0.6960134660230950112739150826197E-01_rk,  -0.4110133936262089348982212084666E-01_rk, &
     +0.3914495866689626881561143705244E-02_rk,  -0.4906395650548979161280935450774E-03_rk, &
     +0.7157479001377036380760894141825E-04_rk,  -0.1153071634131232833808232847912E-04_rk, &
     +0.1994670590201997635052314867709E-05_rk,  -0.3642666471599222873936118430711E-06_rk, &
     +0.6944372610005012589931277214633E-07_rk,  -0.1371220902104366019534605141210E-07_rk, &
     +0.2788389661007137131963860348087E-08_rk,  -0.5814164724331161551864791050316E-09_rk, &
     +0.1238920491752753181180168817950E-09_rk,  -0.2690639145306743432390424937889E-10_rk, &
     +0.5942614350847910982444709683840E-11_rk,  -0.1332386735758119579287754420570E-11_rk, &
     +0.3028046806177132017173697243304E-12_rk,  -0.6966648814941032588795867588954E-13_rk, &
     +0.1620854541053922969812893227628E-13_rk,  -0.3809934465250491999876913057729E-14_rk, &
     +0.9040487815978831149368971012975E-15_rk,  -0.2164006195089607347809812047003E-15_rk, &
     +0.5222102233995854984607980244172E-16_rk,  -0.1269729602364555336372415527780E-16_rk, &
     +0.3109145504276197583836227412951E-17_rk,  -0.7663762920320385524009566714811E-18_rk, &
     +0.1900819251362745202536929733290E-18_rk,  -0.4742207279069039545225655999965E-19_rk, &
     +0.1189649200076528382880683078451E-19_rk,  -0.3000035590325780256845271313066E-20_rk, &
     +0.7602993453043246173019385277098E-21_rk,  -0.1935909447606872881569811049130E-21_rk, &
     +0.4951399124773337881000042386773E-22_rk,  -0.1271807481336371879608621989888E-22_rk, &
     +0.3280049600469513043315841652053E-23_rk,  -0.8492320176822896568924792422399E-24_rk, &
     +0.2206917892807560223519879987199E-24_rk,  -0.5755617245696528498312819507199E-25_rk, &
     +0.1506191533639234250354144051199E-25_rk,  -0.3954502959018796953104285695999E-26_rk, &
     +0.1041529704151500979984645051733E-26_rk,  -0.2751487795278765079450178901333E-27_rk, &
     +0.7290058205497557408997703680000E-28_rk,  -0.1936939645915947804077501098666E-28_rk, &
     +0.5160357112051487298370054826666E-29_rk,  -0.1378419322193094099389644800000E-29_rk, &
     +0.3691326793107069042251093333333E-30_rk,  -0.9909389590624365420653226666666E-31_rk, &
     +0.2666491705195388413323946666666E-31_rk                                             /)
!
!   Chebyshev series coefficients (former "erfccs") for the 
!   interval 0.00 to 2.50E-01, with weighted error 1.53..E-31
    integer(ik), parameter      :: d3            = 59_ik
    real   (rk), parameter      :: c3(1_ik:d3)   =                                       (/ &
      +0.715179310202924774503697709496E-01_rk,   -0.265324343376067157558893386681E-01_rk, &
      +0.171115397792085588332699194606E-02_rk,   -0.163751663458517884163746404749E-03_rk, &
      +0.198712935005520364995974806758E-04_rk,   -0.284371241276655508750175183152E-05_rk, &
      +0.460616130896313036969379968464E-06_rk,   -0.822775302587920842057766536366E-07_rk, &
      +0.159214187277090112989358340826E-07_rk,   -0.329507136225284321486631665072E-08_rk, &
      +0.722343976040055546581261153890E-09_rk,   -0.166485581339872959344695966886E-09_rk, &
      +0.401039258823766482077671768814E-10_rk,   -0.100481621442573113272170176283E-10_rk, &
      +0.260827591330033380859341009439E-11_rk,   -0.699111056040402486557697812476E-12_rk, &
      +0.192949233326170708624205749803E-12_rk,   -0.547013118875433106490125085271E-13_rk, &
      +0.158966330976269744839084032762E-13_rk,   -0.472689398019755483920369584290E-14_rk, &
      +0.143587337678498478672873997840E-14_rk,   -0.444951056181735839417250062829E-15_rk, &
      +0.140481088476823343737305537466E-15_rk,   -0.451381838776421089625963281623E-16_rk, &
      +0.147452154104513307787018713262E-16_rk,   -0.489262140694577615436841552532E-17_rk, &
      +0.164761214141064673895301522827E-17_rk,   -0.562681717632940809299928521323E-18_rk, &
      +0.194744338223207851429197867821E-18_rk,   -0.682630564294842072956664144723E-19_rk, &
      +0.242198888729864924018301125438E-19_rk,   -0.869341413350307042563800861857E-20_rk, &
      +0.315518034622808557122363401262E-20_rk,   -0.115737232404960874261239486742E-20_rk, &
      +0.428894716160565394623737097442E-21_rk,   -0.160503074205761685005737770964E-21_rk, &
      +0.606329875745380264495069923027E-22_rk,   -0.231140425169795849098840801367E-22_rk, &
      +0.888877854066188552554702955697E-23_rk,   -0.344726057665137652230718495566E-23_rk, &
      +0.134786546020696506827582774181E-23_rk,   -0.531179407112502173645873201807E-24_rk, &
      +0.210934105861978316828954734537E-24_rk,   -0.843836558792378911598133256738E-25_rk, &
      +0.339998252494520890627359576337E-25_rk,   -0.137945238807324209002238377110E-25_rk, &
      +0.563449031183325261513392634811E-26_rk,   -0.231649043447706544823427752700E-26_rk, &
      +0.958446284460181015263158381226E-27_rk,   -0.399072288033010972624224850193E-27_rk, &
      +0.167212922594447736017228709669E-27_rk,   -0.704599152276601385638803782587E-28_rk, &
      +0.297976840286420635412357989444E-28_rk,   -0.126252246646061929722422632994E-28_rk, &
      +0.539543870454248793985299653154E-29_rk,   -0.238099288253145918675346190062E-29_rk, &
      +0.109905283010276157359726683750E-29_rk,   -0.486771374164496572732518677435E-30_rk, &
      +0.152587726411035756763200828211E-30_rk                                             /)
!
!   coefficients n1, n2 and n3 to achieve the required precision according kind specification
    integer(ik),parameter :: pn1(3,2) = reshape( (/13,13,21, 13,13,15/), shape(pn1) )
    integer(ik),parameter ::  n1      = pn1( rki, compiler )
    integer(ik),parameter :: pn2(3,2) = reshape( (/25,25,49, 25,25,30/), shape(pn2) )
    integer(ik),parameter ::  n2      = pn2( rki, compiler )
    integer(ik),parameter :: pn3(3,2) = reshape( (/26,26,59, 26,26,33/), shape(pn3) )
    integer(ik),parameter ::  n3      = pn3( rki, compiler )
!
!
!   -∞ ≤ x ≤ limiterfcxL
         if( x <= limiterfcxL ) then             ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   zero < z=|x| < sqrteps
         if( z < sqrteps ) then
!            f = one - two * x / sqrtpi          ! original
             f = one - twodivsqrtpi * x          ! faster: 1 - (2/√pi) * x
!
!   sqrteps ≤ z=|x| ≤ (one = 1.00E+00)
    else if( z <= one ) then
             t  = two * x*x - one; u  = two * t  ! n1-term Chebyshev series
             b0 = zero;            b1 = zero
             do i = n1,  1_ik, -1_ik
                b2 = b1; b1 = b0; b0 = u*b1 - b2 + c1(i)
             end do
             f = half * (b0 - b2)                ! finalize Clenshaw
             f = (one - x*(one + f)) * EXP(z*z)  ! finalize calculation
!
!   (one = 1.00E+00) < z=|x| ≤ (two = 2.00E+00)
    else if( z <= two ) then
             zz = z * z                          ! (z=|x|)²
             t  = (eight/zz - five)/three        ! n2-term Chebyshev series
             u  = two * t
             b0 = zero;            b1 = zero
             do i = n2, 1_ik, -1_ik
                b2 = b1; b1 = b0; b0 = u*b1 - b2 + c2(i)
             end do
             f = half * (b0 - b2)                ! finalize Clenshaw
             f = one / z * (half + f )           ! finalize erfcx(z)
             if( x < zero ) f = two*EXP(zz) - f ! erfcx(-x)=2*exp(z²)-erfcx(z)
!
!   two < z=|x| ≤ ( limiterfcxBig = SQRT(one/EPSILON(x)) )
    else if( z <= limiterfcxBig ) then
             zz = z * z                          ! (z=|x|)²
             t  = eight/zz - one;  u  = two * t  ! n3-term Chebyshev series
             b0 = zero;            b1 = zero
             do i = n3, 1_ik, -1_ik
                b2 = b1; b1 = b0; b0 = u*b1 - b2 + c3(i)
             end do
             f = half * (b0 - b2)                ! finalize Clenshaw
             f = one / z * (half + f )           ! finalize erfcx(z)
             if( x < zero ) f = two*EXP(zz) - f ! erfcx(-x)=2*exp(z²)-erfcx(z)
!
!   limiterfcxBig < z=|x| ≤ limiterfcxH
    else if( z <= limiterfcxH ) then
             f = one / (sqrtpi * x)              ! 1/(√pi * x)
!
!   limiterfcxH < z ≤ +∞
    else
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculations
    end  if                                      ! end if cases
!
    return
    end function erfcxSlatec
!
!-----------------------------------------------------------------------------
!
!   erfcxOoura8a
!     Complementary Error Function f = erfc(x) with Rational Function
!     referenced Ooura (1996)
!   Reference:
!     Masatake Mori, "A Method for Evaluation of the Error Function of Real
!      and Complex Variable with High Relative Accuracy", Publications of the
!      Research Institute for Mathematical Sciences (PRIMS), Kyoto University
!      1983, Volume 19, Issue 3, Page 1081–1094,
!      https://doi.org/10.2977/PRIMS/1195182021
!     Copyright(C) 1996 Takuya Ooura, email:ooura@mmm.t.u-tokyo.ac.jp.
!      You may use, copy, modify this code for any purpose and without fee.
!      You may distribute this ORIGINAL package. 
!      Internet: http://www.kurims.kyoto-u.ac.jp/~ooura/gamerf.html
!      here: referenced file derfc.f, function derfc
!   Remark:
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )   := +∞  = HUGE(x)
!     Parameter "sqrteps" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ sqrteps )       := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfcxBig" is defined in "Constants" for erfcx,
!      where y = erfcx( z ≤ limiterfcxBig ) := 1/(√pi*(x + 0.5/x)),
!      but here SLATEC calculation. Remark: Big-definition from Cody.
!     Parameter "limiterfcxH" is defined in "Constants" for erfcx,
!      where erfcx( z < limiterfcxH )   := (1/√pi)/x = 1/(√pi*x)
!      where erfcx( z ≥ limiterfcxH )   := 0.0
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxOoura8a( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, half, one, two,                   &
                      onedivsqrtpi, twodivsqrtpi, sqrteps,    &
                      limiterfcxL, limiterfcxBig, limiterfcxH
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfc(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: t, u          ! variable
!
!   parameters for the Rational Function
    real   (rk), parameter      :: pa  =  3.97886080735226000E+00_rk
    real   (rk), parameter      :: p(0_ik:22_ik) =                 (/ &
            2.75374741597376782E-01_rk,   4.90165080585318424E-01_rk, &
            7.74368199119538609E-01_rk,   1.07925515155856677E+00_rk, &
            1.31314653831023098E+00_rk,   1.37040217682338167E+00_rk, &
            1.18902982909273333E+00_rk,   8.05276408752910567E-01_rk, &
            3.57524274449531043E-01_rk,   1.66207924969367356E-02_rk, &
           -1.19463959964325415E-01_rk,  -8.38864557023001992E-02_rk, &
            2.49367200053503304E-03_rk,   3.90976845588484035E-02_rk, &
            1.61315329733252248E-02_rk,  -1.33823644533460069E-02_rk, &
           -1.27223813782122755E-02_rk,   3.83335126264887303E-03_rk, &
            7.73672528313526668E-03_rk,  -8.70779635317295828E-04_rk, &
           -3.96385097360513500E-03_rk,   1.19314022838340944E-04_rk, &
            1.27109764952614092E-03_rk                                /)
!
!
!   -∞ ≤ x ≤ limiterfcxL
         if( x <= limiterfcxL ) then             ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ limiterfcxBig
    else if( z <= limiterfcxBig ) then
             t = pa / ( pa + z )
             u = t - half
             f = t*(p( 0_ik) + u*(p( 1_ik) + u*(p( 2_ik) +  &
               & u*(p( 3_ik) + u*(p( 4_ik) + u*(p( 5_ik) +  &
               & u*(p( 6_ik) + u*(p( 7_ik) + u*(p( 8_ik) +  &
               & u*(p( 9_ik) + u*(p(10_ik) + u*(p(11_ik) +  &
               & u*(p(12_ik) + u*(p(13_ik) + u*(p(14_ik) +  &
               & u*(p(15_ik) + u*(p(16_ik) + u*(p(17_ik) +  &
               & u*(p(18_ik) + u*(p(19_ik) + u*(p(20_ik) +  &
               & u*(p(21_ik) + u* p(22_ik)                  &
               &   ))))))))      )))))))       )))))))
             if( x < zero ) f = two*EXP(z*z) -f ! erfcx(-x)=2*exp(z²)-erfcx(z)
!
!   limiterfcxBig < z ≤ limiterfcxH
    else if( z <= limiterfcxH ) then
             f = onedivsqrtpi / z                ! f = (1/√pi)*(1/z)
!
!   limiterfcxH < z=|x| ≤ +∞
    else                                         ! cut off "High"
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcxOoura8a
!
!-----------------------------------------------------------------------------
!
!   erfcxOoura8b
!     Complementary Error Function f = erfc(x) with Rational Function
!     referenced Ooura (1996)
!   Reference:
!     Masatake Mori, "A Method for Evaluation of the Error Function of Real
!      and Complex Variable with High Relative Accuracy", Publications of the
!      Research Institute for Mathematical Sciences (PRIMS), Kyoto University
!      1983, Volume 19, Issue 3, Page 1081–1094,
!      https://doi.org/10.2977/PRIMS/1195182021
!     Copyright(C) 1996 Takuya Ooura, email:ooura@mmm.t.u-tokyo.ac.jp.
!      You may use, copy, modify this code for any purpose and without fee.
!      You may distribute this ORIGINAL package. 
!      Internet: http://www.kurims.kyoto-u.ac.jp/~ooura/gamerf.html
!      here: referenced file gamerf2a.f, function derfc
!   Remark:
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )   := +∞  = HUGE(x)
!     Parameter "sqrteps" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ sqrteps )       := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfcxBig" is defined in "Constants" for erfcx,
!      where y = erfcx( z ≤ limiterfcxBig ) := 1/(√pi*(x + 0.5/x)),
!      but here SLATEC calculation. Remark: Big-definition from Cody.
!     Parameter "limiterfcxH" is defined in "Constants" for erfcx,
!      where erfcx( z < limiterfcxH )   := (1/√pi)/x = 1/(√pi*x)
!      where erfcx( z ≥ limiterfcxH )   := 0.0
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxOoura8b( x )   result( f )
!
    use kinds, only : rk
    use const, only : zero, half, one, two,                   &
                      onedivsqrtpi, twodivsqrtpi, sqrteps,    &
                      limiterfcxL, limiterfcxBig, limiterfcxH
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfc(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: zz            ! z² = |x|²
!
!   parameters for the Rational Function
    real   (rk), parameter      ::                                      &
     pv = 1.26974899965115684E+01_rk, ph = 6.10399733098688199E+00_rk,  &
     p0 = 2.96316885199227378E-01_rk, p1 = 1.81581125134637070E-01_rk,  &
     p2 = 6.81866451424939493E-02_rk, p3 = 1.56907543161966709E-02_rk,  &
     p4 = 2.21290116681517573E-03_rk, p5 = 1.91395813098742864E-04_rk,  &
     p6 = 9.71013284010551623E-06_rk, p7 = 1.66642447174307753E-07_rk,  &
     q0 = 6.12158644495538758E-02_rk, q1 = 5.50942780056002085E-01_rk,  &
     q2 = 1.53039662058770397E+00_rk, q3 = 2.99957952311300634E+00_rk,  &
     q4 = 4.95867777128246701E+00_rk, q5 = 7.41471251099335407E+00_rk,  &
     q6 = 1.04765104356545238E+01_rk, q7 = 1.48455557345597957E+01_rk
!
!
!   -∞ ≤ x ≤ limiterfcxL
         if( x <= limiterfcxL ) then             ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ limiterfcxBig
    else if( z <= limiterfcxBig ) then
             zz = x * x                           ! zz = x²
             f  = ( p7 / (zz + q7) + p6 / (zz + q6) +  &
                &   p5 / (zz + q5) + p4 / (zz + q4) +  &
                &   p3 / (zz + q3) + p2 / (zz + q2) +  &
                &   p1 / (zz + q1) + p0 / (zz + q0)      )
             if( x < ph ) then                   ! check ph
                 f = (f * x) + (two*EXP(zz)) / ( EXP( pv*x ) + one )
             else
                 f =  f * x
             end if
!
!   limiterfcxBig < z ≤ limiterfcxH
    else if( z <= limiterfcxH ) then
             f = onedivsqrtpi / z                ! f = (1/√pi)*(1/z)
!
!   limiterfcxH < z=|x| ≤ +∞
    else                                         ! cut off "High"
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcxOoura8b
!
!-----------------------------------------------------------------------------
!
!   erfcxOoura16
!     Complementary Error Function f = erfc(x) with Polynomials referenced
!     Ooura (1996)
!   Reference:
!     Masatake Mori, "A Method for Evaluation of the Error Function of Real
!      and Complex Variable with High Relative Accuracy", Publications of the
!      Research Institute for Mathematical Sciences (PRIMS), Kyoto University
!      1983, Volume 19, Issue 3, Page 1081–1094,
!      https://doi.org/10.2977/PRIMS/1195182021
!     Copyright(C) 1996 Takuya Ooura, email:ooura@mmm.t.u-tokyo.ac.jp.
!      You may use, copy, modify this code for any purpose and without fee.
!      You may distribute this ORIGINAL package. 
!      Internet: http://www.kurims.kyoto-u.ac.jp/~ooura/gamerf.html
!      here: referenced file gamerf2a.f, function qerfc
!   Remark:
!     This version is especially developed for quadruple precision.
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )   := +∞  = HUGE(x)
!     Parameter "sqrteps" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ sqrteps )       := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfcxBig" is defined in "Constants" for erfcx,
!      where y = erfcx( z ≤ limiterfcxBig ) := 1/(√pi*(x + 0.5/x)),
!      but here SLATEC calculation. Remark: Big-definition from Cody.
!     Parameter "limiterfcxH" is defined in "Constants" for erfcx,
!      where erfcx( z < limiterfcxH )   := (1/√pi)/x = 1/(√pi*x)
!      where erfcx( z ≥ limiterfcxH )   := 0.0
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxOoura16( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, one, two,                         &
                      onedivsqrtpi, twodivsqrtpi, sqrteps,    &
                      limiterfcxL, limiterfcxBig, limiterfcxH
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfc(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! absolute x
    real   (rk)                 :: zz            ! z² = |x|²
!a  integer(ik)                 :: i             ! do ...
!
!   Parameter for the Polynomials
    real   (rk), parameter      ::                           &
       pv = 1.8296570980424689847157930974106706835E+01_rk,  &
       ph = 8.9588287394342176848213494031807385567E+00_rk
    real   (rk), parameter      :: p(0_ik:16_ik) =                                               (/ &
    2.1226887418241545314975570224238841543E-01_rk, 1.6766968820663231170102487414107148110E-01_rk, &
    1.0461429607758480243524362040994242137E-01_rk, 5.1557963860512142911764627378588661742E-02_rk, &
    2.0070986488528139460346647533434778000E-02_rk, 6.1717726506718148117513762897928828534E-03_rk, &
    1.4990611906920858646769185063310410160E-03_rk, 2.8760540416705806615617926157307107830E-04_rk, &
    4.3585593590380741491013549969419946961E-05_rk, 5.2174364856655433775383935118049845471E-06_rk, &
    4.9333351722974670085736982894474122277E-07_rk, 3.6846914376723888190666722894010079935E-08_rk, &
    2.1729515092764086499231043367920037215E-09_rk, 9.9870022842895735663712411206346261651E-11_rk, &
    3.1775163189596489863458236395414830880E-12_rk, 4.5657943993597540327708145643160878201E-14_rk, &
    1.1940964427370412648558173558044106203E-16_rk                                                 /)
    real   (rk), parameter      :: q(0_ik:16_ik) =                                               (/ &
    2.9482230394292049252878077330764031337E-02_rk, 2.6534007354862844327590269604581049764E-01_rk, &
    7.3705575985730123132195272141160572532E-01_rk, 1.4446292893203104133929687855854497896E+00_rk, &
    2.3880606619376559912235584857800710490E+00_rk, 3.5673498777093386979273977202889759348E+00_rk, &
    4.9824969366355296879760903991854492762E+00_rk, 6.6335018387405633238409855625402006223E+00_rk, &
    8.5203645862651289478197632097553870199E+00_rk, 1.0643085317662274170216548777166393329E+01_rk, &
    1.3001669850030489723387515813223808078E+01_rk, 1.5596282517377690399267249728222735970E+01_rk, &
    1.8429903207271748464995406180854691072E+01_rk, 2.1533907893494593530979123915138686107E+01_rk, &
    2.5076752889217226137869837117288885077E+01_rk, 2.9515380437412601845256918753602002410E+01_rk, &
    3.5792848810704122499184545805923520658E+01_rk                                                 /)
!
!
!   -∞ ≤ x ≤ limiterfcxL
         if( x <= limiterfcxL ) then             ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ limiterfcxBig
    else if( z <= limiterfcxBig ) then
             zz = x * x                          ! zz = x²
             f = ( p(16_ik)/(zz+q(16_ik)) + p(15_ik)/(zz+q(15_ik)) + &
               &   p(14_ik)/(zz+q(14_ik)) + p(13_ik)/(zz+q(13_ik)) + &
               &   p(12_ik)/(zz+q(12_ik)) + p(11_ik)/(zz+q(11_ik)) + &
               &   p(10_ik)/(zz+q(10_ik)) + p( 9_ik)/(zz+q( 9_ik)) + &
               &   p( 8_ik)/(zz+q( 8_ik)) + p( 7_ik)/(zz+q( 7_ik)) + &
               &   p( 6_ik)/(zz+q( 6_ik)) + p( 5_ik)/(zz+q( 5_ik)) + &
               &   p( 4_ik)/(zz+q( 4_ik)) + p( 3_ik)/(zz+q( 3_ik)) + &
               &   p( 2_ik)/(zz+q( 2_ik)) + p( 1_ik)/(zz+q( 1_ik)) + &
               &   p( 0_ik)/(zz+q( 0_ik))                              )
             if( x < ph ) then                   ! check ph
                 f = (f * x) + (two*EXP(zz)) / ( EXP( pv*x ) + one )
             else
                 f =  f * x
             end if
!
!   limiterfcxBig < z ≤ limiterfcxH
    else if( z <= limiterfcxH ) then
             f = onedivsqrtpi / z                ! f = (1/√pi)*(1/z)
!
!   limiterfcxH < z=|x| ≤ +∞
    else                                         ! cut off "High"
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcxOoura16
!
!-----------------------------------------------------------------------------
!
!   erfcxJohnson
!     Scaled Complementary Error Function erfcx(x) with Chebyshev
!     Approximation referenced Johnson/Wuttke (2012/2013)
!   Reference:
!     Steven G. Johnson, Massachusetts Institute of Technology, 2012, core
!      author, (C) 2012 Massachusetts Institute of Technology
!     Joachim Wuttke, Forschungszentrum Juelich, 2013, package maintainer,
!      (C) 2013 Forschungszentrum Juelich GmbH
!     Licence: Permission is hereby granted, free of charge, to any person
!      obtaining a copy of this software and associated documentation files
!      (the "Software"), to deal in the Software without restriction,
!      including without limitation the rights to use, copy, modify, merge,
!      publish, distribute, sublicense, and/or sell copies of the Software,
!      and to permit persons to whom the Software is furnished to do so,
!      subject to the following conditions: The above copyright notice and
!      this permission notice shall be included in all copies or substantial
!      portions of the Software. The License was for the original routines
!      in "C".
!     Website for documentation and/or download
!      https://apps.jcns.fz-juelich.de and then liberf 2.4
!   Remark:
!     This function combines a few different ideas (by Steven G. Johnson,
!      dated October 2012):
!       First, for x > 50, it uses a continued-fraction expansion (same as for
!        the Faddeeva function, but with algebraic simplifications for z=i*x).
!       Second, for 0 <= x <= 50, it uses Chebyshev polynomial approximations,
!        but with two twists:
!         a) It maps x to y = 4 / (4+x) in [0,1].  This simple transformation,
!            inspired by a similar transformation in the octave-forge/specfun
!            erfcx by Soren Hauberg, results in much faster Chebyshev
!            convergence than other simple transformations I have examined.
!         b) Instead of using a single Chebyshev polynomial for the entire
!            [0,1] y interval, we break the interval up into 100 equal sub-
!            intervals, with a switch/lookup table, and use much lower degree
!            Chebyshev polynomials in each subinterval. This greatly improves
!            performance in my tests.
!     For x < 0, the relationship erfcx(-x) = 2*exp(x²) - erfc(x), with the
!      usual checks for overflow etcetera. Performance-wise, it seems to be
!      substantially faster than either the SLATEC DERFC function [or an erfcx
!      function derived therefrom] or Cody's CALERF function (from netlib.org/
!      specfun), while retaining near machine precision in accuracy.
!     The pure function "erfcxJohnson.f95" requires the Look-Up table of the
!      pure function "erfcxJohnsonLUp.f95".
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )   := +∞  = HUGE(x)
!     Parameter "sqrteps" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ sqrteps )       := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfcxH" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ limiterfcxH )   := (1/√pi)/x = 1/(√pi*x)
!      where erfcx( z > limiterfcxH )   := 0.0
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxJohnson( x )   result( f )
!
    use kinds,  only : rk
    use const,  only : zero, one, two,                   &
                       onedivsqrtpi, twodivsqrtpi,       &
                       sqrteps, limiterfcxL, limiterfcxH
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfcx(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
!
!
!   -∞ ≤ x ≤ limiterfcxL
         if( x <= limiterfcxL ) then             ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
!   limiterfcxL < x < -6.10E+00
    else if( x < -6.10E+00_rk ) then
             f = two * EXP( x*x )                ! f = 2 * exp(x²)
!
!   -6.10E+00 ≤ x < (zero = 0.00E+00)
    else if( x < zero       ) then
             f = two * EXP(x*x) - erfcxJohnsonLUp( 400.00_rk/(4.00_rk - x) )
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ 50.00E+00
    else if( z <= 50.00E+00_rk ) then
             f = erfcxJohnsonLUp( 400.00_rk/(4.00_rk + x) )
!
!   50.00E+00 < z=|x| ≤ 5.00E+07
    else if( z <=  5.00E+07_rk ) then
!            5-term Continued Fraction Expansion, simplified form:
!            onedivsqrtpi / (x+0.5/(x+1/(x+1.5/(x+2/x))))
             f =   onedivsqrtpi                          &
                 *       ((x*x)*(x*x+4.50_rk) + 2.00_rk) &
                 / ( x * ((x*x)*(x*x+5.00_rk) + 3.75_rk) )
!
!   5.0E+07 < z ≤ limiterfcxH
    else if( z <= limiterfcxH ) then             ! 1-term expansion
             f = onedivsqrtpi / z                ! to avoid an overflow
!
!   limiterfcxH < z=|x| ≤ +∞
    else                                         ! cut off "High"
             f = zero                            ! f = 0.0
!
    end if                                       ! end if calculation
    end if                                       ! end if cases
!
    return
    end function erfcxJohnson
!
!-----------------------------------------------------------------------------
!
!   erfcxJohnsonLUp
!     Look-Up table belongig to the function "erfcxJohnson", referenced
!     Johnson/Wuttke (2012/2013),
!     here: Look-Up table uses transformed x = 100*y, y = 4/(4+x)
!   Reference:
!     Steven G. Johnson, Massachusetts Institute of Technology, 2012, core
!      author, (C) 2012 Massachusetts Institute of Technology
!     Joachim Wuttke, Forschungszentrum Juelich, 2013, package maintainer,
!      (C) 2013 Forschungszentrum Juelich GmbH
!     Licence: Permission is hereby granted, free of charge, to any person
!      obtaining a copy of this software and associated documentation files
!      (the "Software"), to deal in the Software without restriction,
!      including without limitation the rights to use, copy, modify, merge,
!      publish, distribute, sublicense, and/or sell copies of the Software,
!      and to permit persons to whom the Software is furnished to do so,
!      subject to the following conditions: The above copyright notice and
!      this permission notice shall be included in all copies or substantial
!      portions of the Software. The License was for the original routines
!      in "C". Website for documentation and/or download
!      https://apps.jcns.fz-juelich.de and then liberf 2.4
!   Remark:
!     Uses a look-up table of 100 different Chebyshev polynomials for y
!      intervals [0,0.01], [0.01,0.02], ...., [0.99,1], generated with the
!      help of Maple and a little shell script.
!     This allows the Chebyshev polynomials to be of significantly lower
!      degree (about 1/4) compared to fitting the whole [0,1] interval with
!      a single polynomial.
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
!
    pure function erfcxJohnsonLUp( x )   result( f )
!
    use kinds, only : ik, rk
    use const, only : one, two
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfcx(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   lokale variable
    real   (rk)                 :: t             ! help variable
    integer(ik)                 :: j             ! approx. number
!
    real(rk), parameter :: p1(1_ik:7_ik,0_ik:20_ik) = reshape( (/ &
    0.70878032454106438663E-03_rk, 0.71234091047026302958E-03_rk, & !  0
    0.35779077297597742384E-05_rk, 0.17403143962587937815E-07_rk, &
    0.81710660047307788845E-10_rk, 0.36885022360434957634E-12_rk, &
    0.15917038551111111111E-14_rk,                                &
    0.21479143208285144230E-02_rk, 0.72686402367379996033E-03_rk, & !  1
    0.36843175430938995552E-05_rk, 0.18071841272149201685E-07_rk, &
    0.85496449296040325555E-10_rk, 0.38852037518534291510E-12_rk, &
    0.16868473576888888889E-14_rk,                                &
    0.36165255935630175090E-02_rk, 0.74182092323555510862E-03_rk, & !  2
    0.37948319957528242260E-05_rk, 0.18771627021793087350E-07_rk, &
    0.89484715122415089123E-10_rk, 0.40935858517772440862E-12_rk, &
    0.17872061464888888889E-14_rk,                                &
    0.51154983860031979264E-02_rk, 0.75722840734791660540E-03_rk, & !  3
    0.39096425726735703941E-05_rk, 0.19504168704300468210E-07_rk, &
    0.93687503063178993915E-10_rk, 0.43143925959079664747E-12_rk, &
    0.18939926435555555556E-14_rk,                                &
    0.66457513172673049824E-02_rk, 0.77310406054447454920E-03_rk, & !  4
    0.40289510589399439385E-05_rk, 0.20271233238288381092E-07_rk, &
    0.98117631321709100264E-10_rk, 0.45484207406017752971E-12_rk, &
    0.20076352213333333333E-14_rk,                                &
    0.82082389970241207883E-02_rk, 0.78946629611881710721E-03_rk, & !  5
    0.41529701552622656574E-05_rk, 0.21074693344544655714E-07_rk, &
    0.10278874108587317989E-09_rk, 0.47965201390613339638E-12_rk, &
    0.21285907413333333333E-14_rk,                                &
    0.98039537275352193165E-02_rk, 0.80633440108342840956E-03_rk, & !  6
    0.42819241329736982942E-05_rk, 0.21916534346907168612E-07_rk, &
    0.10771535136565470914E-09_rk, 0.50595972623692822410E-12_rk, &
    0.22573462684444444444E-14_rk,                                &
    0.11433927298290302370E-01_rk, 0.82372858383196561209E-03_rk, & !  7
    0.44160495311765438816E-05_rk, 0.22798861426211986056E-07_rk, &
    0.11291291745879239736E-09_rk, 0.53386189365816880454E-12_rk, &
    0.23944209546666666667E-14_rk,                                &
    0.13099232878814653979E-01_rk, 0.84167002467906968214E-03_rk, & !  8
    0.45555958988457506002E-05_rk, 0.23723907357214175198E-07_rk, &
    0.11839789326602695603E-09_rk, 0.56346163067550237877E-12_rk, &
    0.25403679644444444444E-14_rk,                                &
    0.14800987015587535621E-01_rk, 0.86018092946345943214E-03_rk, & !  9
    0.47008265848816866105E-05_rk, 0.24694040760197315333E-07_rk, &
    0.12418779768752299093E-09_rk, 0.59486890370320261949E-12_rk, &
    0.26957764568888888889E-14_rk,                                &
    0.16540351739394069380E-01_rk, 0.87928458641241463952E-03_rk, & ! 10
    0.48520195793001753903E-05_rk, 0.25711774900881709176E-07_rk, &
    0.13030128534230822419E-09_rk, 0.62820097586874779402E-12_rk, &
    0.28612737351111111111E-14_rk,                                &
    0.18318536789842392647E-01_rk, 0.89900542647891721692E-03_rk, & ! 11
    0.50094684089553365810E-05_rk, 0.26779777074218070482E-07_rk, &
    0.13675822186304615566E-09_rk, 0.66358287745352705725E-12_rk, &
    0.30375273884444444444E-14_rk,                                &
    0.20136801964214276775E-01_rk, 0.91936908737673676012E-03_rk, & ! 12
    0.51734830914104276820E-05_rk, 0.27900878609710432673E-07_rk, &
    0.14357976402809042257E-09_rk, 0.70114790311043728387E-12_rk, &
    0.32252476000000000000E-14_rk,                                &
    0.21996459598282740954E-01_rk, 0.94040248155366777784E-03_rk, & ! 13
    0.53443911508041164739E-05_rk, 0.29078085538049374673E-07_rk, &
    0.15078844500329731137E-09_rk, 0.74103813647499204269E-12_rk, &
    0.34251892320000000000E-14_rk,                                &
    0.23898877187226319502E-01_rk, 0.96213386835900177540E-03_rk, & ! 14
    0.55225386998049012752E-05_rk, 0.30314589961047687059E-07_rk, &
    0.15840826497296335264E-09_rk, 0.78340500472414454395E-12_rk, &
    0.36381553564444444445E-14_rk,                                &
    0.25845480155298518485E-01_rk, 0.98459293067820123389E-03_rk, & ! 15
    0.57082915920051843672E-05_rk, 0.31613782169164830118E-07_rk, &
    0.16646478745529630813E-09_rk, 0.82840985928785407942E-12_rk, &
    0.38649975768888888890E-14_rk,                                &
    0.27837754783474696598E-01_rk, 0.10078108563256892757E-02_rk, & ! 16
    0.59020366493792212221E-05_rk, 0.32979263553246520417E-07_rk, &
    0.17498524159268458073E-09_rk, 0.87622459124842525110E-12_rk, &
    0.41066206488888888890E-14_rk,                                &
    0.29877251304899307550E-01_rk, 0.10318204245057349310E-02_rk, & ! 17
    0.61041829697162055093E-05_rk, 0.34414860359542720579E-07_rk, &
    0.18399863072934089607E-09_rk, 0.92703227366365046533E-12_rk, &
    0.43639844053333333334E-14_rk,                                &
    0.31965587178596443475E-01_rk, 0.10566560976716574401E-02_rk, & ! 18
    0.63151633192414586770E-05_rk, 0.35924638339521924242E-07_rk, &
    0.19353584758781174038E-09_rk, 0.98102783859889264382E-12_rk, &
    0.46381060817777777779E-14_rk,                                &
    0.34104450552588334840E-01_rk, 0.10823541191350532574E-02_rk, & ! 19
    0.65354356159553934436E-05_rk, 0.37512918348533521149E-07_rk, &
    0.20362979635817883229E-09_rk, 0.10384187833037282363E-11_rk, &
    0.49300625262222222221E-14_rk,                                &
    0.36295603928292425716E-01_rk, 0.11089526167995268200E-02_rk, & ! 20
    0.67654845095518363577E-05_rk, 0.39184292949913591646E-07_rk, &
    0.21431552202133775150E-09_rk, 0.10994259106646731797E-11_rk, &
    0.52409949102222222221E-14_rk /),           (/ 7_ik, 21_ik /) )
!
    real(rk), parameter :: p2(1_ik:7_ik,1_ik:20_ik) = reshape( (/ &
    0.38540888038840509795E-01_rk, 0.11364917134175420009E-02_rk, & ! 21
    0.70058230641246312003E-05_rk, 0.40943644083718586939E-07_rk, &
    0.22563034723692881631E-09_rk, 0.11642841011361992885E-11_rk, &
    0.55721092871111111110E-14_rk,                                &
    0.40842225954785960651E-01_rk, 0.11650136437945673891E-02_rk, & ! 22
    0.72569945502343006619E-05_rk, 0.42796161861855042273E-07_rk, &
    0.23761401711005024162E-09_rk, 0.12332431172381557035E-11_rk, &
    0.59246802364444444445E-14_rk,                                &
    0.43201627431540222422E-01_rk, 0.11945628793917272199E-02_rk, & ! 23
    0.75195743532849206263E-05_rk, 0.44747364553960993492E-07_rk, &
    0.25030885216472953674E-09_rk, 0.13065684400300476484E-11_rk, &
    0.63000532853333333334E-14_rk,                                &
    0.45621193513810471438E-01_rk, 0.12251862608067529503E-02_rk, & ! 24
    0.77941720055551920319E-05_rk, 0.46803119830954460212E-07_rk, &
    0.26375990983978426273E-09_rk, 0.13845421370977119765E-11_rk, &
    0.66996477404444444445E-14_rk,                                &
    0.48103121413299865517E-01_rk, 0.12569331386432195113E-02_rk, & ! 25
    0.80814333496367673980E-05_rk, 0.48969667335682018324E-07_rk, &
    0.27801515481905748484E-09_rk, 0.14674637611609884208E-11_rk, &
    0.71249589351111111110E-14_rk,                                &
    0.50649709676983338501E-01_rk, 0.12898555233099055810E-02_rk, & ! 26
    0.83820428414568799654E-05_rk, 0.51253642652551838659E-07_rk, &
    0.29312563849675507232E-09_rk, 0.15556512782814827846E-11_rk, &
    0.75775607822222222221E-14_rk,                                &
    0.53263363664388864181E-01_rk, 0.13240082443256975769E-02_rk, & ! 27
    0.86967260015007658418E-05_rk, 0.53662102750396795566E-07_rk, &
    0.30914568786634796807E-09_rk, 0.16494420240828493176E-11_rk, &
    0.80591079644444444445E-14_rk,                                &
    0.55946601353500013794E-01_rk, 0.13594491197408190706E-02_rk, & ! 28
    0.90262520233016380987E-05_rk, 0.56202552975056695376E-07_rk, &
    0.32613310410503135996E-09_rk, 0.17491936862246367398E-11_rk, &
    0.85713381688888888890E-14_rk,                                &
    0.58702059496154081813E-01_rk, 0.13962391363223647892E-02_rk, & ! 29
    0.93714365487312784270E-05_rk, 0.58882975670265286526E-07_rk, &
    0.34414937110591753387E-09_rk, 0.18552853109751857859E-11_rk, &
    0.91160736711111111110E-14_rk,                                &
    0.61532500145144778048E-01_rk, 0.14344426411912015247E-02_rk, & ! 30
    0.97331446201016809696E-05_rk, 0.61711860507347175097E-07_rk, &
    0.36325987418295300221E-09_rk, 0.19681183310134518232E-11_rk, &
    0.96952238400000000000E-14_rk,                                &
    0.64440817576653297993E-01_rk, 0.14741275456383131151E-02_rk, & ! 31
    0.10112293819576437838E-04_rk, 0.64698236605933246196E-07_rk, &
    0.38353412915303665586E-09_rk, 0.20881176114385120186E-11_rk, &
    0.10310784480000000000E-13_rk,                                &
    0.67430045633130393282E-01_rk, 0.15153655418916540370E-02_rk, & ! 32
    0.10509857606888328667E-04_rk, 0.67851706529363332855E-07_rk, &
    0.40504602194811140006E-09_rk, 0.22157325110542534469E-11_rk, &
    0.10964842115555555556E-13_rk,                                &
    0.70503365513338850709E-01_rk, 0.15582323336495709827E-02_rk, & ! 33
    0.10926868866865231089E-04_rk, 0.71182482239613507542E-07_rk, &
    0.42787405890153386710E-09_rk, 0.23514379522274416437E-11_rk, &
    0.11659571751111111111E-13_rk,                                &
    0.73664114037944596353E-01_rk, 0.16028078812438820413E-02_rk, & ! 34
    0.11364423678778207991E-04_rk, 0.74701423097423182009E-07_rk, &
    0.45210162777476488324E-09_rk, 0.24957355004088569134E-11_rk, &
    0.12397238257777777778E-13_rk,                                &
    0.76915792420819562379E-01_rk, 0.16491766623447889354E-02_rk, & ! 35
    0.11823685320041302169E-04_rk, 0.78420075993781544386E-07_rk, &
    0.47781726956916478925E-09_rk, 0.26491544403815724749E-11_rk, &
    0.13180196462222222222E-13_rk,                                &
    0.80262075578094612819E-01_rk, 0.16974279491709504117E-02_rk, & ! 36
    0.12305888517309891674E-04_rk, 0.82350717698979042290E-07_rk, &
    0.50511496109857113929E-09_rk, 0.28122528497626897696E-11_rk, &
    0.14010889635555555556E-13_rk,                                &
    0.83706822008980357446E-01_rk, 0.17476561032212656962E-02_rk, & ! 37
    0.12812343958540763368E-04_rk, 0.86506399515036435592E-07_rk, &
    0.53409440823869467453E-09_rk, 0.29856186620887555043E-11_rk, &
    0.14891851591111111111E-13_rk,                                &
    0.87254084284461718231E-01_rk, 0.17999608886001962327E-02_rk, & ! 38
    0.13344443080089492218E-04_rk, 0.90900994316429008631E-07_rk, &
    0.56486134972616465316E-09_rk, 0.31698707080033956934E-11_rk, &
    0.15825697795555555556E-13_rk,                                &
    0.90908120182172748487E-01_rk, 0.18544478050657699758E-02_rk, & ! 39
    0.13903663143426120077E-04_rk, 0.95549246062549906177E-07_rk, &
    0.59752787125242054315E-09_rk, 0.33656597366099099413E-11_rk, &
    0.16815130613333333333E-13_rk,                                &
    0.94673404508075481121E-01_rk, 0.19112284419887303347E-02_rk, & ! 40
    0.14491572616545004930E-04_rk, 0.10046682186333613697E-06_rk, &
    0.63221272959791000515E-09_rk, 0.35736693975589130818E-11_rk, &
    0.17862931591111111111E-13_rk /),           (/ 7_ik, 20_ik /) )
!
    real(rk), parameter :: p3(1_ik:7_ik,1_ik:20_ik) = reshape( (/ &
    0.98554641648004456555E-01_rk, 0.19704208544725622126E-02_rk, & ! 41
    0.15109836875625443935E-04_rk, 0.10567036667675984067E-06_rk, &
    0.66904168640019354565E-09_rk, 0.37946171850824333014E-11_rk, &
    0.18971959040000000000E-13_rk,                                &
    0.10255677889470089531E+00_rk, 0.20321499629472857418E-02_rk, & ! 42
    0.15760224242962179564E-04_rk, 0.11117756071353507391E-06_rk, &
    0.70814785110097658502E-09_rk, 0.40292553276632563925E-11_rk, &
    0.20145143075555555556E-13_rk,                                &
    0.10668502059865093318E+00_rk, 0.20965479776148731610E-02_rk, & ! 43
    0.16444612377624983565E-04_rk, 0.11700717962026152749E-06_rk, &
    0.74967203250938418991E-09_rk, 0.42783716186085922176E-11_rk, &
    0.21385479360000000000E-13_rk,                                &
    0.11094484319386444474E+00_rk, 0.21637548491908170841E-02_rk, & ! 44
    0.17164995035719657111E-04_rk, 0.12317915750735938089E-06_rk, &
    0.79376309831499633734E-09_rk, 0.45427901763106353914E-11_rk, &
    0.22696025653333333333E-13_rk,                                &
    0.11534201115268804714E+00_rk, 0.22339187474546420375E-02_rk, & ! 45
    0.17923489217504226813E-04_rk, 0.12971465288245997681E-06_rk, &
    0.84057834180389073587E-09_rk, 0.48233721206418027227E-11_rk, &
    0.24079890062222222222E-13_rk,                                &
    0.11988259392684094740E+00_rk, 0.23071965691918689601E-02_rk, & ! 46
    0.18722342718958935446E-04_rk, 0.13663611754337957520E-06_rk, &
    0.89028385488493287005E-09_rk, 0.51210161569225846701E-11_rk, &
    0.25540227111111111111E-13_rk,                                &
    0.12457298393509812907E+00_rk, 0.23837544771809575380E-02_rk, & ! 47
    0.19563942105711612475E-04_rk, 0.14396736847739470782E-06_rk, &
    0.94305490646459247016E-09_rk, 0.54366590583134218096E-11_rk, &
    0.27080225920000000000E-13_rk,                                &
    0.12941991566142438816E+00_rk, 0.24637684719508859484E-02_rk, & ! 48
    0.20450821127475879816E-04_rk, 0.15173366280523906622E-06_rk, &
    0.99907632506389027739E-09_rk, 0.57712760311351625221E-11_rk, &
    0.28703099555555555556E-13_rk,                                &
    0.13443048593088696613E+00_rk, 0.25474249981080823877E-02_rk, & ! 49
    0.21385669591362915223E-04_rk, 0.15996177579900443030E-06_rk, &
    0.10585428844575134013E-08_rk, 0.61258809536787882989E-11_rk, &
    0.30412080142222222222E-13_rk,                                &
    0.13961217543434561353E+00_rk, 0.26349215871051761416E-02_rk, & ! 50
    0.22371342712572567744E-04_rk, 0.16868008199296822247E-06_rk, &
    0.11216596910444996246E-08_rk, 0.65015264753090890662E-11_rk, &
    0.32210394506666666666E-13_rk,                                &
    0.14497287157673800690E+00_rk, 0.27264675383982439814E-02_rk, & ! 51
    0.23410870961050950197E-04_rk, 0.17791863939526376477E-06_rk, &
    0.11886425714330958106E-08_rk, 0.68993039665054288034E-11_rk, &
    0.34101266222222222221E-13_rk,                                &
    0.15052089272774618151E+00_rk, 0.28222846410136238008E-02_rk, & ! 52
    0.24507470422713397006E-04_rk, 0.18770927679626136909E-06_rk, &
    0.12597184587583370712E-08_rk, 0.73203433049229821618E-11_rk, &
    0.36087889048888888890E-13_rk,                                &
    0.15626501395774612325E+00_rk, 0.29226079376196624949E-02_rk, & ! 53
    0.25664553693768450545E-04_rk, 0.19808568415654461964E-06_rk, &
    0.13351257759815557897E-08_rk, 0.77658124891046760667E-11_rk, &
    0.38173420035555555555E-13_rk,                                &
    0.16221449434620737567E+00_rk, 0.30276865332726475672E-02_rk, & ! 54
    0.26885741326534564336E-04_rk, 0.20908350604346384143E-06_rk, &
    0.14151148144240728728E-08_rk, 0.82369170665974313027E-11_rk, &
    0.40360957457777777779E-13_rk,                                &
    0.16837910595412130659E+00_rk, 0.31377844510793082301E-02_rk, & ! 55
    0.28174873844911175026E-04_rk, 0.22074043807045782387E-06_rk, &
    0.14999481055996090039E-08_rk, 0.87348993661930809254E-11_rk, &
    0.42653528977777777779E-13_rk,                                &
    0.17476916455659369953E+00_rk, 0.32531815370903068316E-02_rk, & ! 56
    0.29536024347344364074E-04_rk, 0.23309632627767074202E-06_rk, &
    0.15899007843582444846E-08_rk, 0.92610375235427359475E-11_rk, &
    0.45054073102222222221E-13_rk,                                &
    0.18139556223643701364E+00_rk, 0.33741744168096996041E-02_rk, & ! 57
    0.30973511714709500836E-04_rk, 0.24619326937592290996E-06_rk, &
    0.16852609412267750744E-08_rk, 0.98166442942854895573E-11_rk, &
    0.47565418097777777779E-13_rk,                                &
    0.18826980194443664549E+00_rk, 0.35010775057740317997E-02_rk, & ! 58
    0.32491914440014267480E-04_rk, 0.26007572375886319028E-06_rk, &
    0.17863299617388376116E-08_rk, 0.10403065638343878679E-10_rk, &
    0.50190265831111111110E-13_rk,                                &
    0.19540403413693967350E+00_rk, 0.36342240767211326315E-02_rk, & ! 59
    0.34096085096200907289E-04_rk, 0.27479061117017637474E-06_rk, &
    0.18934228504790032826E-08_rk, 0.11021679075323598664E-10_rk, &
    0.52931171733333333334E-13_rk,                                &
    0.20281109560651886959E+00_rk, 0.37739673859323597060E-02_rk, & ! 60
    0.35791165457592409054E-04_rk, 0.29038742889416172404E-06_rk, &
    0.20068685374849001770E-08_rk, 0.11673891799578381999E-10_rk, &
    0.55790523093333333334E-13_rk /),           (/ 7_ik, 20_ik /) )
!
    real(rk), parameter :: p4(1_ik:7_ik,1_ik:20_ik) = reshape( (/ &
    0.21050455062669334978E+00_rk, 0.39206818613925652425E-02_rk, & ! 61
    0.37582602289680101704E-04_rk, 0.30691836231886877385E-06_rk, &
    0.21270101645763677824E-08_rk, 0.12361138551062899455E-10_rk, &
    0.58770520160000000000E-13_rk,                                &
    0.21849873453703332479E+00_rk, 0.40747643554689586041E-02_rk, & ! 62
    0.39476163820986711501E-04_rk, 0.32443839970139918836E-06_rk, &
    0.22542053491518680200E-08_rk, 0.13084879235290858490E-10_rk, &
    0.61873153262222222221E-13_rk,                                &
    0.22680879990043229327E+00_rk, 0.42366354648628516935E-02_rk, & ! 63
    0.41477956909656896779E-04_rk, 0.34300544894502810002E-06_rk, &
    0.23888264229264067658E-08_rk, 0.13846596292818514601E-10_rk, &
    0.65100183751111111110E-13_rk,                                &
    0.23545076536988703937E+00_rk, 0.44067409206365170888E-02_rk, & ! 64
    0.43594444916224700881E-04_rk, 0.36268045617760415178E-06_rk, &
    0.25312606430853202748E-08_rk, 0.14647791812837903061E-10_rk, &
    0.68453122631111111110E-13_rk,                                &
    0.24444156740777432838E+00_rk, 0.45855530511605787178E-02_rk, & ! 65
    0.45832466292683085475E-04_rk, 0.38352752590033030472E-06_rk, &
    0.26819103733055603460E-08_rk, 0.15489984390884756993E-10_rk, &
    0.71933206364444444445E-13_rk,                                &
    0.25379911500634264643E+00_rk, 0.47735723208650032167E-02_rk, & ! 66
    0.48199253896534185372E-04_rk, 0.40561404245564732314E-06_rk, &
    0.28411932320871165585E-08_rk, 0.16374705736458320149E-10_rk, &
    0.75541379822222222221E-13_rk,                                &
    0.26354234756393613032E+00_rk, 0.49713289477083781266E-02_rk, & ! 67
    0.50702455036930367504E-04_rk, 0.42901079254268185722E-06_rk, &
    0.30095422058900481753E-08_rk, 0.17303497025347342498E-10_rk, &
    0.79278273368888888890E-13_rk,                                &
    0.27369129607732343398E+00_rk, 0.51793846023052643767E-02_rk, & ! 68
    0.53350152258326602629E-04_rk, 0.45379208848865015485E-06_rk, &
    0.31874057245814381257E-08_rk, 0.18277905010245111046E-10_rk, &
    0.83144182364444444445E-13_rk,                                &
    0.28426714781640316172E+00_rk, 0.53983341916695141966E-02_rk, & ! 69
    0.56150884865255810638E-04_rk, 0.48003589196494734238E-06_rk, &
    0.33752476967570796349E-08_rk, 0.19299477888083469086E-10_rk, &
    0.87139049137777777779E-13_rk,                                &
    0.29529231465348519920E+00_rk, 0.56288077305420795663E-02_rk, & ! 70
    0.59113671189913307427E-04_rk, 0.50782393781744840482E-06_rk, &
    0.35735475025851713168E-08_rk, 0.20369760937017070382E-10_rk, &
    0.91262442613333333334E-13_rk,                                &
    0.30679050522528838613E+00_rk, 0.58714723032745403331E-02_rk, & ! 71
    0.62248031602197686791E-04_rk, 0.53724185766200945789E-06_rk, &
    0.37827999418960232678E-08_rk, 0.21490291930444538307E-10_rk, &
    0.95513539182222222221E-13_rk,                                &
    0.31878680111173319425E+00_rk, 0.61270341192339103514E-02_rk, & ! 72
    0.65564012259707640976E-04_rk, 0.56837930287837738996E-06_rk, &
    0.40035151353392378882E-08_rk, 0.22662596341239294792E-10_rk, &
    0.99891109760000000000E-13_rk,                                &
    0.33130773722152622027E+00_rk, 0.63962406646798080903E-02_rk, & ! 73
    0.69072209592942396666E-04_rk, 0.60133006661885941812E-06_rk, &
    0.42362183765883466691E-08_rk, 0.23888182347073698382E-10_rk, &
    0.10439349811555555556E-12_rk,                                &
    0.34438138658041336523E+00_rk, 0.66798829540414007258E-02_rk, & ! 74
    0.72783795518603561144E-04_rk, 0.63619220443228800680E-06_rk, &
    0.44814499336514453364E-08_rk, 0.25168535651285475274E-10_rk, &
    0.10901861383111111111E-12_rk,                                &
    0.35803744972380175583E+00_rk, 0.69787978834882685031E-02_rk, & ! 75
    0.76710543371454822497E-04_rk, 0.67306815308917386747E-06_rk, &
    0.47397647975845228205E-08_rk, 0.26505114141143050509E-10_rk, &
    0.11376390933333333333E-12_rk,                                &
    0.37230734890119724188E+00_rk, 0.72938706896461381003E-02_rk, & ! 76
    0.80864854542670714092E-04_rk, 0.71206484718062688779E-06_rk, &
    0.50117323769745883805E-08_rk, 0.27899342394100074165E-10_rk, &
    0.11862637614222222222E-12_rk,                                &
    0.38722432730555448223E+00_rk, 0.76260375162549802745E-02_rk, & ! 77
    0.85259785810004603848E-04_rk, 0.75329383305171327677E-06_rk, &
    0.52979361368388119355E-08_rk, 0.29352606054164086709E-10_rk, &
    0.12360253370666666667E-12_rk,                                &
    0.40282355354616940667E+00_rk, 0.79762880915029728079E-02_rk, & ! 78
    0.89909077342438246452E-04_rk, 0.79687137961956194579E-06_rk, &
    0.55989731807360403195E-08_rk, 0.30866246101464869050E-10_rk, &
    0.12868841946666666667E-12_rk,                                &
    0.41914223158913787649E+00_rk, 0.83456685186950463538E-02_rk, & ! 79
    0.94827181359250161335E-04_rk, 0.84291858561783141014E-06_rk, &
    0.59154537751083485684E-08_rk, 0.32441553034347469291E-10_rk, &
    0.13387957943111111111E-12_rk,                                &
    0.43621971639463786896E+00_rk, 0.87352841828289495773E-02_rk, & ! 80
    0.10002929142066799966E-03_rk, 0.89156148280219880024E-06_rk, &
    0.62480008150788597147E-08_rk, 0.34079760983458878910E-10_rk, &
    0.13917107176888888889E-12_rk /),           (/ 7_ik, 20_ik /) )
!
    real(rk), parameter :: p5(1_ik:7_ik,1_ik:19_ik) = reshape( (/ &
    0.45409763548534330981E+00_rk, 0.91463027755548240654E-02_rk, & ! 81
    0.10553137232446167258E-03_rk, 0.94293113464638623798E-06_rk, &
    0.65972492312219959885E-08_rk, 0.35782041795476563662E-10_rk, &
    0.14455745872000000000E-12_rk,                                &
    0.47282001668512331468E+00_rk, 0.95799574408860463394E-02_rk, & ! 82
    0.11135019058000067469E-03_rk, 0.99716373005509038080E-06_rk, &
    0.69638453369956970347E-08_rk, 0.37549499088161345850E-10_rk, &
    0.15003280712888888889E-12_rk,                                &
    0.49243342227179841649E+00_rk, 0.10037550043909497071E-01_rk, & ! 83
    0.11750334542845234952E-03_rk, 0.10544006716188967172E-05_rk, &
    0.73484461168242224872E-08_rk, 0.39383162326435752965E-10_rk, &
    0.15559069118222222222E-12_rk,                                &
    0.51298708979209258326E+00_rk, 0.10520454564612427224E-01_rk, & ! 84
    0.12400930037494996655E-03_rk, 0.11147886579371265246E-05_rk, &
    0.77517184550568711454E-08_rk, 0.41283980931872622611E-10_rk, &
    0.16122419680000000000E-12_rk,                                &
    0.53453307979101369843E+00_rk, 0.11030120618800726938E-01_rk, & ! 85
    0.13088741519572269581E-03_rk, 0.11784797595374515432E-05_rk, &
    0.81743383063044825400E-08_rk, 0.43252818449517081051E-10_rk, &
    0.16692592640000000000E-12_rk,                                &
    0.55712643071169299478E+00_rk, 0.11568077107929735233E-01_rk, & ! 86
    0.13815797838036651289E-03_rk, 0.12456314879260904558E-05_rk, &
    0.86169898078969313597E-08_rk, 0.45290446811539652525E-10_rk, &
    0.17268801084444444444E-12_rk,                                &
    0.58082532122519320968E+00_rk, 0.12135935999503877077E-01_rk, & ! 87
    0.14584223996665838559E-03_rk, 0.13164068573095710742E-05_rk, &
    0.90803643355106020163E-08_rk, 0.47397540713124619155E-10_rk, &
    0.17850211608888888889E-12_rk,                                &
    0.60569124025293375554E+00_rk, 0.12735396239525550361E-01_rk, & ! 88
    0.15396244472258863344E-03_rk, 0.13909744385382818253E-05_rk, &
    0.95651595032306228245E-08_rk, 0.49574672127669041550E-10_rk, &
    0.18435945564444444444E-12_rk,                                &
    0.63178916494715716894E+00_rk, 0.13368247798287030927E-01_rk, & ! 89
    0.16254186562762076141E-03_rk, 0.14695084048334056083E-05_rk, &
    0.10072078109604152350E-07_rk, 0.51822304995680707483E-10_rk, &
    0.19025081422222222222E-12_rk,                                &
    0.65918774689725319200E+00_rk, 0.14036375850601992063E-01_rk, & ! 90
    0.17160483760259706354E-03_rk, 0.15521885688723188371E-05_rk, &
    0.10601827031535280590E-07_rk, 0.54140790105837520499E-10_rk, &
    0.19616655146666666667E-12_rk,                                &
    0.68795950683174433822E+00_rk, 0.14741765091365869084E-01_rk, & ! 91
    0.18117679143520433835E-03_rk, 0.16392004108230585213E-05_rk, &
    0.11155116068018043001E-07_rk, 0.56530360194925690374E-10_rk, &
    0.20209663662222222222E-12_rk,                                &
    0.71818103808729967036E+00_rk, 0.15486504187117112279E-01_rk, & ! 92
    0.19128428784550923217E-03_rk, 0.17307350969359975848E-05_rk, &
    0.11732656736113607751E-07_rk, 0.58991125287563833603E-10_rk, &
    0.20803065333333333333E-12_rk,                                &
    0.74993321911726254661E+00_rk, 0.16272790364044783382E-01_rk, & ! 93
    0.20195505163377912645E-03_rk, 0.18269894883203346953E-05_rk, &
    0.12335161021630225535E-07_rk, 0.61523068312169087227E-10_rk, &
    0.21395783431111111111E-12_rk,                                &
    0.78330143531283492729E+00_rk, 0.17102934132652429240E-01_rk, & ! 94
    0.21321800585063327041E-03_rk, 0.19281661395543913713E-05_rk, &
    0.12963340087354341574E-07_rk, 0.64126040998066348872E-10_rk, &
    0.21986708942222222222E-12_rk,                                &
    0.81837581041023811832E+00_rk, 0.17979364149044223802E-01_rk, & ! 95
    0.22510330592753129006E-03_rk, 0.20344732868018175389E-05_rk, &
    0.13617902941839949718E-07_rk, 0.66799760083972474642E-10_rk, &
    0.22574701262222222222E-12_rk,                                &
    0.85525144775685126237E+00_rk, 0.18904632212547561026E-01_rk, & ! 96
    0.23764237370371255638E-03_rk, 0.21461248251306387979E-05_rk, &
    0.14299555071870523786E-07_rk, 0.69543803864694171934E-10_rk, &
    0.23158593688888888889E-12_rk,                                &
    0.89402868170849933734E+00_rk, 0.19881418399127202569E-01_rk, & ! 97
    0.25086793128395995798E-03_rk, 0.22633402747585233180E-05_rk, &
    0.15008997042116532283E-07_rk, 0.72357609075043941261E-10_rk, &
    0.23737194737777777778E-12_rk,                                &
    0.93481333942870796363E+00_rk, 0.20912536329780368893E-01_rk, & ! 98
    0.26481403465998477969E-03_rk, 0.23863447359754921676E-05_rk, &
    0.15746923065472184451E-07_rk, 0.75240468141720143653E-10_rk, &
    0.24309291271111111111E-12_rk,                                &
    0.97771701335885035464E+00_rk, 0.22000938572830479551E-01_rk, & ! 99
    0.27951610702682383001E-03_rk, 0.25153688325245314530E-05_rk, &
    0.16514019547822821453E-07_rk, 0.78191526829368231251E-10_rk, &
    0.24873652355555555556E-12_rk /),           (/ 7_ik, 19_ik /) )
!
!   Look-Up table with 7 rows and 100 columns
    real(rk), parameter :: p(1_ik:7_ik, 0_ik:99_ik) =                &
               reshape( (/ p1, p2, p3, p4, p5 /), (/ 7_ik, 100_ik /) )
!
!
    j = INT( x, kind = ik )                      ! j = integer(x)
    if( j == 100_ik ) then                       ! close to x == 0,
        f = one                                  ! or       x == 0
!       only coming to this position if y = 1 = erfcx(x), resulting
!       from e.g. |x| < 4*eps, meaning less then 1.00E-15 distance
!       from 1.0
    else
!       LookUp in the table for 100 stored values (j=0 until j=99)
        t = two * x - REAL( 2_ik * j + 1_ik, kind = rk )
        f =  p(1_ik,j)                           +  &
          & (p(2_ik,j) + (p(3_ik,j) + (p(4_ik,j) +  &
          & (p(5_ik,j) + (p(6_ik,j) +  p(7_ik,j) *  &
          &      t)*t) *      t)*t) *       t)*t
    end if
!
    return
    end function erfcxJohnsonLUp
!
!-----------------------------------------------------------------------------
!
!   erfcxZaghloul
!     Scaled Complementary Error Function f = erfcx(x) with various
!     algorithms/calculations referenced Zaghloul (2024)
!   Reference:
!     Mofreh R. Zaghloul, "Efficient multiple-precision computation of the
!      scaled complementary error function and the Dawson integral", Numerical
!      Algorithms, 2024, Volume 95, Issue 3, Pages 1291–1308,
!      https://doi.org/10.1007/s11075-023-01608-8
!     Series Expansion:
!      M.M. Shepherd and J.G. Laframboise, "Chebyshev Approximation of
!       (1 + 2x)exp(x²)erfc x in 0 ≤ x < ∞", Mathematics of Computation,
!       1981, Volume 36, Issue 153, Pages 249-253,
!       https://doi.org/10.1090/S0025-5718-1981-0595058-X
!       here: page 251, formula (7)
!     Continued Fraction:
!      Irene A. Stegun and Ruth Zucker, "Automatic Computing Methods for
!       Special Functions", Journal of Research of the National Bureau of
!       Standards - B. Mathematical Sciences, 1970, Volume 74B, Issue 3,
!       Pages 211-224, http://dx.doi.org/10.6028/jres.074B.019
!       here: page 214, 4th redefined formula
!   Remark:
!     Source, Copyright and Approval
!      The function "erfcxZaghloul" is derived from the original
!      source function "erfcx_rk" by M.R. Zaghloul, a copyright
!      protected source. For more details and approval see the
!      chapter "Copyright and Approvals".
!      A written approval is available since 26. August 2024 .
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )   := +∞  = HUGE(x)
!     The function requires the file "4100-Erfcx-High-include.f95"
!      with the parameters for the Chebyshev Polynomials
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxZaghloul( x )   result( f )
!
    use kinds, only : ik, rk,                        &
                      sp2008, dp2008, qp2008, rk2008
    use const, only : zero, half, one, two,          &
                      onedivsqrtpi, twodivsqrtpi,    &
                      limiterfcxL
!
    implicit none
!
!   interface
     real   (rk), intent(in )   :: x             ! x from erfcx(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
    real   (rk)                 :: zz            ! zz = (z=|x|)² = z*z
!
!-----------------------------------------------------------------------------
!
!   Series Expansion for z=|x| << 1, Zaghloul (2024), page 1293, formula (5)
!
!   "zeps12" (Zaghloul (2024), p.1306, t.10) for the entry parameters =
!   = [sp:2.6486..E-01 / dp:4.9606..E-02 / ep:2.6278..E-02 / qp: 1.5501..E-03]
    real   (rk), parameter      :: zeps12        = EPSILON(x)**(one/12.0_rk)
!
!   border parameter for series expansion, p. 1306, table 10, definitions are:
!    zeps(1) = [sp:2.4414..E-4/dp:1.0536..E-8/ep:2.3283..E-10/qp:9.8130..E-18]
!    zeps(2) = [sp:2.6486..E-6/dp:4.9606..E-7/ep:2.6278..E-07/qp:1.5501..E-08]
!    zeps(3) = [sp:1.3243..E-3/dp:2.4803..E-4/ep:1.3139..E-04/qp:7.7509..E-06]
!    zeps(4) = [sp:2.6486..E-2/dp:4.9606..E-3/ep:2.6278..E-03/qp:1.5501..E-04]
!    zeps(5) = [sp:1.8540..E-1/dp:3.4724..E-2/ep:1.8394..E-02/qp:1.0851..E-03]
!    zeps(6) = [sp:4.7675..E-1/dp:8.9291..E-2/ep:4.7300..E-02/qp:2.7903..E-03]
!    zeps(7) = [sp:6.0919..E-1/dp:1.1409..E-1/ep:1.8131..E-01/qp:1.6401..E-02]
    real   (rk), parameter      :: zeps(1_ik:7_ik)   =       (/ &
            SQRT( EPSILON(x)/two ),     1.000E-05_rk * zeps12,  &
            0.005E+00_rk * zeps12,      0.100E+00_rk * zeps12,  &
            0.700E+00_rk * zeps12,      1.800E+00_rk * zeps12,  &
           (2.300E+00_rk + (rk2008/qp2008)*4.60E+00_rk)*zeps12 /)
!
!   parameter for partial Series Expansion of EXP(x²) formula,
!   Zaghloul(2024), p. 1293, formula (5), 1st part, definitions are:
!    ceven( 1)  = 1 / [0! = 1                 =      1 ]           ! 01
!    ceven( 2)  = 1 / [1! = 1                 =      1 ]           ! 02
!    ceven( 3)  = 1 / [2! = 1*2               =      2 ]           ! 03
!    ceven( 4)  = 1 / [3! = 1*2*3             =      6 ]           ! 04
!    ceven( 5)  = 1 / [4! = 1*2*3*4           =     24 ]           ! 05
!    ceven( 6)  = 1 / [5! = 1*2*3*4*5         =    120 ]           ! 06
!    ceven( 7)  = 1 / [6! = 1*2*3*4*5*6       =    720 ]           ! 07
!    ceven( 8)  = 1 / [7! = 1*2*3*4*5*6*7     =   5040 ]           ! 08
!    ceven( 9)  = 1 / [8! = 1*2*3*4*5*6*7*8   =  40320 ]           ! 09
!    ceven(10)  = 1 / [9! = 1*2*3*4*5*6*7*8*9 = 362880 ]           ! 10
    real   (rk), parameter      :: ceven(1_ik:10_ik) =       (/ &
         +1.0000000000000000000000000000000000000000E+0000_rk,  &  ! 01
         +1.0000000000000000000000000000000000000000E+0000_rk,  &  ! 02
         +0.5000000000000000000000000000000000000000E+0000_rk,  &  ! 03
         +0.1666666666666666666666666666666666666667E+0000_rk,  &  ! 04
         +0.4166666666666666666666666666666666666667E-0001_rk,  &  ! 05
         +0.8333333333333333333333333333333333333333E-0002_rk,  &  ! 06
         +0.1388888888888888888888888888888888888888E-0002_rk,  &  ! 07
         +0.1984126984126984126984126984126984126981E-0003_rk,  &  ! 08
         +0.2480158730158730158730158730158730158730E-0004_rk,  &  ! 09
         +0.2755731922398589065255731922398589065256E-0005_rk  /)  ! 10
!
!   parameter for partial Sieries Expansion formula, 
!   Zaghloul(2024), p. 1293, formula (5), 2nd part, definitions are:
!    codd( 1)  = -2/√pi * 2^0/[( 1)!! = 1                       ]  ! 01
!    codd( 2)  = -2/√pi * 2^1/[( 3)!! = 1*3                     ]  ! 02
!    codd( 3)  = -2/√pi * 2^2/[( 5)!! = 1*3*5                   ]  ! 03
!    codd( 4)  = -2/√pi * 2^3/[( 7)!! = 1*3*5*7                 ]  ! 04
!    codd( 5)  = -2/√pi * 2^4/[( 9)!! = 1*3*5*7*9               ]  ! 05
!    codd( 6)  = -2/√pi * 2^5/[(11)!! = 1*3*5*7*9*11            ]  ! 06
!    codd( 7)  = -2/√pi * 2^6/[(13)!! = 1*3*5*7*9*11*13         ]  ! 07
!    codd( 8)  = -2/√pi * 2^7/[(15)!! = 1*3*5*7*9*11*13*15      ]  ! 08
!    codd( 9)  = -2/√pi * 2^8/[(17)!! = 1*3*5*7*9*11*13*15*17   ]  ! 09
!    codd(10)  = -2/√pi * 2^9/[(19)!! = 1*3*5*7*9*11*13*15*17*19]  ! 10
    real   (rk), parameter      :: codd(1_ik:10_ik) =   (/ &
         -1.1283791670955125738961589031215451716880E+0000_rk,  &  ! 01
         -0.7522527780636750492641059354143634477919E+0000_rk,  &  ! 02
         -0.3009011112254700197056423741657453791169E+0000_rk,  &  ! 03
         -0.8597174606442000563018353547592725117621E-0001_rk,  &  ! 04
         -0.1910483245876000125115189677242827803916E-0001_rk,  &  ! 05
         -0.3473605901592727500209435776805141461666E-0002_rk,  &  ! 06
         -0.5344009079373426923399131964315602248717E-0003_rk,  &  ! 07
         -0.7125345439164569231198842619087469664956E-0004_rk,  &  ! 08
         -0.8382759340193610860233932493044081958772E-0005_rk,  &  ! 09
         -8.8239572002038009055094026242569283776530E-0007_rk  /)  ! 10
!
!-----------------------------------------------------------------------------
!
!   Chebyshev Polynomials, Zaghloul (2024), page 1294f
!
!   upper limit x for Chebyshev Polynomials, page 1295f, table 1 and table 2
!   cheby = [sp: 7.80E+00 / dp: 7.80E+00 / ep or qp: 48.00E+00]
    real   (rk), parameter      :: cheby         =                    &
                           7.80E+00_rk + (rk2008 / qp2008)*40.20E+00_rk
!
!   mapping variable and parameter, page 1295, formula (8)
    real   (rk)                 :: t             ! mapping variable t=c/(x+c)
    real   (rk), parameter      :: c             = 2.10E+00_rk
    integer(ik)                 :: jmin, j, i    ! variable do...
!
!   1st definition: start and end point of the of parameters, page 1294f
!       Np         = [sp:    4 / dp:    8 / ep or qp:   12]
!       N_loop_max = [sp:    1 / dp:    2 / ep or qp:    3]
!       Np_pls_1   = [sp:    5 / dp:    9 / ep or qp:   13]
!       N_cff_1    = [sp: 1481 / dp: 1301 / ep or qp:    1], idx 1st  coeff.
!       N_dvs      = [sp:   20 / dp:   20 / ep or qp:  100]
!       N_cff      = [sp:  100 / dp:  180 / ep or qp: 1300]
!       N_cff_end  = [sp: 1580 / dp: 1480 / ep or qp: 1300], idx last coeff.
!
    integer(ik), parameter      ::                                   &
        Np         =  4_ik * (rk2008/sp2008) * (sp2008/rk2008) +     &
                      8_ik * (rk2008/dp2008) * (dp2008/rk2008) +     &
                     12_ik * (rk2008/qp2008) * (qp2008/rk2008),      &
        N_loop_max = Np / 4_ik,                                      &
        Np_pls_1   = Np + 1_ik,                                      &
        N_cff_1    =     1_ik + (1_ik - (rk2008/qp2008)) *           &
                     (1300_ik + (1_ik - (rk2008/dp2008)) * 180_ik),  &
        N_dvs      = 20_ik + 80_ik*(rk2008/qp2008),                  &
        N_cff      = N_dvs   * Np_pls_1,                             &
        N_cff_end  = N_cff_1 + N_cff - 1_ik
!

!   2nd definition: establish the table of parameters in "coeff_t20(1:1580)",
!                   coefficient table from file "4100-Erfcx-High-include.f95"
    include "4100-Erfcx-High-include.f95"
!
!   3rd definition: define the relevant parameters from "cff(1:1580)"
    real   (rk), parameter :: cffs(1_ik:N_cff) =                      &
                 reshape(  (/ cff(N_cff_1:N_cff_end) /), (/ N_cff /)  )
!
!-----------------------------------------------------------------------------
!
!   Laplace Continued Fraction as 11-term Rational Function,
!   Zaghloul (2024), page 1295f
!
!   local variable
    real   (rk)                 :: num, dnm      ! numerator, denominator
!
!   limits for the Continued Fraction, page 1296, table 2, borders are:
!   CFborder(1) = [sp: 42.00E+00 / dp: 42.00E+00 / ep or qp: 1600.00E+00]
!   CFborder(2) = [sp: 17.20E+00 / dp: 17.20E+00 / ep or qp:  300.00E+00]
!   CFborder(3) = [sp: 10.40E+00 / dp: 10.40E+00 / ep or qp:  100.00E+00]
!   CFborder(4) = [sp:  7.80E+00 / dp:  7.80E+00 / ep or qp:   48.00E+00]
    real   (rk), parameter      :: CFborder(1_ik: 4_ik) =          (/ &
                  42.00E+00_rk + ( rk2008/qp2008 ) * 1558.00E+00_rk,  &
                  17.20E+00_rk + ( rk2008/qp2008 ) *  283.80E+00_rk,  &
                  10.40E+00_rk + ( rk2008/qp2008 ) *   89.60E+00_rk,  &
                   7.80E+00_rk + ( rk2008/qp2008 ) *   40.20E+00_rk  /)
!
!   11-term Rational Function parameters, Zaghloul (2024), p.1296, f. (9):
!
!   5-term
!   CFnum =                                   x^4 +    4.5000x^2 +   2.00000
!   -----   -----------------------------------------------------------------
!   CFdnm =                                   x^4 +    5.0000x^2 +   3.75000)
!
!   7-term
!   CFnum =                       x^6 +  10.00x^4 +   21.7500x^2 +   6.00000
!   -----   -----------------------------------------------------------------
!   CFdnm =                       x^6 +  10.50x^4 +   26.2500x^2 +  13.12500)
!
!   9-term
!   CFnum =            x^8 +  17.5x^6 +  86.25x^4 +  121.8750x^2 +  24.00000
!   -----   -----------------------------------------------------------------
!   CFdnm =            x^8 +  18.0x^6 +  94.50x^4 +  157,5000x^2 +  59.06250)
!
!   11-term
!   CFnum = x^10 + 27.0x^8 + 234.5x^6 + 761.25x^4 +  790.3125x^2 + 120.00000
!   -----   -----------------------------------------------------------------
!   CFdnm = x^10 + 27.5x^8 + 247.5x^6 + 866.25x^4 + 1082.8125x^2 + 324.84375)
!
    real   (rk), parameter      :: CFnum(1_ik:14_ik) =                    (/ &
                                               4.5000E+0_rk,  2.00000E+0_rk, &
                               10.000E+0_rk,  21.7500E+0_rk,  6.00000E+0_rk, &
                 17.500E+0_rk, 86.250E+0_rk, 121.8750E+0_rk, 24.00000E+0_rk, &
     27.0E+0_rk,234.500E+0_rk,761.250E+0_rk, 790.3125E+0_rk,120.00000E+0_rk /)
    real   (rk), parameter      :: Cfdnm(1_ik:14_ik) =                    (/ &
                                               5.0000E+0_rk,  3.75000E+0_rk, &
                               10.500E+0_rk,  26.2500E+0_rk, 13.12500E+0_rk, &
                 18.000E+0_rk, 94.500E+0_rk, 157.5000E+0_rk, 59.06250E+0_rk, &
     27.5E+0_rk,247.500E+0_rk,866.250E+0_rk,1082.8125E+0_rk,324.84375E+0_rk /)
!
!-----------------------------------------------------------------------------
!
!   parameter for "big", upper limit for Laplace Continued Fraction,
!                        Zaghloul (2024), page 1296, table 2 
!                        = [ sp: +42.10E+00 / dp: +10042.10E+00 / 
!                            ep or qp: +6.60000421E+07 = +66000042.10E+00 ]
!
    real   (rk), parameter      :: big           =                          &
     42.10E+00_rk + (rk2008/dp2008)*1.00E+04_rk + (rk2008/qp2008)*6.60E+07_rk
!
!-----------------------------------------------------------------------------
!
!   parameter for "vbig", upper limit for Approximation 1/(√pi*(z + 0.5/z)),
!                         Zaghloul (2024), page 1296, table 2
!                         = [ sp: +6.7100E+03 / dp: +6.7100E+09 /
!                             ep: +6.7100E+15 / qp: +6.7100E+15  ]
!
    real   (rk), parameter      :: vbig          =                          &
                 6.71E+00_rk * 10.00E+00_rk ** (15_ik - 6_ik*(dp2008/rk2008))
!
!-----------------------------------------------------------------------------
!
!
!   start of cases and calculations
!
!   -∞ ≤ x ≤ (limiterfcxL = -SQRT( LOG( HUGE(one)/two )
!                         = [ sp:   -9.382..E+00 / dp:  -26.628..E+00 /
!                             ep: -106.563..E+00 / qp: -106.563..E+00 ]  )
         if( x <= limiterfcxL ) then             ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
!   limiterfcxL < x < (-9.00E+00, Zaghloul (2024), page 1303)
    else if( x < -9.00E+00_rk ) then             ! x < -9.0
             f = two * EXP( x*x )                ! f = 2 * EXP(x²)
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z=|x| : 0,...,+∞
!
!   0.00 ≤ z ≤ ( zeps(1) = SQRT( EPSILON(x)/two )
!                        = [sp: 2.4414..E-04 / dp: 1.0536..E-08 /
!                           ep: 2.3283..E-10 / qp: 9.8130..E-18 ]  )
         if( z <= zeps(1_ik) ) then
             f = one - twodivsqrtpi * x          ! 1 - (2/√pi) * x
!
!   zeps(1) < z ≤ ( zeps(7) = [ sp: 6.0919..E-01 / dp: 1.1409..E-01 /
!                               ep: 1.8131..E-01 / qp: 1.6401..E-02 ]  )
    else if( z <= zeps(7_ik) ) then
!            partial Series Expansion for z=|x| << 1, Zaghloul (2024),
!            page 1293, formula (5) with precalculated parameters
             zz = z * z                          ! zz = z² = |x|²
             f  = zero                           ! function value
             do j = 2_ik, 7_ik, 1_ik             ! do interval
                if( z <= zeps(j) ) then          ! check limit
                    do i = j, 2_ik, -1_ik        ! calculate
                       f = zz * ( f + ceven(i) + codd(i) * x )
                    end do                       ! end calculate
                    f = f + ceven(1_ik) + codd(1_ik) * x
                    exit                         ! exit j loop
                end if                           ! end if check
             end do                              ! end if interval
!
!   zeps(7) < z ≤ ( cheby = [ sp:  +7.80E+00 / dp:  +7.80E+00 /
!                             ep: +48.00E+00 / qp: +48.00E+00 ]  )
    else if( z <= cheby ) then
!            Chebyshev 4-step subinterval polynomial approximation,
!            Zaghloul (2024), page 1294f
             t    = c / (z + c)                  ! p.1295, f(8), c=2.1
             jmin = n_dvs * t
             jmin = jmin  * np_pls_1 + 1_ik
             f    = zero
             do j = 1_ik, n_loop_max, 1_ik
                f    = (((( f + cffs(jmin       ) ) * t +   &
                                cffs(jmin + 1_ik) ) * t +   &
                                cffs(jmin + 2_ik) ) * t +   &
                                cffs(jmin + 3_ik) ) * t
                jmin = jmin + 4_ik
             end do
             f = f + cffs(jmin)
             if( x < zero ) f = two*EXP(z*z) -f ! erfcx(-x)=2*exp(z²)-erfcx(z)
!
!   cheby < z ≤ ( big = [ sp:       +42.10E+00 / dp:    +10042.10E+00 /
!                         ep: +66000042.10E+00 / qp: +66000042.10E+00  ]  )
    else if( z <= big ) then
!            Laplace Continued Fraction in up to 11-term rational function
!            form, page 1296, formula (9)
             zz  = z * z                         ! zz = z² = |x|²
             num = one                           ! prepare numerator
             dnm = one                           ! prepare denominator
             do j = 1_ik, 4_ik, 1_ik             ! do term
                if( CFborder(j) <= z ) then      ! check border
                    i = (j * (1_ik + j)) / 2_ik  ! prepare i
                    do jmin = i, i + j           ! do calculate
                       num = zz * num  + CFnum(jmin) ! calc. numerator
                       dnm = zz * dnm  + CFdnm(jmin) ! calc. denominator
                    end do                       ! end do calculation
                    f = (onedivsqrtpi / z) * num/dnm ! finalize f
                    exit                         ! exits do j
                end if                           ! end if border
             end do                              !end do term
             if( x < zero ) f = two*EXP(zz) -f  ! erfcx(-x)=2*exp(z²)-erfcx(z)
!
!   big < z ≤ ( vbig = [ sp: +6.71E+03 / dp: +6.71E+09 /
!                        ep: +6.71E+15 / qp: +6.71E+15 ]  )
    else if( z <= vbig ) then                    ! Zaghloul(2024), p.1296, t.2
             f = onedivsqrtpi / (z + half/z)     ! 1/(√pi*(z + 0.5/z))
!
!   vbig < z ≤ ( +∞ = HUGE(x) = 
!                     [ sp: +3.40..E+0038 / dp: +1.79..E+0308 /
!                       ep: +3.36..E+4932 / qp: +3.36..E+4932 ]  )
    else                                         ! Zaghloul(2024), p.1296, t.2
             f = onedivsqrtpi / z                ! 1/(√pi*z)
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcxZaghloul
!
!-----------------------------------------------------------------------------
!
!   erfcxZaghloulopt
!     Scaled Complementary Error Function f = erfcx(x) with various
!     algorithms/calculations referenced Zaghloul (2024)
!   Reference:
!     Mofreh R. Zaghloul, "Efficient multiple-precision computation of the
!      scaled complementary error function and the Dawson integral", Numerical
!      Algorithms, 2024, Volume 95, Issue 3, Pages 1291–1308,
!      https://doi.org/10.1007/s11075-023-01608-8
!     Series Expansion:
!      M.M. Shepherd and J.G. Laframboise, "Chebyshev Approximation of
!       (1 + 2x)exp(x²)erfc x in 0 ≤ x < ∞", Mathematics of Computation,
!       1981, Volume 36, Issue 153, Pages 249-253,
!       https://doi.org/10.1090/S0025-5718-1981-0595058-X
!       here: page 251, formula (7)
!     Continued Fraction:
!      Irene A. Stegun and Ruth Zucker, "Automatic Computing Methods for
!       Special Functions", Journal of Research of the National Bureau of
!       Standards - B. Mathematical Sciences, 1970, Volume 74B, Issue 3,
!       Pages 211-224, http://dx.doi.org/10.6028/jres.074B.019
!       here: page 214, 4th redefined formula
!   Remark:
!     Source, Copyright and Approval
!      The function "erfcxZaghloul" is derived from the original
!      source function "erfcx_rk" by M.R. Zaghloul, a copyright
!      protected source. For more details and approval see the
!      chapter "Copyright and Approvals".
!      A written approval is available since 26. August 2024 .
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )   := +∞ = +HUGE(x)
!     Parameter "limiterfcxL1" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL1 )  := 2.0 * EXP(x²)
!     Parameter "sqrteps" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ sqrteps )       := 1.0 - (2.0/√pi) * z
!     Parameter "limiterfcxBig" is defined in "Constants" for erfcx,
!      where erfcx( z < limiterfcxBig ) := 1.0/(√pi*(z + 0.5/z))
!     Parameter "limiterfcxH" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ limiterfcxH )   := (1/√pi)/x = 1/(√pi*x)
!      where erfcx( z > limiterfcxH )   := 0.0
!     The function requires the file "4100-Erfcx-High-include.f95"
!      with the parameters for the Chebyshev Polynomials
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxZaghloulopt( x )   result( f )
!
    use kinds, only : ik, rk, sp2008, dp2008, qp2008, rk2008
    use const, only : zero, half, one, two,                &
                      onedivsqrtpi, twodivsqrtpi,          &
                      limiterfcxL, limiterfcxL1, sqrteps,  &
                      limiterfcxBig, limiterfcxH
!
    implicit none
!
!   interface
     real   (rk), intent(in )   :: x             ! x from erfcx(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
    real   (rk)                 :: zz            ! zz = (z=|x|)² = z*z
!
!-----------------------------------------------------------------------------
!
!   Series Expansion for z=|x| << 1
!   Zaghloul (2024), page 1293, formula (5) (or compare: Shepherd/Laframboise
!   (1981), page 251, formula (7))
!
!   "zeps12" (Zaghloul (2024), p.1306, t.10) for the entry parameters =
!   = [sp:2.6486..E-01 / dp:4.9606..E-02 / ep:2.6278..E-02 / qp: 1.5501..E-03]
    real   (rk), parameter      :: zeps12        = EPSILON(x)**(one/12.0_rk)
!
!   border parameter for series expansion, p. 1306, table 10, definitions are:
!    zeps(1) = [sp:2.4414..E-4/dp:1.0536..E-8/ep:2.3283..E-10/qp:9.8130..E-18]
!    zeps(2) = [sp:2.6486..E-6/dp:4.9606..E-7/ep:2.6278..E-07/qp:1.5501..E-08]
!    zeps(3) = [sp:1.3243..E-3/dp:2.4803..E-4/ep:1.3139..E-04/qp:7.7509..E-06]
!    zeps(4) = [sp:2.6486..E-2/dp:4.9606..E-3/ep:2.6278..E-03/qp:1.5501..E-04]
!    zeps(5) = [sp:1.8540..E-1/dp:3.4724..E-2/ep:1.8394..E-02/qp:1.0851..E-03]
!    zeps(6) = [sp:4.7675..E-1/dp:8.9291..E-2/ep:4.7300..E-02/qp:2.7903..E-03]
!    zeps(7) = [sp:6.0919..E-1/dp:1.1409..E-1/ep:1.8131..E-01/qp:1.6401..E-02]
    real   (rk), parameter      :: zeps(1_ik:7_ik)   =       (/ &
            SQRT( EPSILON(x)/two ),     1.000E-05_rk * zeps12,  &
            0.005E+00_rk * zeps12,      0.100E+00_rk * zeps12,  &
            0.700E+00_rk * zeps12,      1.800E+00_rk * zeps12,  &
           (2.300E+00_rk + (rk2008/qp2008)*4.60E+00_rk)*zeps12 /)
!
!   parameter for partial Series Expansion of EXP(x²) formula,
!   Zaghloul(2024), p. 1293, formula (5), 1st part, definitions are:
!    ceven( 1)  = 1 / [0! = 1                 =      1 ]           ! 01
!    ceven( 2)  = 1 / [1! = 1                 =      1 ]           ! 02
!    ceven( 3)  = 1 / [2! = 1*2               =      2 ]           ! 03
!    ceven( 4)  = 1 / [3! = 1*2*3             =      6 ]           ! 04
!    ceven( 5)  = 1 / [4! = 1*2*3*4           =     24 ]           ! 05
!    ceven( 6)  = 1 / [5! = 1*2*3*4*5         =    120 ]           ! 06
!    ceven( 7)  = 1 / [6! = 1*2*3*4*5*6       =    720 ]           ! 07
!    ceven( 8)  = 1 / [7! = 1*2*3*4*5*6*7     =   5040 ]           ! 08
!    ceven( 9)  = 1 / [8! = 1*2*3*4*5*6*7*8   =  40320 ]           ! 09
!    ceven(10)  = 1 / [9! = 1*2*3*4*5*6*7*8*9 = 362880 ]           ! 10
    real   (rk), parameter      :: ceven(1_ik:10_ik) =       (/ &
         +1.0000000000000000000000000000000000000000E+0000_rk,  &  ! 01
         +1.0000000000000000000000000000000000000000E+0000_rk,  &  ! 02
         +0.5000000000000000000000000000000000000000E+0000_rk,  &  ! 03
         +0.1666666666666666666666666666666666666667E+0000_rk,  &  ! 04
         +0.4166666666666666666666666666666666666667E-0001_rk,  &  ! 05
         +0.8333333333333333333333333333333333333333E-0002_rk,  &  ! 06
         +0.1388888888888888888888888888888888888888E-0002_rk,  &  ! 07
         +0.1984126984126984126984126984126984126981E-0003_rk,  &  ! 08
         +0.2480158730158730158730158730158730158730E-0004_rk,  &  ! 09
         +0.2755731922398589065255731922398589065256E-0005_rk  /)  ! 10
!
!   parameter for partial Sieries Expansion formula, 
!   Zaghloul(2024), p. 1293, formula (5), 2nd part, definitions are:
!    codd( 1)  = -2/√pi * 2^0/[( 1)!! = 1                       ]  ! 01
!    codd( 2)  = -2/√pi * 2^1/[( 3)!! = 1*3                     ]  ! 02
!    codd( 3)  = -2/√pi * 2^2/[( 5)!! = 1*3*5                   ]  ! 03
!    codd( 4)  = -2/√pi * 2^3/[( 7)!! = 1*3*5*7                 ]  ! 04
!    codd( 5)  = -2/√pi * 2^4/[( 9)!! = 1*3*5*7*9               ]  ! 05
!    codd( 6)  = -2/√pi * 2^5/[(11)!! = 1*3*5*7*9*11            ]  ! 06
!    codd( 7)  = -2/√pi * 2^6/[(13)!! = 1*3*5*7*9*11*13         ]  ! 07
!    codd( 8)  = -2/√pi * 2^7/[(15)!! = 1*3*5*7*9*11*13*15      ]  ! 08
!    codd( 9)  = -2/√pi * 2^8/[(17)!! = 1*3*5*7*9*11*13*15*17   ]  ! 09
!    codd(10)  = -2/√pi * 2^9/[(19)!! = 1*3*5*7*9*11*13*15*17*19]  ! 10
    real   (rk), parameter      :: codd(1_ik:10_ik) =   (/ &
         -1.1283791670955125738961589031215451716880E+0000_rk,  &  ! 01
         -0.7522527780636750492641059354143634477919E+0000_rk,  &  ! 02
         -0.3009011112254700197056423741657453791169E+0000_rk,  &  ! 03
         -0.8597174606442000563018353547592725117621E-0001_rk,  &  ! 04
         -0.1910483245876000125115189677242827803916E-0001_rk,  &  ! 05
         -0.3473605901592727500209435776805141461666E-0002_rk,  &  ! 06
         -0.5344009079373426923399131964315602248717E-0003_rk,  &  ! 07
         -0.7125345439164569231198842619087469664956E-0004_rk,  &  ! 08
         -0.8382759340193610860233932493044081958772E-0005_rk,  &  ! 09
         -8.8239572002038009055094026242569283776530E-0007_rk  /)  ! 10
!
!-----------------------------------------------------------------------------
!
!   Chebyshev Polynomials, Zaghloul (2024), page 1294f
!
!   upper limit x for Chebyshev Polynomials, page 1295f, table 1 and table 2
!   cheby = [sp: 7.80E+00 / dp: 7.80E+00 / ep or qp: 48.00E+00]
    real   (rk), parameter      :: cheby         =                    &
                           7.80E+00_rk + (rk2008 / qp2008)*40.20E+00_rk
!
!   mapping variable and parameter, page 1295, formula (8)
    real   (rk)                 :: t             ! mapping variable t=c/(x+c)
    real   (rk), parameter      :: c             = 2.10E+00_rk
    integer(ik)                 :: jmin, j, i    ! variable do...
!
!   1st definition: start and end point of the of parameters, page 1294f
!       Np         = [sp:    4 / dp:    8 / ep or qp:   12]
!       N_loop_max = [sp:    1 / dp:    2 / ep or qp:    3]
!       Np_pls_1   = [sp:    5 / dp:    9 / ep or qp:   13]
!       N_cff_1    = [sp: 1481 / dp: 1301 / ep or qp:    1], idx 1st  coeff.
!       N_dvs      = [sp:   20 / dp:   20 / ep or qp:  100]
!       N_cff      = [sp:  100 / dp:  180 / ep or qp: 1300]
!       N_cff_end  = [sp: 1580 / dp: 1480 / ep or qp: 1300], idx last coeff.
!
    integer(ik), parameter      ::                                   &
        Np         =  4_ik * (rk2008/sp2008) * (sp2008/rk2008) +     &
                      8_ik * (rk2008/dp2008) * (dp2008/rk2008) +     &
                     12_ik * (rk2008/qp2008) * (qp2008/rk2008),      &
        N_loop_max = Np / 4_ik,                                      &
        Np_pls_1   = Np + 1_ik,                                      &
        N_cff_1    =     1_ik + (1_ik - (rk2008/qp2008)) *           &
                     (1300_ik + (1_ik - (rk2008/dp2008)) * 180_ik),  &
        N_dvs      = 20_ik + 80_ik*(rk2008/qp2008),                  &
        N_cff      = N_dvs   * Np_pls_1,                             &
        N_cff_end  = N_cff_1 + N_cff - 1_ik
!
!   2nd definition: establish the table of parameters in "coeff_t20(1:1580)",
!                   coefficient table from file "4100-Erfcx-High-include.f95"
    include "4100-Erfcx-High-include.f95"
!
!   3rd definition: define the relevant parameters from "cff(1:1580)"
    real   (rk), parameter :: cffs(1_ik:N_cff) =                      &
                 reshape(  (/ cff(N_cff_1:N_cff_end) /), (/ N_cff /)  )
!
!-----------------------------------------------------------------------------
!
!   Laplace Continued Fraction as 11-term Rational Function,
!   Zaghloul (2024), page 1295f
!
!   local variable
    real   (rk)                 :: num, dnm      ! numerator, denominator
!
!   limits for the Continued Fraction, page 1296, table 2, borders are:
!   CFborder(1) = [sp: 42.00E+00 / dp: 42.00E+00 / ep or qp: 1600.00E+00]
!   CFborder(2) = [sp: 17.20E+00 / dp: 17.20E+00 / ep or qp:  300.00E+00]
!   CFborder(3) = [sp: 10.40E+00 / dp: 10.40E+00 / ep or qp:  100.00E+00]
!   CFborder(4) = [sp:  7.80E+00 / dp:  7.80E+00 / ep or qp:   48.00E+00]
    real   (rk), parameter      :: CFborder(1_ik: 4_ik) =          (/ &
                  42.00E+00_rk + ( rk2008/qp2008 ) * 1558.00E+00_rk,  &
                  17.20E+00_rk + ( rk2008/qp2008 ) *  283.80E+00_rk,  &
                  10.40E+00_rk + ( rk2008/qp2008 ) *   89.60E+00_rk,  &
                   7.80E+00_rk + ( rk2008/qp2008 ) *   40.20E+00_rk  /)
!
!   11-term Rational Function parameters, Zaghloul (2024), p.1296, f. (9):
!
!   5-term
!   CFnum =                                   x^4 +    4.5000x^2 +   2.00000
!   -----   -----------------------------------------------------------------
!   CFdnm =                                   x^4 +    5.0000x^2 +   3.75000)
!
!   7-term
!   CFnum =                       x^6 +  10.00x^4 +   21.7500x^2 +   6.00000
!   -----   -----------------------------------------------------------------
!   CFdnm =                       x^6 +  10.50x^4 +   26.2500x^2 +  13.12500)
!
!   9-term
!   CFnum =            x^8 +  17.5x^6 +  86.25x^4 +  121.8750x^2 +  24.00000
!   -----   -----------------------------------------------------------------
!   CFdnm =            x^8 +  18.0x^6 +  94.50x^4 +  157,5000x^2 +  59.06250)
!
!   11-term
!   CFnum = x^10 + 27.0x^8 + 234.5x^6 + 761.25x^4 +  790.3125x^2 + 120.00000
!   -----   -----------------------------------------------------------------
!   CFdnm = x^10 + 27.5x^8 + 247.5x^6 + 866.25x^4 + 1082.8125x^2 + 324.84375)
!
    real   (rk), parameter      :: CFnum(1_ik:14_ik) =                    (/ &
                                               4.5000E+0_rk,  2.00000E+0_rk, &
                               10.000E+0_rk,  21.7500E+0_rk,  6.00000E+0_rk, &
                 17.500E+0_rk, 86.250E+0_rk, 121.8750E+0_rk, 24.00000E+0_rk, &
     27.0E+0_rk,234.500E+0_rk,761.250E+0_rk, 790.3125E+0_rk,120.00000E+0_rk /)
    real   (rk), parameter      :: Cfdnm(1_ik:14_ik) =                    (/ &
                                               5.0000E+0_rk,  3.75000E+0_rk, &
                               10.500E+0_rk,  26.2500E+0_rk, 13.12500E+0_rk, &
                 18.000E+0_rk, 94.500E+0_rk, 157.5000E+0_rk, 59.06250E+0_rk, &
     27.5E+0_rk,247.500E+0_rk,866.250E+0_rk,1082.8125E+0_rk,324.84375E+0_rk /)
!
!-----------------------------------------------------------------------------
!
!   parameter for "big", upper limit for Laplace Continued Fraction,
!                        Zaghloul (2024), page 1296, table 2 
!                        = [ sp: +42.10E+00 / dp: +10042.10E+00 / 
!                            ep or qp: +6.60000421E+07 = +66000042.10E+00 ]
!
    real   (rk), parameter      :: big           =                          &
     42.10E+00_rk + (rk2008/dp2008)*1.00E+04_rk + (rk2008/qp2008)*6.60E+07_rk
!
!-----------------------------------------------------------------------------
!
!
!   start of cases and calculations
!
!   -∞ ≤ x ≤ ( limiterfcxL = -SQRT( LOG( HUGE(one)/two ) ) =
!                          = [ sp:   -9.3824...E+00 / dp:  -26.6287...E+00 /
!                              ep: -106.5637...E+00 / qp: -106.5637...E+00 ] )
         if( x <= limiterfcxL  ) then            ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
!   limiterfcxL < x ≤ (limiterfcxL1 = -SQRT(-LOG( sqrtpi*EPSILON(one)/two)) =
!                      [ sp: -4.0078...E+00 / dp: -6.0136...E+00 /
!                        ep: -6.6173...E+00 / qp: -8.8177...E+00])
!   limiterfcxL1 defines erfc(x) ≈ 2 => erfcx(x)=2*exp(x²)
    else if( x <= limiterfcxL1 ) then            ! cut off "neg." at erfc(x)=2
             f = two * EXP( x*x )                ! f = 2.00 * exp(x²)
!
    else                                         ! continue with |x|
             z  = ABS( x )                       ! z=|x| : 0, ..., +∞
!
!
!   0.0 ≤ z=|x| ≤ ( sqrteps = SQRT( EPSILON(one) ) =
!                           = [ sp: 3.452..E-04 / dp: 1.4901..E-08 /
!                               ep: 3.292..E-10 / qp: 1.3877..E-17 ] )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1 - (2/√pi) * x
!
!-----------------------------------------------------------------------------
!
!   sqrteps < z ≤ ( zeps(7) = [ sp: 6.0919..E-01 / dp: 1.1409..E-01 /
!                               ep: 1.8131..E-01 / qp: 1.6401..E-02 ]  )
    else if( z <= zeps(7_ik) ) then
!            partial Series Expansion for z=|x| << 1, Zaghloul (2024),
!            page 1293, formula (5) with precalculated parameters
             zz = z * z                          ! zz = z² = |x|²
             f  = zero                           ! function value
             do j = 2_ik, 7_ik, 1_ik             ! do interval
                if( z <= zeps(j) ) then          ! check limit
                    do i = j, 2_ik, -1_ik        ! calculate
                       f = zz * ( f + ceven(i) + codd(i) * x )
                    end do                       ! end calculate
                    f = f + ceven(1_ik) + codd(1_ik) * x
                    exit                         ! exit j loop
                end if                           ! end if check
             end do                              ! end if interval
!
!   zeps(7) < z ≤ ( cheby = [ sp:  +7.80E+00 / dp:  +7.80E+00 /
!                             ep: +48.00E+00 / qp: +48.00E+00 ]  )
    else if( z <= cheby ) then
!            Chebyshev 4-step subinterval polynomial approximation,
!            Zaghloul (2024), page 1294f
             t    = c / (z + c)                  ! p.1295, f(8), c=2.1
             jmin = n_dvs * t
             jmin = jmin  * np_pls_1 + 1_ik
             f    = zero
             do j = 1_ik, n_loop_max, 1_ik
                f    = (((( f + cffs(jmin       ) ) * t +   &
                                cffs(jmin + 1_ik) ) * t +   &
                                cffs(jmin + 2_ik) ) * t +   &
                                cffs(jmin + 3_ik) ) * t
                jmin = jmin + 4_ik
             end do
             f = f + cffs(jmin)
             if( x < zero ) f = two*EXP(z*z) -f ! erfcx(-x)=2*exp(z²)-erfcx(z)
!
!   cheby < z ≤ ( big = [ sp:       +42.10E+00 / dp:    +10042.10E+00 /
!                         ep: +66000042.10E+00 / qp: +66000042.10E+00  ]  )
    else if( z <= big ) then
!            Laplace Continued Fraction in up to 11-term rational function
!            form, page 1296, formula (9)
             zz  = z * z                         ! zz = z² = |x|²
             num = one                           ! prepare numerator
             dnm = one                           ! prepare denominator
             do j = 1_ik, 4_ik, 1_ik             ! do term
                if( CFborder(j) <= z ) then      ! check border
                    i = (j * (1_ik + j)) / 2_ik  ! prepare i
                    do jmin = i, i + j           ! do calculate
                       num = zz * num  + CFnum(jmin) ! calc. numerator
                       dnm = zz * dnm  + CFdnm(jmin) ! calc. denominator
                    end do                       ! end do calculation
                    f = (onedivsqrtpi / z) * num/dnm ! finalize f
                    exit                         ! exits do j
                end if                           ! end if border
             end do                              !end do term
             if( x < zero ) f = two*EXP(zz) -f  ! erfcx(-x)=2*exp(z²)-erfcx(z)
!
!-----------------------------------------------------------------------------
!
!   big < z=|x| ≤ ( limiterfcxBig = SQRT( one/EPSILON(x) ) =
!                                 = [ sp: 2.8963..E+03 / dp: 6.7108..E+07 /
!                                     ep: 3.0370..E+09 / qp: 7.2057..E+16 ] )
    else if( z <= limiterfcxBig ) then
             f = onedivsqrtpi / (z + half/z)     ! 1/(√pi*(z + 0.5/z))
!
!-----------------------------------------------------------------------------
!
!   limiterfcxBig < z=|x| ≤ ( limiterfcxH = one / ( sqrtpi * TINY(one) ) =
!                             = [ sp: 4.7995...E+0037 / dp: 2.5355...E+0307 / 
!                                 ep: 1.6780...E+4931 / qp: 1.6780...E+4931 ])
    else if( z <= limiterfcxH ) then
             f = onedivsqrtpi / z                ! (1/√pi)/z = 1/(√pi*z)
!
!-----------------------------------------------------------------------------
!
!   limiterfcxH < z ≤ ( +∞ = HUGE(x) = 
!                            [ sp: +3.40..E+0038 / dp: +1.79..E+0308 /
!                              ep: +1.18..E+4932 / qp: +1.18..E+4932 ]  )
    else
             f = zero                            ! f = 0.0
!
!-----------------------------------------------------------------------------
!
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcxZaghloulopt
!
!-----------------------------------------------------------------------------
!
!   erfcxTH
!     Scaled Complementary Error Function erfcx(x) with various approximations
!     composed by Hoering (2024)
!   Reference:
!     Series Expansion (SE)
!      Shanjie Zhang and Jianming Jin, "Computation of Special
!       Functions", 1996, XXVI, 717 pages, John Wiley & Sons Inc.,
!       New York, ISBN: 0-471-11963-6
!       here: page 621, formula (16.1.4), reformulated SE by Hoering
!     Rational Approximation (without table LookUp)
!      Reformulated algorithm/function erfcOoura8a.f95 by Hoering:
!       Copyright(C) 1996 Takuya Ooura, email:ooura@mmm.t.u-tokyo.ac.jp
!       You may use, copy, modify this code for any purpose and without
!       fee. You may distribute this ORIGINAL package.
!       Internet: http://www.kurims.kyoto-u.ac.jp/~ooura/gamerf.html
!       here: compare with "erfcxOoura16.f95"
!     Asymptotic Series Expansion (ASE)
!      Shanjie Zhang and Jianming Jin, "Computation of Special
!       Functions", 1996, XXVI, 717 pages, John Wiley & Sons Inc.,
!       New York, ISBN: 0-471-11963-6,
!       here: page 621, formula (16.1.5), reformulated ASE by Hoering
!      Mofreh R. Zaghloul, "Efficient multiple-precision computation of
!       the scaled complementary error function and the Dawson integral",
!       Numerical Algorithms, 2024, Volume 95, Issue 3, Pages 1291–1308,
!       https://doi.org/10.1007/s11075-023-01608-8
!       here: page 1296, table 2 (borders for the ASE)
!   Remarks:
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )   := +∞ = +HUGE(x)
!     Parameter "limiterfcxL1" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL1 )  := 2.0 * EXP(x²)
!     Parameter "sqrteps" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ sqrteps )       := 1.0 - (2.0/√pi) * z
!     Parameter "limiterfcxBig" is defined in "Constants" for erfcx,
!      where erfcx( z < limiterfcxBig ) := 1.0/(√pi*(z + 0.5/z))
!     Parameter "limiterfcxH" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ limiterfcxH )   := (1/√pi)/x = 1/(√pi*x)
!      where erfcx( z > limiterfcxH )   := 0.0
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxTH( x )   result( f )
!
    use kinds, only : ik, rki, rk
    use const, only : zero, half, one, two, four,          &
                      sqrtpi, onedivsqrtpi, twodivsqrtpi,  &
                      eps, eMax, sqrteps,                  &
                      limiterfcxL,   limiterfcxL1,         &
                      limiterfcxBig, limiterfcxH
!
    implicit none
!
!   interface
     real   (rk), intent(in )   :: x             ! x from erfcx(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
    real   (rk)                 :: zz            ! zz = z² = |x|²
    integer(ik)                 :: i             ! do ...
!
!-----------------------------------------------------------------------------
!
!   Power Series Expansion (PSE),
!   Zhang/Jin (1996), p. 621, f. (16.1.4), reformulated PSE by Hoering (2024)
!
!   1. upper limit for Power Series Expansion (PSE)
!   PSE := [ sp: 0.250E+00 / dp: 3.000E-01 / ep: 3.000E-02 / qp: 3.000E-02 ]
    real   (rk), parameter      :: PSEp(1_ik:3_ik) =            (/ &
                                                     0.250E+00_rk, & ! sp
                                                     3.000E-01_rk, & ! dp
                                                     3.000E-02_rk /) ! ep/qp
    real   (rk), parameter      :: PSE             = PSEp( rki )
!
!   2. Power Series Expansion  referenced Zhang/Jin (1996), p.621, f.(16.1.4)
!   reformulated PSE by Hoering (2024), parameters: +1.0/(k + 0.5)
    real   (rk)                 :: sPSE            ! incremental sum
    integer(ik), parameter      :: PSEmax          = 50_ik  ! max. parameters
    real   (rk), parameter      :: qx(1_ik:PSEmax) =               (/ &  ! 00
     one/ 1.5E+0_rk, one/ 2.5E+0_rk, one/ 3.5E+0_rk, one/ 4.5E+0_rk,  &  ! 04
     one/ 5.5E+0_rk, one/ 6.5E+0_rk, one/ 7.5E+0_rk, one/ 8.5E+0_rk,  &  ! 08
     one/ 9.5E+0_rk, one/10.5E+0_rk, one/11.5E+0_rk, one/12.5E+0_rk,  &  ! 12
     one/13.5E+0_rk, one/14.5E+0_rk, one/15.5E+0_rk, one/16.5E+0_rk,  &  ! 16
     one/17.5E+0_rk, one/18.5E+0_rk, one/19.5E+0_rk, one/20.5E+0_rk,  &  ! 20
     one/21.5E+0_rk, one/22.5E+0_rk, one/23.5E+0_rk, one/24.5E+0_rk,  &  ! 24
     one/25.5E+0_rk, one/26.5E+0_rk, one/27.5E+0_rk, one/28.5E+0_rk,  &  ! 28
     one/29.5E+0_rk, one/30.5E+0_rk, one/31.5E+0_rk, one/32.5E+0_rk,  &  ! 32
     one/33.5E+0_rk, one/34.5E+0_rk, one/35.5E+0_rk, one/36.5E+0_rk,  &  ! 36
     one/37.5E+0_rk, one/38.5E+0_rk, one/39.5E+0_rk, one/40.5E+0_rk,  &  ! 40
     one/41.5E+0_rk, one/42.5E+0_rk, one/43.5E+0_rk, one/44.5E+0_rk,  &  ! 44
     one/45.5E+0_rk, one/46.5E+0_rk, one/47.5E+0_rk, one/48.5E+0_rk,  &  ! 48
     one/49.5E+0_rk, one/50.5E+0_rk                                /)    ! 50
!
!-----------------------------------------------------------------------------
!
!   Rational Function (RF),
!   Ooura (1996), referenced in Ooura8b.f95 and Ooura16.f95, reformulated RF
!   by Hoering (2024)
!
!   1. upper limit for the Rational Function (RF)
!   RF := [ sp: eMax=9.195.. / dp: eMax=26.542.. / ep or qp: 70.000 ]
    real   (rk), parameter      :: RFp(1_ik:3_ik)  =            (/ &
                                                             eMax, & ! sp
                                                             eMax, & ! dp
                                                      70.0E+00_rk /) ! ep/qp
    real   (rk), parameter      :: RF              = RFp( rki )
!
!   2. parameters "pv" and "ph"
!   Rational Function referenced Ooura (1996), see "erfcOoura8b.f95" and
!   "erfcOoura16.f95", where here "dp" is set identically to "sp"
!
!   parameters "pv"
    real   (rk), parameter      :: pvp(1_ik:3_ik)  =            (/ &
                                       1.26974899965115684E+01_rk, & ! sp
                                       1.26974899965115684E+01_rk, & ! dp
                   1.8296570980424689847157930974106706835E+01_rk /) ! ep/qp
    real   (rk), parameter      :: pv              = pvp( rki )
!
!   parameters "ph"
    real   (rk), parameter      :: php(1_ik:3_ik)  =            (/ &
                                      6.10399733098688199E+00_rk,  & ! sp
                                      6.10399733098688199E+00_rk,  & ! dp
                  8.9588287394342176848213494031807385567E+00_rk  /) ! ep/qp
    real   (rk), parameter      :: ph              = php( rki )
!
!   3. parameter vectors p and q for Rational Function (RF)
!   parameters p(1: 8) and q(1: 8) for kind=1 (sp) and kind=3 (dp)
!   parameters p(9:25) and q(9:25) for kind=3 (ep or qp)
!
    integer(ik), parameter      :: from(1_ik:3_ik) = (/ 1, 1,  9 /)  ! start
    integer(ik), parameter      :: to  (1_ik:3_ik) = (/ 8, 8, 25 /)  ! end
    integer(ik), parameter      :: RFfrom          = from( rki )     ! selec.
    integer(ik), parameter      :: RFto            = to  ( rki )     ! selec.
!
    real   (rk), parameter      :: p (1_ik:25_ik)  =            (/ &
                                      2.96316885199227378E-01_rk,  & ! 01sp/dp
                                      1.81581125134637070E-01_rk,  & ! 02sp/dp
                                      6.81866451424939493E-02_rk,  & ! 03sp/dp
                                      1.56907543161966709E-02_rk,  & ! 94sp/dp
                                      2.21290116681517573E-03_rk,  & ! 05sp/dp
                                      1.91395813098742864E-04_rk,  & ! 06sp/dp
                                      9.71013284010551623E-06_rk,  & ! 07sp/dp
                                      1.66642447174307753E-07_rk,  & ! 08sp/dp
                  2.1226887418241545314975570224238841543E-01_rk,  & ! 09eq/qp
                  1.6766968820663231170102487414107148110E-01_rk,  & ! 10eq/qp
                  1.0461429607758480243524362040994242137E-01_rk,  & ! 11eq/qp
                  5.1557963860512142911764627378588661742E-02_rk,  & ! 12eq/qp
                  2.0070986488528139460346647533434778000E-02_rk,  & ! 13eq/qp
                  6.1717726506718148117513762897928828534E-03_rk,  & ! 14eq/qp
                  1.4990611906920858646769185063310410160E-03_rk,  & ! 15eq/qp
                  2.8760540416705806615617926157307107830E-04_rk,  & ! 16eq/qp
                  4.3585593590380741491013549969419946961E-05_rk,  & ! 17eq/qp
                  5.2174364856655433775383935118049845471E-06_rk,  & ! 18eq/qp
                  4.9333351722974670085736982894474122277E-07_rk,  & ! 19eq/qp
                  3.6846914376723888190666722894010079935E-08_rk,  & ! 20eq/qp
                  2.1729515092764086499231043367920037215E-09_rk,  & ! 21eq/qp
                  9.9870022842895735663712411206346261651E-11_rk,  & ! 22eq/qp
                  3.1775163189596489863458236395414830880E-12_rk,  & ! 23eq/qp
                  4.5657943993597540327708145643160878201E-14_rk,  & ! 24eq/qp
                  1.1940964427370412648558173558044106203E-16_rk  /) ! 25eq/qp
!
    real   (rk), parameter      :: q (1_ik:25_ik)  =            (/ &
                                      6.12158644495538758E-02_rk,  & ! 01sp/dp
                                      5.50942780056002085E-01_rk,  & ! 02sp/dp
                                      1.53039662058770397E+00_rk,  & ! 03sp/dp
                                      2.99957952311300634E+00_rk,  & ! 04sp/dp
                                      4.95867777128246701E+00_rk,  & ! 05sp/dp
                                      7.41471251099335407E+00_rk,  & ! 06sp/dp
                                      1.04765104356545238E+01_rk,  & ! 07sp/dp
                                      1.48455557345597957E+01_rk,  & ! 08sp/dp
                  2.9482230394292049252878077330764031337E-02_rk,  & ! 09ep/qp
                  2.6534007354862844327590269604581049764E-01_rk,  & ! 10ep/qp
                  7.3705575985730123132195272141160572532E-01_rk,  & ! 11ep/qp
                  1.4446292893203104133929687855854497896E+00_rk,  & ! 12ep/qp
                  2.3880606619376559912235584857800710490E+00_rk,  & ! 13ep/qp
                  3.5673498777093386979273977202889759348E+00_rk,  & ! 14ep/qp
                  4.9824969366355296879760903991854492762E+00_rk,  & ! 15ep/qp
                  6.6335018387405633238409855625402006223E+00_rk,  & ! 16ep/qp
                  8.5203645862651289478197632097553870199E+00_rk,  & ! 17ep/qp
                  1.0643085317662274170216548777166393329E+01_rk,  & ! 18ep/qp
                  1.3001669850030489723387515813223808078E+01_rk,  & ! 19ep/qp
                  1.5596282517377690399267249728222735970E+01_rk,  & ! 20ep/qp
                  1.8429903207271748464995406180854691072E+01_rk,  & ! 21ep/qp
                  2.1533907893494593530979123915138686107E+01_rk,  & ! 22ep/qp
                  2.5076752889217226137869837117288885077E+01_rk,  & ! 23ep/qp
                  2.9515380437412601845256918753602002410E+01_rk,  & ! 24ep/qp
                  3.5792848810704122499184545805923520658E+01_rk  /) ! 25ep/qp
!
!-----------------------------------------------------------------------------
!
!   Asymptotic Series Expansion (ASE),
!   Zhang/Jin (1996), p. 621, f. (16.1.5), reformulated ASE by Hoering (2024)
!
!   1. upper limit for Asymptotic Series Expansion (ASE),
!      from Zaghloul (2024), page 1296, table 2
!      ASE := [ sp: +42.10E+00 / dp: +10042.10E+00 / 
!               ep or qp: +6.60000421E+07 = +66000042.10E+00 ]
    real   (rk), parameter      :: ASEp(1_ik:3_ik) =              (/ &
                                                         42.1E+0_rk, & ! sp
                                                      10042.1E+0_rk, & ! dp
                                                   66000042.1E+0_rk /) ! ep/qp
    real   (rk), parameter      :: ASE             = ASEp( rki )
!
!   2. Power Series Expansion, referenced Zhang/Jin (1996), p.621, f.(16.1.5)
!      reformulated ASE by Hoering (2024), parameters: +(k + 0.5)
    real   (rk)                 :: sASE            ! incremental sum
    integer(ik), parameter      :: ASEmax          = 50_ik  ! max. parameters
    real   (rk), parameter      :: px(0_ik:ASEmax) =  (/  0.5E+0_rk,  &  ! 00
          1.5E+0_rk,      2.5E+0_rk,      3.5E+0_rk,      4.5E+0_rk,  &  ! 04
          5.5E+0_rk,      6.5E+0_rk,      7.5E+0_rk,      8.5E+0_rk,  &  ! 08
          9.5E+0_rk,     10.5E+0_rk,     11.5E+0_rk,     12.5E+0_rk,  &  ! 12
         13.5E+0_rk,     14.5E+0_rk,     15.5E+0_rk,     16.5E+0_rk,  &  ! 16
         17.5E+0_rk,     18.5E+0_rk,     19.5E+0_rk,     20.5E+0_rk,  &  ! 20
         21.5E+0_rk,     22.5E+0_rk,     23.5E+0_rk,     24.5E+0_rk,  &  ! 24
         25.5E+0_rk,     26.5E+0_rk,     27.5E+0_rk,     28.5E+0_rk,  &  ! 28
         29.5E+0_rk,     30.5E+0_rk,     31.5E+0_rk,     32.5E+0_rk,  &  ! 32
         33.5E+0_rk,     34.5E+0_rk,     35.5E+0_rk,     36.5E+0_rk,  &  ! 36
         37.5E+0_rk,     38.5E+0_rk,     39.5E+0_rk,     40.5E+0_rk,  &  ! 40
         41.5E+0_rk,     42.5E+0_rk,     43.5E+0_rk,     44.5E+0_rk,  &  ! 44
         45.5E+0_rk,     46.5E+0_rk,     47.5E+0_rk,     48.5E+0_rk,  &  ! 48
         49.5E+0_rk,     50.5E+0_rk                                  /)  ! 50
!
!-----------------------------------------------------------------------------
!
!
!   start of cases and calculations
!
!   -∞ ≤ x ≤ ( limiterfcxL = -SQRT( LOG( HUGE(one)/two ) ) =
!                          = [ sp:   -9.3824...E+00 / dp:  -26.6287...E+00 /
!                              ep: -106.5637...E+00 / qp: -106.5637...E+00 ] )
         if( x <= limiterfcxL  ) then            ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
!   limiterfcxL < x ≤ (limiterfcxL1 = -SQRT(-LOG( sqrtpi*EPSILON(one)/two)) =
!                      [ sp: -4.0078...E+00 / dp: -6.0136...E+00 /
!                        ep: -6.6173...E+00 / qp: -8.8177...E+00])
!   limiterfcxL1 defines erfc(x) ≈ 2 => erfcx(x)=2*exp(x²)
    else if( x <= limiterfcxL1 ) then            ! cut off "neg." at erfc(x)=2
             f = two * EXP( x*x )                ! f = 2.00 * exp(x²)
!
    else                                         ! continue with |x|
             z  = ABS( x )                       ! z=|x| : 0, ..., +∞
!
!
!   (zero=0.0) ≤ z=|x| ≤ ( sqrteps = SQRT( EPSILON(one) ) =
!                                  = [ sp: 3.452..E-04 / dp: 1.4901..E-08 /
!                                      ep: 3.292..E-10 / qp: 1.3877..E-17 ] )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1 - (2/√pi) * x
!
!-----------------------------------------------------------------------------
!
!   sqrteps < z=|x| ≤ ( PSE = [ sp: 0.250E+00 / dp: 3.000E-01 /
!                               ep: 3.000E-02 / qp: 3.000E-02 ] )
    else if( z <= PSE ) then
!            Power Series Expansion (PSE), referenced Zhang/Jin (1996),
!            page 621, formula (16.1.4), reformulated by Hoering (2024)
!
!                                  2     ∞  2^n * z^(2n+1)
!            erfcx(z) = exp(z²) - --- *  ∑  --------------
!                                 √pi   n=0   (2n+1)!!
!            or
!                                 2*z        ∞   n     z²
!            erfcx(z) = exp(z²) - --- * [1 + ∑   ∏  ---------]
!                                 √pi       n=1 k=1 (k + 1/2)
!            or
!                                 2*z            z²    z²     z²     z²
!            erfcx(z) = exp(z²) - --- * [1 +((( --- * ---) * ---) * ---) * ..]
!                                 √pi           1.5   2.5    3.5    4.5
!
             zz   = z * z                        ! zz = z² = |x|²
             sPSE = one                          ! product increment = 1.0*
             f    = one                          ! f = 1.0 + (sum=0)
             do i = 1_ik, PSEmax, 1_ik           ! loop
                sPSE = sPSE * qx(i) * zz         ! s* {qx=[1.0/(k+0.5)]} * z²
                f    = f + sPSE                  ! sum for (n≥1)
                if( sPSE <= eps ) exit           ! precision achieved
             end do                              ! end of do...
             f = EXP(zz) - (twodivsqrtpi*x * f)  ! finalize erfcx(x)
!
!-----------------------------------------------------------------------------
!
!   PSE < z=|x| < ( RF = [ sp: eMax=9.195.. / dp: eMax=26.542.. /
!                          ep: 70.000       / qp: 70.000        ] )
    else if( z <= RF ) then
!            Rational Function, referenced Ooura (1996), reformulated from
!            "erfcOoura8b.f95" and "erfcOoura16.f95", modified Hoering (2024)
             zz = x * x                          ! zz = z² = |x|²
             f  = zero                           ! sum = 0.0
             do i = RFfrom, RFto, 1_ik           ! loop from ... to
                f = f + p(i) / (zz + q(i))       ! sum up rational function
             end do                              ! end do...
             if( x < ph ) then                   ! check ph
                 f = (f * x) + (two*EXP(zz)) / ( EXP( pv*x ) + one )
             else
                 f =  f * x
             end if
!
!-----------------------------------------------------------------------------
!   continuing from here: x is always positive, no precautions for x < zero
!
!   RF < z=|x| < ( ASE = [ sp: +42.10E+00 / dp: +10042.10E+00 / 
!                          ep or qp: +6.60000421E+07 = +66000042.10E+00 ] )
    else if( z < ASE ) then
!            Asymptotic Series Expansion (ASE), referenced Zhang/Jin (1996),
!            page 621, formula (16.1.5), reformulated by Hoering (2024)
!            Remark: No check for x < 0.0, due to all x here are > 0.0
!
!                          1             ∞  (-1)^n * (2n-1)!!
!            erfcx(z) = -------- *       ∑  -----------------
!                         √pi           n=0      (2z²)^n
!
!            or as formula (16.1.5)b
!                          1             ∞   n  -(2k-1)
!            erfcx(z) = -------- * [ 1 + ∑   ∏  -------- ]
!                       √pi * z         n=1 k=1    2z²
!
!            or reformulated
!                          1             ∞   n  -(k-1/2)
!            erfcx(z) = -------- * [ 1 + ∑   ∏  -------- ]
!                       √pi * z         n=1 k=1     z²
!            or
!                          1               -0.5   -1.5    -2.5    -3.5
!            erfcx(z) = -------- * [ 1 +(((---- * ----) * ----) * ----) * ...]
!                       √pi * z             z²     z²      z²      z²
!
             zz   = one / (z * z)                ! zz = 1/z² = 1/|x|²
             sASE = one                          ! product increment = 1.0*
             f    = one                          ! f = 1.0 + (sum=0)
             do i = 0_ik, ASEmax, 1_ik           ! loop
                sASE = -sASE * px(i) * zz        ! -/+ s*[px=(k-1/2)]*(1.0/z²)
                f    = f + sASE                  ! sum for (n≥1)
                if( ABS(sASE) <= eps ) exit      ! precision achieved
             end do                              ! end of do...
             f = f / ( sqrtpi * z )              ! finalize erfcx(x), x > 0.0
!
!-----------------------------------------------------------------------------
!
!   ASE ≤ z=|x| ≤ ( limiterfcxBig = SQRT( one/EPSILON(x) ) =
!                                 = [ sp: 2.8963..E+03 / dp: 6.7108..E+07 /
!                                     ep: 3.0370..E+09 / qp: 7.2057..E+16 ] )
    else if( z <= limiterfcxBig ) then
             f = onedivsqrtpi / (z + half/z)     ! 1/(√pi*(z + 0.5/z))
!
!-----------------------------------------------------------------------------
!
!   limiterfcxBig < z=|x| ≤ ( limiterfcxH = one / ( sqrtpi * TINY(one) ) =
!                             = [ sp: 4.7995...E+0037 / dp: 2.5355...E+0307 / 
!                                 ep: 1.6780...E+4931 / qp: 1.6780...E+4931 ])
    else if( z <= limiterfcxH ) then
             f = onedivsqrtpi / z                ! (1/√pi)/z = 1/(√pi*z)
!
!-----------------------------------------------------------------------------
!
!   limiterfcxH < z ≤ ( +∞ = HUGE(x) = 
!                            [ sp: +3.40..E+0038 / dp: +1.79..E+0308 /
!                              ep: +1.18..E+4932 / qp: +1.18..E+4932 ]  )
    else
             f = zero                            ! f = 0.0
!
!-----------------------------------------------------------------------------
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcxTH
!
!-----------------------------------------------------------------------------
!