!
!
!   Remark:
!   1. ./.
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   Scaled Complementary Error Function (erfcx): Low Precision
!
!   real(rk) function erfcxRen        ( x )       result( f )  Ren-MacKenzie        (2007),  Parameter driven Approximation
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   erfcxRen
!     Scaled Complementary Error Function erfcx(x) with approximation
!     referenced Ren/MacKenzie (2007)
!   Reference:
!     C. Ren and A.R. MacKenzie, "Closed-form approximations to the error and
!     complementary error functions and their applications in atmospheric
!     science", Atmospheric Science Letters, 2007, Volume 8, Issue 3,
!     Pages 70-73, https://doi.org/10.1002/asl.154
!     here: page 71, formula (9)
!   Remarks:
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )   := +∞  = HUGE(x)
!     Parameter "sqrteps" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ sqrteps )       := +1.00 - 2.00/SQRT(pi)*x
!     Parameter "limiterfcxBig" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ limiterfcxBig ) := 1/(√pi*(x + 0.5/x))
!     Parameter "limiterfcxH" is defined in "Constants" for erfcx,
!      where erfcx( z ≤ limiterfcxH )   := (1/√pi)/x = 1/(√pi*x)
!      where erfcx( z > limiterfcxH )   := 0.0
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    pure function erfcxRen( x )   result( f )
!
    use kinds, only : rk
    use const, only : zero, one, two,                         &
                      pi, onedivsqrtpi,twodivsqrtpi, sqrteps, &
                      limiterfcxL, limiterfcxBig, limiterfcxH
!
    implicit none
!
!   interface
     real  (rk), intent(in)     :: x             ! x from erfcx(x)
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
    real   (rk)                 :: zz            ! zz = z² = |x|²
!
!   Ren/MacKenzie (2007), adjustment parameter, page 71
    real   (rk), parameter      :: a             = 2.9110E+00_rk
!
!
!   -∞ ≤ x ≤ limiterfcxL
         if( x <= limiterfcxL ) then             ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   (zero = 0.00E+00) ≤ z=|x| ≤ ( sqrteps = SQRT(EPSILON(x)) )
         if( z <= sqrteps ) then
             f = one - twodivsqrtpi * x          ! 1.00 - 2.00/√pi * x
!
!   sqrteps < z=|x| ≤ limiterfcxBig
    else if( z <= limiterfcxBig ) then
!            Ren/MacKenzie (2007), page 71, formula 9, and the adjustment
!            parameter a = 2.9119; the formula overflows for z > 5.00E+1045
             zz = z * z                          ! z² = |x|²
             f  = a / ( (a - one) * SQRT(pi * zz) + SQRT(pi * zz + (a*a)) )
             if( x < zero ) f = two*EXP(zz) - f ! erfcx(-x)=2*exp(z²)-erfcx(z)
!
!   limiterfcxBig < z ≤ limiterfcxH
    else if( z <= limiterfcxH ) then
             f = onedivsqrtpi / z                ! f = (1/√pi)*(1/z)
!
!   limiterfcxH < z=|x| ≤ +∞
    else                                         ! cut off "High"
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
    return
    end function erfcxRen
!
!-----------------------------------------------------------------------------
!