!
!
!   Remark:
!   1. ./.
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   Scaled Complementary Error Function (erfcx): Power Series / Series Expansions
!
!   real(rk) function erfcxSE         ( x, it )   result( f )  SE 1: Shepherd/Lafromboise (1981),  Taylor Series
!                                                              SE 2: Zhang/Jin            (1996),  Power Series Expansion
!   real(rk) function erfcxSEasymp    ( x, it )   result( f )  SE 1: Acton                (1990),  asymptotic Ser. Expansion
!                                                              SE 2: Zhang/Jin            (1996),  asymptotic Ser. Expansion
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   erfcxSE
!     Scaled Complementary Error Function f = erfcx(x) with Taylor Series and
!     other Series Expansions, referenced by various sources, Hoering (2024)
!   Reference:
!     selec == 1
!      M.M. Shepherd and J.G. Laframboise, "Chebyshev Approximation of
!       (1 + 2x)exp(x²)erfc x in 0 ≤ x < ∞", Mathematics of Computation,
!       1981, Volume 36, Issue 153, Pages 249-253,
!       https://doi.org/10.1090/S0025-5718-1981-0595058-X
!       here: page 251, formula (7) with x ≤ 2.83
!      M.R. Zaghloul, "Efficient multiple-precision computation of the
!       scaled complementary error function and the Dawson integral",
!       Numerical Algorithms, 2024, Volume 95, Issue 3,
!       Pages 1291–1308, https://doi.org/10.1007/s11075-023-01608-8
!       here: page 1293, formula (5) with z in [0, ..., 0.037]
!
!     selec == 2
!      Shanjie Zhang and Jianming Jin, "Computation of Special
!       Functions", 1996, XXVI, 717 pages, John Wiley & Sons Inc.,
!       New York, ISBN: 0-471-11963-6
!       here: page 621, formula (16.1.4)
!   Remark:
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL ) := +∞ = HUGE(x)
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfcxSE( x, it )   result( f )
!
    use kinds, only : ik, rki, rk
    use const, only : zero, half, one, two,                &
                      sqrtpi, onedivsqrtpi, twodivsqrtpi,  &
                      eps, eMax, limiterfcxL
!
    implicit none
!
!   interface
     real   (rk), intent(in )   :: x             ! x from erfcx(x)
     integer(ik), intent(out)   :: it            ! iterations
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
    real   (rk)                 :: zz            ! zz = z² = |x|²
    integer(ik), save           :: selec         ! series   selection
    integer(ik), save           :: selecp = 0_ik ! previous selection
!
!   local variable
    real   (rk)                 :: p1, p2        ! variable numerator
    real   (rk)                 :: q1, q2        ! variable denominator
    real   (rk)                 :: s             ! incremental sum
!
!   mathematical & machine-dependent constants/parameters
    integer(ik)                 :: itmax         ! max. number of iterations
    integer(ik), parameter      ::                     & ! vector for itmax
       it1(1_ik:3_ik) = (/  17_ik,  98_ik, 200_ik /),  & ! sp, dp, ep/qp
       it2(1_ik:3_ik) = (/ 200_ik, 200_ik, 200_ik /)     ! sp, dp, ep/qp
!
!
    if( selecp == 0_ik .and. (x <= -HUGE(x) .and. x >= -HUGE(x)) ) then
        write( *,                                                 '(  &
         & t2,"choose selec <1 or 2>:",                            /, &
         & t2,"1: Taylor Expan.: Shepherd/La. (1981), itmax= ",i3, /, &
         & t2,"2: Power Series : Zjang/Jin    (1996), itmax= ",i3  )' ) &
         &                                             it1(rki), it2(rki)
        read(*,*) selec
                  selecp = 1_ik                  ! selec is set
    end if
!
    it = 0_ik                                    ! no iterations
!
!   -∞ ≤ x < limiterfcxL
         if( x < limiterfcxL ) then              ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   z=|x| == zero  =>  erfcx( [z=|x|] ≤ zero ) = one
         if( z <= zero ) then                    ! x = 0.0
             f = one                             ! f = 1.0
!
!   (zero = 0.00) < z=|x| < eMax
    else if( z < eMax ) then
!
             if( selec == 1_ik ) then
!            Taylor expansion (rearranged) referenced
!            Shepherd/Laframboise (1981), page 251, formula (7) or
!            Zaghoul (2024), page 1293, formula (5)
!            for z=|x| element [0,0.037]
!            remark: OEIS A001147 for (2n+1)!!, n=0,1,2,3,...
!
!                                 2*z           ∞   (2z²)^n
!            erfcx(z) = exp(z²) - ---  *        ∑  ----------
!                                 √pi          n=0  (2n+1)!!
!
!                                 2*z           ∞   (2z²)^n
!            erfcx(z) = exp(z²) - ---  * [ 1 +  ∑  ---------- ]
!                                 √pi          n=1  (2n+1)!!
!
!                 ∞  (2z²)^n        2z^2     4z^4          8z^6
!            1 +  ∑  -------- = 1 + ----- + -------- + ----------- + ...
!                n=1 (2n+1)!!       1*3=3   1*3*5=15   1*3*5*7=105
!
             itmax = it1( rki )                  ! max. iterations
             zz    = z*z                         ! zz = z² = |x|²
             p1    = two * zz                    ! 2z²
             p2    = one                         ! (2x²)^n, with n=0
             q1    = one                         ! (n=0)!!
             q2    = one                         ! (2*(n=0)+1)
             f     = one                         ! sum for (n=0)
             do while ( it < itmax ); it=it+1_ik ! iterations
                p2 = p2 * p1                     ! (2x²)**n
                q1 = q1 + two                    ! (2n+1) = 3,5,7,...
                q2 = q2 * q1                     ! (2n+1)!!
                s  = p2 / q2                     ! increment s
                f  = f + s                       ! sum
                if( ABS(s) <= ABS(f)*eps ) exit  ! precision achieved
             end do
             f = EXP(zz) - twodivsqrtpi*z*f      ! erfcx(x), x>0
             if( x < zero ) f = two * EXP(zz) -f ! 2*exp(x²)-erfcx(z)
             end if                              ! end if selec == 1
!
!
             if( selec == 2_ik ) then
!            Power Series Expansion (reformulated), referenced
!            Zhang/Jin (1996), page 621, formula (16.1.4)
!
!                                  2     ∞  2^n * z^(2n+1)
!            erfcx(z) = exp(z²) - --- *  ∑  --------------
!                                 √pi   n=0   (2n+1)!!
!            or
!                                 2*z        ∞   n     z²
!            erfcx(z) = exp(z²) - --- * [1 + ∑   ∏  ---------]
!                                 √pi       n=1 k=1 (k + 1/2)
!            or
!                                 2*z           z^2   z^2  z^2  z^2
!            erfcx(z) = exp[z²] - --- * [1 + (((--- * ---)*---)*---) * ...]
!                                 √pi           1.5   2.5  3.5  4.5
!
             itmax = it2( rki )                  ! max. iterations
             zz    = z*z                         ! zz = z² = |x|²
             q1    = half                        ! [k=0 + (q1=0.5)]
             s     = one                         ! product increment
             f     = one                         ! f = 1 + (sum=0)
             do while ( it < itmax ); it=it+1_ik ! iterations
                q1 = q1 + one                    ! [(k>0) + 0.5]
                s  = s  * zz / q1                ! s* z²/(k+0.5)
                f  = f  + s                      ! sum for (n≥0)
                if( ABS(s) <= ABS(f)*eps ) exit
             end do
             f = EXP(zz) - (twodivsqrtpi*x * f)  ! finalize erfcx(x)
             end if                              ! end if selec == 2
!
!
!   eMax ≤ z=|x| ≤ +∞
    else                                         ! cut off "High"
             f = zero                            ! f = 0.0
!
    end if                                       ! end if calculation
    end if                                       ! end if cases
!
!   check and reset preselec for next new run
    if( selecp == 1_ik .and. (x <= +HUGE(x) .and. x >= +HUGE(x)) ) then
        selecp =  0_ik                           ! reset selecp
    end if
!
    return
    end function erfcxSE
!
!-----------------------------------------------------------------------------
!
!   erfcxSEasymp
!     Scaled Complementary Error Function f = erfcx(x) with asymptotic Series
!     Expansion, referenced by various sources, Hoering (2024)
!   Reference:
!     selec == 1
!      Forman S. Acton, "Numerical Methods That Work", 1990, 2nd printing,
!       updated and revised, XX, 549 pages, Washington, DC: Mathematical
!       Association of America, PPN: 114363080, E-ISBN: 978-1-4704-5727-3,
!       here: page 16, formula (1.13).
!      M.R. Zaghloul, "Efficient multiple-precision computation of the scaled
!       complementary error function and the Dawson integral", Numerical
!       Algorithms, 2024, Volume 95, Issue 3, Pages 1291–1308,
!       https://doi.org/10.1007/s11075-023-01608-8,
!       here: page 1296, formula (10)
!      OEIS: numerator with OEIS A000079, denominator with OEIS A001147
!
!     selec == 2
!      Shanjie Zhang and Jianming Jin, "Computation of Special Functions",
!       1996, XXVI, 717 pages, John Wiley & Sons Inc., New York, 
!       ISBN: 0-471-11963-6,
!       here: page 621, formula (16.1.5)
!   Remark: 
!     Parameter "eMax" is defined in "Constants" for all functions,
!      where "EXP( eMax*eMax )" does not generate an overflow error
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL ) := +∞ = HUGE(x)
!     Cut off at [ABS(x) < 0.25] with f = 0.0 to avoid underflow.
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfcxSEasymp( x, it )   result( f )
!
    use kinds, only : ik, rki, rk
    use const, only : zero, half, one, two,    &
                      sqrtpi, eps, eMax,       &
                      limiterfcxL, limiterfcxH
!
    implicit none
!
!   interface
     real   (rk), intent(in )   :: x             ! x from erfcx(x)
     integer(ik), intent(out)   :: it            ! iterations
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
    real   (rk)                 :: zz            ! zz = z² = |x|²
    integer(ik), save           :: selec         ! selection
    integer(ik), save           :: selecp = 0_ik ! previous selection
!
!   local vriable
    real   (rk)                 :: p1, p2        ! variable numerator
    real   (rk)                 :: q1, q2        ! variable denominator
    real   (rk)                 :: s             ! incremental sum
!
!   mathematical & machine-dependent constants/parameters
    integer(ik)                 :: itmax         ! max. number of iterations
    integer(ik), parameter      ::                     & ! vector for itmax
       it1(1_ik:3_ik) = (/  10_ik,  68_ik, 146_ik /),  & ! sp, dp, ep/qp
       it2(1_ik:3_ik) = (/  10_ik,  68_ik, 146_ik /)     ! sp, dp, ep/qp
!
!
    if( selecp == 0 .and. (x <= -HUGE(x) .and. x >= -HUGE(x)) ) then
        write( *,                                            '(  &
         & t2,"choose selec <1 or 2>:",                       /, &
         & t2,"1: Asymp. SE: Acton      (1990), itmax = ",i3, /, &
         & t2,"2: Asymp. SE: Zhang/Jin  (1996), itmax = ",i3  )' ) &
         &                                        it1(rki), it2(rki)
        read(*,*) selec
                  selecp = 1_ik                  ! selec is set
    end if
!
    it = 0_ik                                    ! no iterations
!
!   -∞ ≤ x < limiterfcxL
         if( x < limiterfcxL ) then              ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   z=|x| == zero  =>  erfcx( [z=|x|] ≤ zero ) = one
!   [here local handling: z=|x| ≤ EPSILON(z) to avoid underflow]
         if( z <= EPSILON(x) ) then              ! x ≈ 0.0
             f = one                             ! f = 1.0
!
!   (EPSILON(x)) < z=|x| < eMax
    else if( z < eMax ) then
!
             if( selec == 1_ik ) then
!            asymptotic Series Expansion, referenced
!            Acton (1990), page 16, formula (1.13)  or
!            Zaghloul (2024), page 1296, formula (10)
!            remark: for (2n-1)!! see OEIS A001147
!
!                        2    1          ∞  (-1)^n * (2n-1)!!
!            erfcx(z) = --- * -- *       ∑  -----------------
!                       √pi   2z        n=0     (2z²)^n
!            or
!                         1              ∞  (-1)^n * (2n-1)!!
!            erfcx(z) = -----    * [ 1 + ∑  ----------------- ]
!                       √pi*z           n=1     (2z²)^n
!
!            with the 1+sum in detail and double factorial(!!):
!
!              ∞          1        1*3      1*3*5    1*3*5*7
!            1+∑ = 1 - ------- + ------- - ------- + ------- - ...
!             n=1      (2z²)^1   (2z²)^2   (2z²)^3   (2z²)^4
!            
!              ∞          1         3        15        105
!            1+∑ = 1 - ------- + ------- - ------- + ------- - ...
!             n=1       2z^2      4z^4      8z^6      16z^8
!
             itmax = it1( rki )                  ! max. iterations
             zz    =  z*z                        ! zz = z² = |x|²
             p1    = -one                        ! (2*(n=0) - 1)
             p2    =  one                        ! -1!! = 1
             q1    =  two * zz                   ! 2z² = 2|x|²
             q2    =  one                        ! (2z²)^(n=0)
             f     =  one                        ! sum for (n=0)
             do while ( it < itmax ); it=it+1_ik ! iterations (n≥1)
                p1 =  p1 + two                   ! (2n-1)=1,3,5,7,...
                p2 = -p2 * p1                    ! -/+ (2n-1)!!
                q2 =  q2 * q1                    ! (2z²)^(n=1,2,3,...)
                s  =  p2 / q2                    ! increment s[n≥1]
                f  =  f  + s                     ! sum for (n≥1)
                if( ABS(s) <= ABS(f)*eps ) exit  ! precision achieved
             end do
             f = f / (sqrtpi * z)                ! erfcx(x), x>0
             if( x < zero ) f = two * EXP(zz) -f ! 2*exp(x²)-erfcx(z)
             end if                              ! end if selec == 1
!
!
             if( selec == 2_ik ) then
!            asymptotic Series Expansion, referenced e.g.
!            Zhang/Jin (1996), page 621, formula (16.1.5)
!
!                           1           ∞  (-1)^n * (2n-1)!!
!            erfcx(z) = ------- *       ∑  -----------------
!                         √pi          n=0      (2z²)^n
!
!            or as formula (16.1.5)b
!                           1           ∞   n  -(2k-1)
!            erfcx(z) = ------- * [ 1 + ∑   ∏  -------- ]
!                       √pi * z        n=1 k=1    2z²
!
!            or reformulated
!                           1           ∞   n  -(k-1/2)
!            erfcx(z) = ------- * [ 1 + ∑   ∏  -------- ]
!                       √pi * z        n=1 k=1     z²
!
!            or
!                           1             -0.5   -1.5    -2.5    -3.5
!            erfcx(z) = ------- * [ 1 +(((---- * ----) * ----) * ----) * ...]
!                       √pi * z            z²     z²      z²      z²
!
             itmax =  it2( rki )                 ! max. iterations
             zz    =  z*z                        ! zz = z² = |x|²
             p1    = -half                       ! -[(k=0)-1/2]
             s     =  one                        ! increment = 1*
             f     =  one                        ! sum for (n=0)
             do while( it < itmax ); it=it+1_ik  ! iterations (n≥1)
                 p1 =  p1 + one                  ! [k-1/2]=0.5,1.5,...
                 s  = -s  * p1 / zz              ! -/+ s * (k-1/2)/z²
                 f  =  f  + s                    ! sum for (n≥1)
                 if( ABS(s) <= ABS(f)*eps ) exit ! precision achieved
             end do
             f = f / (sqrtpi * z)                ! erfcx(x), x>0
             if( x < zero ) f = two * EXP(zz) -f ! 2*exp(x²)-erfcx(z)
             end if                              ! end if selec == 2
!
!
!   eMax ≤ z=|x| ≤ +∞
    else                                         ! cut off "High"
             f = zero                            ! f = 0.0
!
    end  if                                      ! end if calculation
    end  if                                      ! end if cases
!
!   check and reset preselec for next new run
    if( selecp == 1_ik .and. (x <= +HUGE(x) .and. x >= +HUGE(x)) ) then
        selecp =  0_ik                           ! reset selecp
    end if
!
    return
    end function erfcxSEasymp
!
!-----------------------------------------------------------------------------
!