!
!   Remark:
!   1. ./.
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   Scaled Complementary Error Function (erfcx): Continued Fraction (CF)
!
!   real(rk) function erfcxCF         ( x, it )   result( f )  Hoering              (2023)
!                                                              CF 1: Stegun/Zucker (1970),  p. 214, 3rd formula
!                                                                    Jones/Thron   (1984),  p. 208, formula (6.1.38)
!                                                                                           p. 282, formula (7.3.106)
!                                                              CF 2: Stegun/Zucker (1970),  p. 214, 4th formula
!                                                              CF 3: Fettis        (1974),  p. 585, 4th formula  and
!                                                                    Fettis        (1975),  p. 673
!                                                              CF 4: Shepherd/Laf. (1981),  p. 251, formula (9)
!
!----------------------------------------------------------------------------------------------------------------------------------
!
!   erfcxCF
!     Scaled Complementary Error Function f = erfcx(x) with Continued Fraction
!     referenced by various sources, Hoering (2023)
!   Reference:
!     selec == 1
!      Irene A. Stegun and Ruth Zucker, "Automatic Computing Methods for
!       Special Functions", Journal of Research of the National Bureau of
!       Standards - B. Mathematical Sciences, 1970, Volume 74B, Issue 3,
!       Pages 211-224, http://dx.doi.org/10.6028/jres.074B.019,
!       here: page 214, 3rd (redefined by Hoering) formula
!      Willian B. Jones and W.J. Thron, "Continued Fractions : Analytic Theory
!       and Applications", Cambridge University Press, Cambridge, 1984,
!       XXVIII, 428 Pages, PPN: 883353466, E-ISBN: 978-0-511-75955-0, P-ISBN:
!       978-0-521-30231-9, https://doi.org/10.1017/CBO9780511759550
!       here: page 208, formula (6.1.38) and page 282, formula (7.3.106)
!
!     selec == 2
!      Irene A. Stegun and Ruth Zucker, "Automatic Computing Methods for
!       Special Functions", Journal of Research of the National Bureau of
!       Standards - B. Mathematical Sciences, 1970, Volume 74B, Issue 3,
!       Pages 211-224, http://dx.doi.org/10.6028/jres.074B.019,
!       here: page 214, 4th formula
!
!     selec == 3
!      Henry E. Fettis, "A Stable Algorithm for Computing the Inverse Error
!       Function in the "Tail-End" Region", Mathematics of Computation, 1974,
!       Volume 28, Issue 126, Page 585-587,
!       https://doi.org/10.1090/S0025-5718-1974-0341812-5
!       here:  page 585, 4th formula
!      Henry E. Fettis, "Corrigendum: “A stable algorithm for computing the
!       inverse error function in the "tail-end" Region (Math. Comp. 28 
!       (1974), 585-587)", Mathematics of Computation, 1975, Volume 29,
!       Issue 130, Page 673, https://doi.org/10.1090/S0025-5718-1975-0359272-8
!       here: corrigendum of the 4th formula (page 585 as of 1974)
!
!     selec == 4
!      M.M. Shepherd and J.G. Laframboise, "Chebyshev Approximation of
!       (1 + 2x)exp(x²)erfc x in 0 ≤ x < ∞", Mathematics of Computation, 1981,
!       Volume 36, Issue 153, Pages 249-253, 
!       https://doi.org/10.1090/S0025-5718-1981-0595058-X
!       here: page 251, formula (9), x > 2.83 (see comment f.(12))
!
!     Modified Lentz method for Continued Fraction:
!      I.J. Thompson and A.R. Barnett, "Coulomb and Bessel functions of 
!       complex arguments and order", Journal of Computational Physics,
!       1986, Volume 64, Issue 2, Page 490–509,
!       https://doi.org/10.1016/0021-9991(86)90046-X,
!       here: page 507 and 508, (III) Lentz's method
!       Local Tiny (page 508, "small"): "loctiny" set to 1.00E-035, where
!        case 1: if( b(start) == zero ) f     = loctiny
!        case 2: if( C        == zero ) C     = loctiny
!        case 3: if( b+a*D    == zero ) b+a*D = loctiny
!   Remark:
!     Parameter "limiterfcxL" is defined in "Constants" for erfcx,
!      where erfcx( x ≤ limiterfcxL )        := +∞ = HUGE(x)
!     Parameter "limiterfcxH" is defined in "Constants" for erfcx,
!      where erfcx( z=|x| < limiterfcxH )    := (1/√pi)/x = 1/(√pi*x)
!     Parameter "sqrtlimiterfcxH" is defined locally to avoid overflow,
!      where erfcx( z=|x| ≥ sqrtlimiterfcxH) := (1 / √pi) / z
!   Implementation, adjustments and/or extensions by Thomas Hoering,
!     last modification: 31. January 2025
!
    function erfcxCF( x, it )   result( f )
!
    use kinds, only : ik, rk
    use const, only : zero, half, one, two, four, eight,   &
                      sqrtpi, onedivsqrtpi, twodivsqrtpi,  &
                      eps, limiterfcxL, limiterfcxH
!
    implicit none
!
!   interface
     real   (rk), intent(in )   :: x             ! x from erfcx(x)
     integer(ik), intent(out)   :: it            ! iterations
!   end interface
    real   (rk)                 :: f             ! function result
!
!   local variable
    real   (rk)                 :: z             ! z=|x| (absolute x)
    real   (rk)                 :: zz            ! zz = z² = |x|²
    real   (rk)                 :: t             ! 1.0 / (z=|x|)
    real   (rk)                 :: t2            ! t² / 2.0
    real   (rk)                 :: b, a, C, D, h ! Lentz variables
    real   (rk)                 :: k             ! help variable
    integer(ik), save           :: selec         ! series selection
    integer(ik), save           :: selecp = 0_ik ! previous selection
!
!   mathematical & machine-dependent constants/parameters
    real   (rk), parameter      :: sqrtlimiterfcxH = SQRT( limiterfcxH )
    integer(ik), parameter      :: itmax           = 200_ik
!c  real   (rk), parameter      :: loctiny         = 1.00E-0035_rk
!
!
    if( selecp == 0 .and. (x <= -HUGE(x) .and. x >= -HUGE(x)) ) then
        write( *,                                                        '( &
        & t2,"choose selec <1,2,3,4 or 5>:",                             /, &
        & t2,"1: Stegun/Zucker (1970), p.214,3rd formula,  itmax = 200", /, &
        & t2,"2: Stegun/Zucker (1970), p.214,4th formula,  itmax = 200", /, &
        & t2,"3: Fettis        (1974), p.585,4th formula,  itmax = 200", /, &
        & t2,"4: Shepherd/Laf. (1981), p.251,formula(9),   itmax = 200"   )')
        read(*,*) selec
                  selecp = 1_ik                  ! selec is set
    end if
!
    it = 0_ik                                    ! no iterations
!
!   -∞ ≤ x < limiterfcxL
         if( x < limiterfcxL ) then              ! cut off "Low"
             f = +HUGE( x )                      ! f = +∞
!
    else                                         ! continue with |x|
             z = ABS( x )                        ! z = |x|
!
!   z=|x| == zero  =>  erfcx( [z=|x|] ≤ zero ) = one
!c         if( z <= zero ) then                  ! x = 0.0
           if( z <= EPSILON(z/two) ) then        ! avoid overflow in the CF
             f = one                             ! f = 1.0
!
!   (zero = 0.0) < z=|x| < limiterfcxH
    else if( z < limiterfcxH ) then
!
             if( selec == 1_ik ) then
!            Continued Fraction, Stegun/Zucker (1970), page 214,
!            redefinied 3rd formula and modified Lentz method for
!            Continued Fraction
!
!                       2*z                                     a0 =  1
!            erfcx(z) = --- * -------------------------------------------
!                       √pi                                     a1 = 1*2
!                             b0=2z²+1 - --------------------------------
!                                                               a2 = 3*4
!                                        b1=2z²+5 - ---------------------
!                                                               a3 = 5*6
!                                                   b2=2z²+9 - ----------
!                                                                  ...
!            or
!                       2*z       a0      a1      a2          an
!            erfcx(z) = --- * ( ------- ------- ------- ... ------- ... )
!                       √pi     (u+k0)- (u+k1)- (u+k2)-     (u+kn)-
!
!                       with a(0) = 1, a(n) = a1(n) * a2(n),        for n ≥ 1
!                                   a1(1) = 1, a1(n) = a1(n-1) + 2, for n ≥ 2
!                                   a2(1) = 2, a1(n) = a1(n-1) + 2, for n ≥ 2
!                            u    = 2*z²,
!                            k(0) = 1, k(n) = k(n-1) + 4,           for n ≥ 1
!
!            local handling: if z ≥ SQRT(limiterfcxH), f = (1/√pi)/z
!            sqrtlimiterfcxH=[sp:6.927..E+18/dp:5.035..E+153/ep:4.096..E+2465]
             if( z >= sqrtlimiterfcxH ) then     ! early exit
                 f = onedivsqrtpi / z            ! (1/√pi)/z
                 if(x < zero) f=two*EXP(z*z) -f ! erfcx(-x)=2*exp(z²)-erfcx(z)
                 return
             end if
!
             zz = z * z                          ! zz = z*z
             b  = two*zz + one                   ! b0
!c no need   if( b<=zero .and. b>=zero ) then    ! b0 == 0
!c no need       f = loctiny                     ! b0 == 0
!c no need   else
                 f = b                           ! b0 /= 0
!c no need   end if
             k  = -one                           ! prepare a1
             C  = f                              ! C <- f
             D  = zero                           ! D <- 0
             do while( it < itmax ); it=it+1_ik  ! iterations
                k = k + two                      ! prepare next a
                a = k * (k + one)                ! a1, ..., an
                b = b + four                     ! b1, ..., bn
!               calculation of C and D, incl. h = C*D
                C =         b - a/C              ! bn >> - <<
                D = one / ( b - a*D )            ! bn >> - <<
!c no need      C = b - a/C                      ! bn >> - <<
!c no need          if( C <= zero .and. C >= zero ) C = loctiny
!c no need      D = b - a*D                      ! bn >> - <<
!c no need          if( D <= zero .and. D >= zero ) then
!c no need              D = one / loctiny
!c no need          else
!c no need              D = one / D
!c no need          end if
                h = C * D
                f = f * h                        ! new function value
                if( ABS( h - one ) <= eps ) exit ! precision check
             end do
             f = (twodivsqrtpi * z)  /  f        ! finalize f, x > 0
             if( x < zero ) f = two*EXP(zz) - f ! erfcx(-x)=2*exp(z²)-erfcx(z)
             end if                              ! selec == 1
!
!
             if( selec == 2_ik ) then
!            Continued Fraction, Stegun/Zucker (1970), page 214,
!            redefinied 4th formula and modified Lentz method for
!            Continued Fraction
!
!                        1                         a0 = 1.0
!            erfcx(z) = --- * ------------------------------
!                       √pi                        a1 = 0.5
!                             b0=z + -----------------------
!                                                  a2 = 1.0
!                                    b1=z + ----------------
!                                                  a3 = 1.5
!                                           b2=z + ---------
!                                                     ...
!            or
!                        1       a0  a1  a2     an
!            erfcx(z) = --- * (  --  --  -- ... -- ...  )
!                       √pi      z+  z+  z+     z+
!
!                       with a0 = 1, an = n/2 for n ≥ 1
!
             b = z                               ! b0
             f = b                               ! b0 /= 0
             a = zero                            ! prepare a1
             C = f                               ! C <- f
             D  = zero                           ! D <- 0
             do while( it < itmax ); it=it+1_ik  ! iterations
                a = a + half                     ! a1, ..., an
!               calculation of C and D, incl. h = C*D
                C =        b + a/C               ! bn >> + <<
                D = one / (b + a*D)              ! bn >> + <<
                h = C * D
                f = f * h                        ! new function value
                if( ABS( h - one ) <= eps ) exit ! precision check
             end do
             f = onedivsqrtpi / f                ! finalize f, x > 0
             if( x < zero ) f = two*EXP(z*z) -f ! erfcx(-x)=2*exp(z²)-erfcx(z)
             end if                              ! selec == 2
!
!
             if( selec == 3_ik ) then
!            Continued Fraction (redefined formula) referenced Fettis
!            (1974), p. 585, 4th formula and Fettis (1975), p. 673 and
!            modified Lentz method for Continued Fraction
!
!                        1                         a0 = (t = 1/z)
!            erfcx(z) = --- * ------------------------------------
!                       √pi                        a1 = 1*(t²/2)
!                             b0=1 + -----------------------------
!                                                  a2 = 2*(t²/2)
!                                    b1=1 + ----------------------
!                                                  a3 = 3*(t²/2)
!                                           b2=1 + ---------------
!                                                       ...
!            or
!                        1       a0  a1  a2       an
!            erfcx(z) = --- * (  --  --  --  ...  --  ...  )
!                       √pi      1+  1+  1+       1+
!
!                       with a0 = t, an = n*(t²/2) for n ≥ 1
!
             t  = one / z                        ! t  = 1 / z
             t2 = (t * t) / two                  ! t2 = t² / 2
             b  = one                            ! b0
             f  = b                              ! b0 /= 0
             a  = zero                           ! prepare a1
             C  = f                              ! C <- f
             D  = zero                           ! D <- 0
             do while( it < itmax ); it=it+1_ik  ! iterations
                a = a + t2                       ! a1, ..., an
                C =        b + a/C               ! bn >> + <<
                D = one / (b + a*D)              ! bn >> + <<
                h = C * D
                f = f * h                        ! new function value
                if( ABS( h - one ) <= eps ) exit ! precision check
             end do
             f = (onedivsqrtpi * t)  /  f        ! finalize f, x > 0
             if( x < zero ) f = two*EXP(z*z) -f ! erfcx(-x)=2*exp(z²)-erfcx(z)
             end if                              ! selec == 3
!
!
             if( selec == 4_ik ) then
!            Continued Fraction (for x > 2.83) referenced
!            Shepherd/Laframboise (1981), page 251, formula (9) and
!            modified Lentz method for Continued Fraction
!
!                         1                           a0 = 1
!            erfcx(z) = ----- * -----------------------------------
!                       √pi*z                         a1 = 1/(2z²)
!                               b0=1 + ----------------------------
!                                                     a2 = 2/(2z²)
!                                      b1=1 + ---------------------
!                                                     a3 = 3/(2z²)
!                                             b2=1 + --------------
!                                                        ...
!            or
!                        1         a0  a1  a2     an
!            erfcx(z) = ----- * (  --  --  -- ... -- ...  )
!                       √pi*z      1+  1+  1+     1+
!
!                       with a0 = 1, an = n/(2*z²) for n ≥ 1
!
!            local handling: if z ≥ SQRT(limiterfcxH), f = (1/√pi)/z
!            sqrtlimiterfcxH=[sp:6.927..E+18/dp:5.035..E+153/ep:4.096..E+2465]
             if( z >= sqrtlimiterfcxH ) then     ! early exit
                 f = onedivsqrtpi / z            ! (1/√pi)/z
                 if(x < zero) f=two*EXP(z*z) -f ! erfcx(-x)=2*exp(z²)-erfcx(z)
                 return
             end if
!
             zz = z * z                          ! zz = z*z
             t  = one / (two * zz)               ! 1/(2z²)
             b  = one                            ! b0
             f  = b                              ! b0 /= 0
             a  = zero                           ! prepare a1
             C  = f                              ! C <- f
             D  = zero                           ! D <- 0
             do while( it < itmax ); it=it+1_ik  ! iterations
                a = a + t                        ! a1, ..., an
                C =        b + a/C               ! bn >> + <<
                D = one / (b + a*D)              ! bn >> + <<
                h = C * D
                f = f * h                        ! new function value
                if( ABS( h - one ) <= eps ) exit ! precision check
             end do
             f = one / (sqrtpi * z * f )         ! finalize f, x > 0
             if( x < zero ) f = two*EXP(zz) - f ! erfcx(-x)=2*exp(z²)-erfcx(z)
             end if                              ! selec == 4
!
!
!   limiterfcxH ≤ z=|x| ≤ +∞
    else                                         ! cut off "High"
             f = zero                            ! f = 0.0
!
    end if                                       ! end if calculation
    end if                                       ! end if cases
!
!   check and reset preselec for next new run
    if( selecp == 1_ik .and. (x <= +HUGE(x) .and. x >= +HUGE(x)) ) then
        selecp = 0_ik                            ! reset selecp
    end if
!
    return
    end function erfcxCF
!
!-----------------------------------------------------------------------------
!